package org.jpwh.test.fetching;

import org.hibernate.Hibernate;
import org.hibernate.Session;
import org.jpwh.env.JPATest;
import org.jpwh.model.fetching.profile.Bid;
import org.jpwh.model.fetching.profile.Item;
import org.jpwh.model.fetching.profile.User;
import org.jpwh.shared.util.CalendarUtil;
import org.jpwh.shared.util.TestData;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

import javax.persistence.EntityManager;
import javax.transaction.UserTransaction;
import java.math.BigDecimal;

import static org.testng.Assert.*;

public class Profile extends JPATest {

    @Override
    public void configurePersistenceUnit() throws Exception {
        configurePersistenceUnit("FetchingProfilePU");
    }

    public FetchTestData storeTestData() throws Exception {
        UserTransaction tx = TM.getUserTransaction();
        tx.begin();
        EntityManager em = JPA.createEntityManager();

        Long[] itemIds = new Long[3];
        Long[] userIds = new Long[3];

        User jandomanski = new User("jandomanski");
        em.persist(jandomanski);
        userIds[0] = jandomanski.getId();

        User janinadomanska = new User("janinadomanska");
        em.persist(janinadomanska);
        userIds[1] = janinadomanska.getId();

        User robertdomanski = new User("robertdomanski");
        em.persist(robertdomanski);
        userIds[2] = robertdomanski.getId();

        Item item = new Item("Przedmiot pierwszy", CalendarUtil.TOMORROW.getTime(), jandomanski);
        em.persist(item);
        itemIds[0] = item.getId();
        for (int i = 1; i <= 3; i++) {
            Bid bid = new Bid(item, robertdomanski, new BigDecimal(9 + i));
            item.getBids().add(bid);
            em.persist(bid);
        }

        item = new Item("Przedmiot drugi", CalendarUtil.TOMORROW.getTime(), jandomanski);
        em.persist(item);
        itemIds[1] = item.getId();
        for (int i = 1; i <= 1; i++) {
            Bid bid = new Bid(item, janinadomanska, new BigDecimal(2 + i));
            item.getBids().add(bid);
            em.persist(bid);
        }

        item = new Item("Przedmiot trzeci", CalendarUtil.AFTER_TOMORROW.getTime(), janinadomanska);
        em.persist(item);
        itemIds[2] = item.getId();
        for (int i = 1; i <= 1; i++) {
            Bid bid = new Bid(item, jandomanski, new BigDecimal(3 + i));
            item.getBids().add(bid);
            em.persist(bid);
        }

        tx.commit();
        em.close();

        FetchTestData testData = new FetchTestData();
        testData.items = new TestData(itemIds);
        testData.users = new TestData(userIds);
        return testData;
    }

    @Test
    public void fetchWithProfile() throws Exception {
        FetchTestData testData = storeTestData();

        UserTransaction tx = TM.getUserTransaction();
        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();

            Long ITEM_ID = testData.items.getFirstId();

            /* 
                <code>Item#seller</code> jest mapowany leniwie, dlatego domyślny plan pobierania 
                pobierze tylko egzemplarz <code>Item</code>.
             */
            Item item = em.find(Item.class, ITEM_ID);

            assertFalse(Hibernate.isInitialized(item.getSeller()));

            em.clear();
            /* 
                   Potrzebny jest API Hibernate do włączenia profilu. Po włączeniu będzie on aktywny dla wszystkich operacji 
                   w tej jednostce pracy. <code>Item#seller</code> będzie pobrany w złączeniu z tą samą instrukcją SQL 
                   za kazdym razem, gdy <code>EntityManager</code> załaduje egzemplarz <code>Item</code> .
             */
            em.unwrap(Session.class).enableFetchProfile(Item.PROFILE_JOIN_SELLER);
            item = em.find(Item.class, ITEM_ID);

            em.clear();
            assertNotNull(item.getSeller().getUsername());

            em.clear();
            /* 
                  Na tę samą jednostkę pracy można nałożyć inny profil. W takim przypadku kolekcje <code>Item#seller</code> 
                  i <code>Item#bids</code> będą pobierane jako złączenie w jednej instrukcji SQL 
                  za każdym razem, gdy jest ładowany egzemplarz <code>Item</code>.
             */
            em.unwrap(Session.class).enableFetchProfile(Item.PROFILE_JOIN_BIDS);
            item = em.find(Item.class, ITEM_ID);

            em.clear();
            assertNotNull(item.getSeller().getUsername());
            assertTrue(item.getBids().size() > 0);

            tx.commit();
            em.close();
        } finally {
            TM.rollback();
        }
    }


}
