package org.jpwh.test.fetching;

import org.jpwh.env.JPATest;
import org.jpwh.model.fetching.batch.Bid;
import org.jpwh.model.fetching.batch.Item;
import org.jpwh.model.fetching.batch.User;
import org.jpwh.shared.util.CalendarUtil;
import org.jpwh.shared.util.TestData;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

import javax.persistence.EntityManager;
import javax.transaction.UserTransaction;
import java.math.BigDecimal;
import java.util.List;

import static org.testng.Assert.assertNotNull;
import static org.testng.Assert.assertTrue;

public class Batch extends JPATest {

    @Override
    public void configurePersistenceUnit() throws Exception {
        configurePersistenceUnit("FetchingBatchPU");
    }

    public FetchTestData storeTestData() throws Exception {
        UserTransaction tx = TM.getUserTransaction();
        tx.begin();
        EntityManager em = JPA.createEntityManager();

        Long[] itemIds = new Long[3];
        Long[] userIds = new Long[3];

        User jandomanski = new User("jandomanski");
        em.persist(jandomanski);
        userIds[0] = jandomanski.getId();

        User janinadomanska = new User("janinadomanska");
        em.persist(janinadomanska);
        userIds[1] = janinadomanska.getId();

        User robertdomanski = new User("robertdomanski");
        em.persist(robertdomanski);
        userIds[2] = robertdomanski.getId();

        Item item = new Item("Przedmiot pierwszy", CalendarUtil.TOMORROW.getTime(), jandomanski);
        em.persist(item);
        itemIds[0] = item.getId();
        for (int i = 1; i <= 3; i++) {
            Bid bid = new Bid(item, robertdomanski, new BigDecimal(9 + i));
            item.getBids().add(bid);
            em.persist(bid);
        }

        item = new Item("Przedmiot drugi", CalendarUtil.TOMORROW.getTime(), jandomanski);
        em.persist(item);
        itemIds[1] = item.getId();
        for (int i = 1; i <= 1; i++) {
            Bid bid = new Bid(item, janinadomanska, new BigDecimal(2 + i));
            item.getBids().add(bid);
            em.persist(bid);
        }

        item = new Item("Przedmiot trzeci", CalendarUtil.AFTER_TOMORROW.getTime(), janinadomanska);
        em.persist(item);
        itemIds[2] = item.getId();
        for (int i = 1; i <= 1; i++) {
            Bid bid = new Bid(item, jandomanski, new BigDecimal(3 + i));
            item.getBids().add(bid);
            em.persist(bid);
        }

        tx.commit();
        em.close();

        FetchTestData testData = new FetchTestData();
        testData.items = new TestData(itemIds);
        testData.users = new TestData(userIds);
        return testData;
    }

    @Test
    public void fetchProxyBatches() throws Exception {
        storeTestData();

        UserTransaction tx = TM.getUserTransaction();
        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();

            List<Item> items = em.createQuery("select i from Item i").getResultList();
            // select * from ITEM

            for (Item item : items) {
                assertNotNull(item.getSeller().getUsername());
                // select * from USERS where ID in (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            }
            em.clear();


            // Właściwy test
            /*
                UWAGA: Ten test nie działa z powodu dziwactwa w obsłudze proxy przez Hibernate. 
                Dostęp do pierwszego obiektu proxy spowoduje  załadowanie jego danych i jednocześnie  danych 
                dla innych obiektów proxy aktualnie w kontekście utrwalania (innych sprzedawców, z których 
                dotychczas nie  korzystaliśmy). Ale tylko sprzedający, do którego sięgaliśmy zostanie faktycznie 
                zainicjowany, co oznacza, że to on będzie "połączony" z załadowanymi danymi. Inne obiekty 
                proxy są nadal w stanie "niezainicjowany", mimo, że ich dane są teraz dostępne w kontekście 
                utrwalania. Kiedy sięgniemy do takiego obiektu proxy po wyczyszczeniu (zamknięciu) kontekstu 
                utrwalania, zostanie zgłoszony wyjątek leniwego ładowania. Trzeba sięgnąć do niezainicjowanych 
                obiektów proxy w czasie trwania cyklu życia kontekstu utrwalania. Dzięki temu będzie on 
                "podłączony" do załadowanych danych w kontekście utrwalania. Jeśli chcesz, aby dane były 
                dostępne w stanie odłączonym, możesz to uzyskać za pomocą ustawienia  FetchType.EAGER. 
                Oczywiście wtedy nie mamy już do czynienia z leniwym pobieraniem wsadowym.

            items = em.createQuery("select i from Item i").getResultList();
            // Dostęp powinien załadować wszystkich sprzedawców (mamy tylko 3, rozmiar partii wynosi 10)
            assertNotNull(items.iterator().next().getSeller().getUsername());
            em.clear(); // Odłączamy wszystko

            for (Item item : items) {
                assertNotNull(item.getSeller().getUsername());
            }
            */


            tx.commit();
            em.close();
        } finally {
            TM.rollback();
        }
    }

    @Test
    public void fetchCollectionBatches() throws Exception {
        storeTestData();

        UserTransaction tx = TM.getUserTransaction();
        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();

            List<Item> items = em.createQuery("select i from Item i").getResultList();
            // select * from ITEM

            for (Item item : items) {
                assertTrue(item.getBids().size() > 0);
                // select * from BID where ITEM_ID in (?, ?, ?, ?, ?)
            }

            // Właściwy test
            em.clear();
            items = em.createQuery("select i from Item i").getResultList();
            // Dostęp powinien załadować wszystkie (masowo, choć mamy tylko 3) kolekcje
            assertTrue(items.iterator().next().getBids().size() > 0);
            em.clear(); // Odłączamy wszystko
            for (Item item : items) {
                assertTrue(item.getBids().size() > 0);
            }

            tx.commit();
            em.close();
        } finally {
            TM.rollback();
        }
    }

}
