import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;
import java.util.*;
import java.util.concurrent.*;
import javax.swing.*;

/**
 * Niniejszy program tworzy animacj algorytmu sortwania.
 * @version 1.01 2007-05-18
 * @author Cay Horstmann
 */
public class AlgorithmAnimation
{
   public static void main(String[] args)
   {
      EventQueue.invokeLater(new Runnable()
         {
            public void run()
            {
               JFrame frame = new AnimationFrame();
               frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
               frame.setVisible(true);
            }
         });
   }
}

/**
 * Ta ramka pokazuje tablic w czasie sortowania oraz przyciski pozwalajce uruchomi animacj krok po kroku lub
 * bez zatrzymywania.
 */
class AnimationFrame extends JFrame
{
   public AnimationFrame()
   {
      ArrayComponent comp = new ArrayComponent();
      add(comp, BorderLayout.CENTER);

      final Sorter sorter = new Sorter(comp);

      JButton runButton = new JButton("Start");
      runButton.addActionListener(new ActionListener()
         {
            public void actionPerformed(ActionEvent event)
            {
               sorter.setRun();
            }
         });

      JButton stepButton = new JButton("Etapy");
      stepButton.addActionListener(new ActionListener()
         {
            public void actionPerformed(ActionEvent event)
            {
               sorter.setStep();
            }
         });

      JPanel buttons = new JPanel();
      buttons.add(runButton);
      buttons.add(stepButton);
      add(buttons, BorderLayout.NORTH);
      setSize(DEFAULT_WIDTH, DEFAULT_HEIGHT);

      Thread t = new Thread(sorter);
      t.start();
   }

   private static final int DEFAULT_WIDTH = 300;
   private static final int DEFAULT_HEIGHT = 300;
}

/**
 * Niniejsza klasa wykonuje algorytm sortowania. Kiedy dwa elementy s porwnywane, algorytm wstrzymuje dziaanie i
 * aktualizuje komponent.
 */
class Sorter implements Runnable
{
   /**
    * Tworzy obiekt typu Sorter.
    * @param values tablica do posortowania
    * @param comp komponent, na ktrym maj by prezentowany postp sortowania
    */
   public Sorter(ArrayComponent comp)
   {
      values = new Double[VALUES_LENGTH];
      for (int i = 0; i < values.length; i++)
         values[i] = new Double(Math.random());
      this.component = comp;
      this.gate = new Semaphore(1);
      this.run = false;
   }

   /**
    * Ustawia obiekt Sorter w tryb dziaania bez przerwy. Wywoywana na rzecz wtku dystrybucji zdarze.
    */
   public void setRun()
   {
      run = true;
      gate.release();
   }

   /**
    * Ustawia obiekt Sorter w tryb dziaania etapami. Wywoywana na rzecz wtku dystrybucji zdarze.
    */
   public void setStep()
   {
      run = false;
      gate.release();
   }

   public void run()
   {
      Comparator<Double> comp = new Comparator<Double>()
         {
            public int compare(Double i1, Double i2)
            {
               component.setValues(values, i1, i2);
               try
               {
                  if (run) Thread.sleep(DELAY);
                  else gate.acquire();
               }
               catch (InterruptedException exception)
               {
                  Thread.currentThread().interrupt();
               }
               return i1.compareTo(i2);
            }
         };
      Arrays.sort(values, comp);
      component.setValues(values, null, null);
   }

   private Double[] values;
   private ArrayComponent component;
   private Semaphore gate;
   private static final int DELAY = 100;
   private volatile boolean run;
   private static final int VALUES_LENGTH = 30;
}

/**
 * Niniejszy komponent rysuje tablic i zaznacza w niej dwa elementy.
 */
class ArrayComponent extends JComponent
{
   /**
    * Ustawia wartoci, ktre maj by narysowane. Wywoywana na rzecz wtku sortujcego.
    * @param values tablica wartoci do wywietlenia
    * @param marked1 pierwszy oznaczony element
    * @param marked2 drugi oznaczony element
    */
   public synchronized void setValues(Double[] values, Double marked1, Double marked2)
   {
      this.values = values.clone();
      this.marked1 = marked1;
      this.marked2 = marked2;
      repaint();
   }

   public synchronized void paintComponent(Graphics g) // Wywoywana na rzecz wtku dystrybucji zdarze.
   {
      if (values == null) return;
      Graphics2D g2 = (Graphics2D) g;
      int width = getWidth() / values.length;
      for (int i = 0; i < values.length; i++)
      {
         double height = values[i] * getHeight();
         Rectangle2D bar = new Rectangle2D.Double(width * i, 0, width, height);
         if (values[i] == marked1 || values[i] == marked2) g2.fill(bar);
         else g2.draw(bar);
      }
   }

   private Double marked1;
   private Double marked2;
   private Double[] values;
}
