package executors;

import java.io.*;
import java.nio.file.*;
import java.time.*;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.*;

/**
 * Program demonstrujcy interfejs Callable i egzekutory.
 * @version 1.01.2021-05-30
 * @author Cay Horstmann
 */
public class ExecutorDemo
{
   /**
    * Liczy wystpienia danego sowa w pliku.
    * @return liczba wystpie danego sowa
    */
   public static long occurrences(String word, Path path)
   {
      try (var in = new Scanner(path))
      {
         int count = 0;
         while (in.hasNext())
            if (in.next().equals(word)) count++;
         return count;
      }
      catch (IOException ex)
      {
         return 0;
      }
   }

   /**
    * Zwraca wszystkie podkatalogi danego katalogu  zobacz rozdziay 1. i 2. w tomie II.
    * @param rootDir katalog gwny
    * @return zbir wszystkich podkatalogw katalogu gwnego
    */
   public static Set<Path> descendants(Path rootDir) throws IOException
   {
      try (Stream<Path> entries = Files.walk(rootDir))
      {
         return entries.filter(Files::isRegularFile)
            .collect(Collectors.toSet());
      }
   }

   /**
    * Tworzy zadanie szukajce sowa w pliku.
    * @param word szukane sowo
    * @param path plik do przeszukania
    * @return zadanie wyszukiwania, ktre zwraca ciek w przypadku powodzenia
    */
   public static Callable<Path> searchForTask(String word, Path path)
   {
      return () -> {
         try (var in = new Scanner(path))
         {
            while (in.hasNext())
            {
               if (in.next().equals(word)) return path;
               if (Thread.currentThread().isInterrupted())
               {
                  System.out.println("Szukanie w " + path + " anulowano.");
                  return null;
               }
            }
            throw new NoSuchElementException();
         }
      };
   }

   public static void main(String[] args)
         throws InterruptedException, ExecutionException, IOException
   {
      try (var in = new Scanner(System.in))
      {
         System.out.print("Wpisz ciek do katalogu podstawowego (np. /opt/jdk-9-src): ");
         String start = in.nextLine();
         System.out.print("Wpisz sowo kluczowe (np. volatile): ");
         String word = in.nextLine();
      
         Set<Path> files = descendants(Path.of(start));
         var tasks = new ArrayList<Callable<Long>>();
         for (Path file : files)
         {
            Callable<Long> task = () -> occurrences(word, file);          
            tasks.add(task);
         }
         ExecutorService executor = Executors.newCachedThreadPool();
         // uyj egzekutora wtku, aby sprawdzi, czy
         // wiksza liczba wtkw przyspiesza wyszukiwanie
         // ExecutorService executor = Executors.newSingleThreadExecutor();
         
         Instant startTime = Instant.now();
         List<Future<Long>> results = executor.invokeAll(tasks);
         long total = 0;
         for (Future<Long> result : results)
            total += result.get();
         Instant endTime = Instant.now();
         System.out.println("Liczba wystpie sowa " + word + ": " + total);
         System.out.println("Czas: "
            + Duration.between(startTime, endTime).toMillis() + " ms");

         var searchTasks = new ArrayList<Callable<Path>>();
         for (Path file : files)
            searchTasks.add(searchForTask(word, file));
         Path found = executor.invokeAny(searchTasks);
         System.out.println(word + " occurs in: " + found);

         if (executor instanceof ThreadPoolExecutor tpExecutor)
             // jednowtkowy egzekutor nie jest
         System.out.println("Najwikszy rozmiar puli: "
             + tpExecutor.getLargestPoolSize();
         executor.shutdown();
     }
   }
}