/* PersistentCacheVector.java */
 package jwiley.effective;
 
 import java.io.*;
 import java.util.*;
 
 /**
  *  Ekwiwalent klasy Vector uywajcy bufora LRU i umieszczajcy
  *  na dysku nadmiarowe obiekty.
  */
 public class PersistentCacheVector implements Cloneable, Serializable
 {
    /** Licznik utworzonych obiektw. */
    static long lCount;
    /** Maksymalny rozmiar bufora. */
    int iMaxCacheSize;
    /** Biecy rozmiar bufora. */
    int iCacheSize;
    /** Bufor. */
    Hashtable cache = new Hashtable();
    /** Odbicie klasy Vector. Przechowuje jedynie 
        obiekt klasy VectorEntry dla kadego wstawianego obiektu. */
    Vector rows = new Vector();
    /** ObjectFile przechowuje obiekty na dysku.
        @see ObjectFile
    */
    ObjectFile of;
    /** Nazwa pliku, w ktrym przechowywane s obiekty. */
    String sTmpName;
    /** Pocztek listy LRU.  Pierwszy element reprezentuje 
     najczciej (ostatnio) uywany element bufora. */
    CacheEntry firstCacheEntry;
    /** Koniec listy LRU.  Ostatni element reprezentuje. 
     najrzadziej (ostatnio) uywany element bufora. */
    CacheEntry lastCacheEntry;
 
    /**
     * Klasa wewntrzna namiastki.
     */
    class StubEntry
    {
       /** Znacznik informujcy czy element znajduje si w buforze, czy na dysku. */
       boolean inCache;
       /** Wskanik pozycji pliku, gdy element znajduje si na dysku. */
       long filePointer = -1;
    }
 
    /**
     * Klasa wewntrzna reprezentujca element bufora.
     */
    class CacheEntry
    {
       /** Klucz elementu bufora. Stosujemy obiekt Integer 
           odpowiadajcy indeksowi elementu wektora. */
       Integer key;
       /** Obiekt przechowywany w buforze. */
       Object o;
       /** Referencje poprzedniego i nastpnego elementu listy. */
       CacheEntry prev, next;
    }
 
    /**
     * Konstruktor okrelajcy rozmiar bufora.  Plik nie jest otwierany
     * dopki nie zostanie przekroczony rozmiar bufora.
     * @param iCacheSize Maksymalny rozmiar bufora.
     */
    public PersistentCacheVector(int iCacheSize)
    {
       this.iMaxCacheSize = iCacheSize;
    }
 
    private final void openTempFile() throws IOException
    {
       boolean bInValid = true;
 
       while (bInValid)
       {
          sTmpName = "tmp" + (System.currentTimeMillis()) 
                           + (++lCount) + ".obf";
          File f = new File(sTmpName);
          if (!f.exists())
             bInValid = false;
       }
 
       of = new ObjectFile(sTmpName);
    }
 
    /**
     * Metoda wstawiania elementu do wektora. Element zostaje 
     * umieszczony w buforze lub na dysku.
     * @param o Obiekt dodawany do wektora.
     */
    public final void add(Serializable o) throws IOException
    {
       StubEntry e = new StubEntry();
 
       rows.add(e);
 
       if (iCacheSize < iMaxCacheSize)
       {
          e.inCache = true;
          CacheEntry ce = new CacheEntry();
          ce.o = o;
          ce.key = new Integer(rows.size() - 1);
          cache.put(ce.key, ce);
          iCacheSize++;
 
          // wstawia element na list
          if (lastCacheEntry != null)
          {
             // umieszcza go na pocztku listy
             ce.next = firstCacheEntry;
             firstCacheEntry.prev = ce;
             firstCacheEntry = ce;
          }
          else
          {
             // lista jest pusta
             firstCacheEntry = lastCacheEntry = ce;
          }
       }
       else
       {
         if (of == null)
             openTempFile();
 
         e.filePointer = of.writeObject(o);
       }
    }
 
    /**
     * Metoda dostpu do elementu wektora o podanym indeksie.
     * Obiekt pobierany jest z bufora lub pliku.
     * @param idx Indeks pobieranego elementu.
     * @returns obiekt umieszczony w wektorze.
     */
    public final Object get(int idx) throws IndexOutOfBoundsException,
                                            IOException
    {
       if (idx < 0 || idx >= rows.size())
          throw new IndexOutOfBoundsException("Index: " + idx 
                                          + " out of bounds.");
 
       StubEntry e = (StubEntry) rows.get(idx);
 
       Object o=null;
 
       if (e.inCache)
       {
          // pobiera element
          CacheEntry ce = null;
          ce = (CacheEntry) cache.get(new Integer(idx));
 
          if (ce == null)
             throw new IOException("Element at idx " + idx + " is NULL!");
 
          if ((ce != null && ce.o == null))
             throw new IOException("Cache Element's object at idx " 
                                   + idx + " NOT in cache!");
 
          o = ce.o;
 
          if (ce != firstCacheEntry)
          {
             // usuwa go z listy
             if (ce.next != null)
                ce.next.prev = ce.prev;
             else // ostatni element listy
                 lastCacheEntry = ce.prev;
 
             ce.prev.next = ce.next;
 
             // i wstawia na pocztek listy
             ce.next = firstCacheEntry;
             ce.prev = null;
             firstCacheEntry.prev = ce;
             firstCacheEntry = ce;
          }
       }
       else
       {
          // obiekt w buforze
          e.inCache = true;
 
          // pobieranie i wstawianie do bufora
          try
          {
             o = of.readObject(e.filePointer);
          } catch (ClassNotFoundException cnfe)
            { throw new IOException(cnfe.getMessage()); }
 
          //*** Sprawdza, czy bufor nie jest peen!
          if (iCacheSize == iMaxCacheSize)
          {
             // usuwa element znajdujcy si na kocu listy.
             CacheEntry leastUsed = lastCacheEntry;
             if (leastUsed.prev != null)
             {
                leastUsed.prev.next = null;
                lastCacheEntry = leastUsed.prev;
                lastCacheEntry.next = null;
             }
             else
             {
                // usuwa jedyny element listy
                firstCacheEntry = lastCacheEntry = null;
             }
 
             // wstawia pobrany element do bufora
             CacheEntry ce = new CacheEntry();
             ce.o = o;
             ce.key = new Integer(idx);
             cache.put(ce.key, ce);
 
             // umieszcza go na licie LRU
             if (lastCacheEntry != null)
             {
                // wstawia na pocztek listy
                ce.next = firstCacheEntry;
                firstCacheEntry.prev = ce;
                firstCacheEntry = ce;
             }
             else
             {
                // lista jest pusta
                firstCacheEntry = lastCacheEntry = ce;
             }
 
             // pobiera StubEntry usuwanego obiektu
             StubEntry outStubEntry = (StubEntry) 
                 rows.get(leastUsed.key.intValue());
 
             // usuwa obiekt z bufora
             CacheEntry outCacheEntry = (CacheEntry) 
                 cache.remove(leastUsed.key);
             if (outCacheEntry == null)
                throw new RuntimeException("Cache Entry at " 
                               + leastUsed.key + " is Null!");
 
             if (outCacheEntry != null && outCacheEntry.o == null)
                throw new RuntimeException("Cache object at " 
                                + leastUsed.key + " is Null!");
 
             Object outObject = outCacheEntry.o;
 
             outStubEntry.inCache = false;
 
             if (outStubEntry.filePointer == -1)
             {
                // umieszczony w buforze
                   outStubEntry.filePointer = 
                            of.writeObject((Serializable)outObject);
             }
             else
             {
                // znajduje ju si w pliku  zmiana rozmiaru?
                 int iCurrentSize = 
                   of.getObjectLength(outStubEntry.filePointer);
 
                 ByteArrayOutputStream baos = new ByteArrayOutputStream();
                 ObjectOutputStream oos = new ObjectOutputStream(baos);
                 oos.writeObject((Serializable) outObject);
                 oos.flush();
                 int datalen = baos.size();
 
                 if (datalen <= iCurrentSize)
                     of.rewriteObject(outStubEntry.filePointer, 
                                      baos.toByteArray());
                 else
                     outStubEntry.filePointer = 
                                 of.writeObject((Serializable)outObject);
 
                 baos = null;
                 oos = null;
                 outObject = null;
             }
          }
          else
          {
             CacheEntry ce = new CacheEntry();
             ce.o = o;
             ce.key = new Integer(idx);
             cache.put(ce.key, ce);
             iCacheSize++;
 
             // wstawia na list LRU
             if (lastCacheEntry != null)
             {
                // na pocztek listy
                ce.next = firstCacheEntry;
                firstCacheEntry.prev = ce;
                firstCacheEntry = ce;
             }
             else
             {
                // lista jest pusta
                firstCacheEntry = lastCacheEntry = ce;
             }
          }
 
       }
 
       return o;
    }
    // *** Metody pominite ze wzgledu na brak miejsca
 }
