package com.oreilly.tiger.ch07;

import java.util.Iterator;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;

/**
 * Klasa umoliwiajca przegldanie pliku tekstowego wierszami.
 * Metoda remove() iteratora generuje wyjtek UnsupportedOperatorException.
 * Iterator obudowuje i ponownie wyrzuca wyjtki IOException jako IllegalArgumentException.
 */
public class TextFile implements Iterable<String> {

  // Uywana przez klas TextFileIterator
  final String filename;

  public TextFile(String filename) { 
    this.filename = filename; 
  }

  // Jedyna metoda interfejsu Iterable
  public Iterator<String> iterator() { 
    return new TextFileIterator(); 
  }


  // Ta klasa skadowa nie jest statyczna 
  // i stanowi implementacj interfejsu Iterator
  class TextFileIterator implements Iterator<String> {

    // Strumie wejciowy
    BufferedReader in;

    // Wynik nastepnego wywoania next()
    String nextline;

    public TextFileIterator() {
      // Otwiera plik, wczytuje i zapamituje pierwszy wiersz.
      // Wiersze s podgldane wprzd dla potrzeb metody hasNext().
      try {
        in = new BufferedReader(new FileReader(filename));
        nextline = in.readLine();
      } catch(IOException e) { 
        throw new IllegalArgumentException(e); 
      }
    }

    // Istnieje nastpny wiersz?
    public boolean hasNext() { 
      return nextline != null; 
    }

    // Zwraca nastpny wiersz, ale najpierw wczytuje kolejny.
    public String next() {
      try {
        String result = nextline;

        // Jeli nie zosta osigniety koniec pliku
        if (nextline != null) {  
          nextline = in.readLine(); // Wczytuje kolejny wiersz
          if (nextline == null) 
            in.close();             // Zamyka plik po napotkaniu koca
        }

        // Zwraca wiersz wczytany poprzednim razem.
        return result;
      } catch(IOException e) { 
        throw new IllegalArgumentException(e); 
      }
    }

    // Plik tylko do odczytu, wiersze nie mog by usuwane.
    public void remove() { 
      throw new UnsupportedOperationException(); 
    }
  }

  public static void main(String[] args) {
    String filename = "TextFile.java";
    if (args.length > 0) 
      filename = args[0];

    for(String line : new TextFile(filename))
      System.out.println(line);
  }
}
