import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.net.*;
import java.util.*;
import java.util.regex.*;
import javax.swing.*;
import javax.swing.table.*;

// Agent wyszukiwania sieciowego
public class SearchCrawler extends JFrame
{
  // Lista maksymalnej liczby adresw URL.
  private static final String[] MAX_URLS =
    {"50", "100", "500", "1000"};

  // Bufor list zabronionego wyszukiwania.
  private HashMap disallowListCache = new HashMap();

  // Elementy interfesju graficznego.
  private JTextField startTextField;
  private JComboBox maxComboBox;
  private JCheckBox limitCheckBox;
  private JTextField logTextField;
  private JTextField searchTextField;
  private JCheckBox caseCheckBox;
  private JButton searchButton;

  // Elementy statystyki wyszukiwania.
  private JLabel crawlingLabel2;
  private JLabel crawledLabel2;
  private JLabel toCrawlLabel2;
  private JProgressBar progressBar;
  private JLabel matchesLabel2;

  // Tabela z adresami zawierajcymi szukany tekst.
  private JTable table;

  // Informacja, czy odbywa si wyszukiwanie.
  private boolean crawling;

  // Referencja do pliku z zapisywanymi wynikami.
  private PrintWriter logFileWriter;

  // Konstruktor klasy.
  public SearchCrawler()
  {
    // Tytu apliakcji.
    setTitle("Agent wyszukiwania");

    // Rozmiar okna.
    setSize(700, 600);

    // Przechwytywanie zdarzenia zamykania okna.
    addWindowListener(new WindowAdapter() {
      public void windowClosing(WindowEvent e) {
        actionExit();
      }
    });

    // Ustawienie menu.
    JMenuBar menuBar = new JMenuBar();
    JMenu fileMenu = new JMenu("Plik");
    fileMenu.setMnemonic(KeyEvent.VK_P);
    JMenuItem fileExitMenuItem = new JMenuItem("Wyjcie",
      KeyEvent.VK_W);
    fileExitMenuItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        actionExit();
      }
    });
    fileMenu.add(fileExitMenuItem);
    menuBar.add(fileMenu);
    setJMenuBar(menuBar);

    // Panel wyszukiwania.
    JPanel searchPanel = new JPanel();
    GridBagConstraints constraints;
    GridBagLayout layout = new GridBagLayout();
    searchPanel.setLayout(layout);

    JLabel startLabel = new JLabel("Adres pocztkowy:");
    constraints = new GridBagConstraints();
    constraints.anchor = GridBagConstraints.EAST;
    constraints.insets = new Insets(5, 5, 0, 0);
    layout.setConstraints(startLabel, constraints);
    searchPanel.add(startLabel);

    startTextField = new JTextField();
    constraints = new GridBagConstraints();
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.gridwidth = GridBagConstraints.REMAINDER;
    constraints.insets = new Insets(5, 5, 0, 5);
    layout.setConstraints(startTextField, constraints);
    searchPanel.add(startTextField);

    JLabel maxLabel = new JLabel("Maks. przeszuka:");
    constraints = new GridBagConstraints();
    constraints.anchor = GridBagConstraints.EAST;
    constraints.insets = new Insets(5, 5, 0, 0);
    layout.setConstraints(maxLabel, constraints);
    searchPanel.add(maxLabel);

    maxComboBox = new JComboBox(MAX_URLS);
    maxComboBox.setEditable(true);
    constraints = new GridBagConstraints();
    constraints.insets = new Insets(5, 5, 0, 0);
    layout.setConstraints(maxComboBox, constraints);
    searchPanel.add(maxComboBox);

    limitCheckBox =
      new JCheckBox("Ogranicz do adresu startowego");
    constraints = new GridBagConstraints();
    constraints.anchor = GridBagConstraints.WEST;
    constraints.insets = new Insets(0, 5, 0, 0);
    layout.setConstraints(limitCheckBox, constraints);
    searchPanel.add(limitCheckBox);

    JLabel blankLabel = new JLabel();
    constraints = new GridBagConstraints();
    constraints.gridwidth = GridBagConstraints.REMAINDER;
    layout.setConstraints(blankLabel, constraints);
    searchPanel.add(blankLabel);

    JLabel logLabel = new JLabel("Zapis wynikw do pliku:");
    constraints = new GridBagConstraints();
    constraints.anchor = GridBagConstraints.EAST;
    constraints.insets = new Insets(5, 5, 0, 0);
    layout.setConstraints(logLabel, constraints);
    searchPanel.add(logLabel);

    String file =
      System.getProperty("user.dir") +
      System.getProperty("file.separator") +
      "crawler.log";
    logTextField = new JTextField(file);
    constraints = new GridBagConstraints();
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.gridwidth = GridBagConstraints.REMAINDER;
    constraints.insets = new Insets(5, 5, 0, 5);
    layout.setConstraints(logTextField, constraints);
    searchPanel.add(logTextField);

    JLabel searchLabel = new JLabel("Wyszukiwany tekst:");
    constraints = new GridBagConstraints();
    constraints.anchor = GridBagConstraints.EAST;
    constraints.insets = new Insets(5, 5, 0, 0);
    layout.setConstraints(searchLabel, constraints);
    searchPanel.add(searchLabel);

    searchTextField = new JTextField();
    constraints = new GridBagConstraints();
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.insets = new Insets(5, 5, 0, 0);
    constraints.gridwidth= 2;
    constraints.weightx = 1.0d;
    layout.setConstraints(searchTextField, constraints);
    searchPanel.add(searchTextField);

    caseCheckBox = new JCheckBox("Rozrniaj wielko liter");
    constraints = new GridBagConstraints();
    constraints.insets = new Insets(5, 5, 0, 5);
    constraints.gridwidth = GridBagConstraints.REMAINDER;
    layout.setConstraints(caseCheckBox, constraints);
    searchPanel.add(caseCheckBox);

    searchButton = new JButton("Szukaj");
    searchButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        actionSearch();
      }
    });
    constraints = new GridBagConstraints();
    constraints.gridwidth = GridBagConstraints.REMAINDER;
    constraints.insets = new Insets(5, 5, 5, 5);
    layout.setConstraints(searchButton, constraints);
    searchPanel.add(searchButton);

    JSeparator separator = new JSeparator();
    constraints = new GridBagConstraints();
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.gridwidth = GridBagConstraints.REMAINDER;
    constraints.insets = new Insets(5, 5, 5, 5);
    layout.setConstraints(separator, constraints);
    searchPanel.add(separator);

    JLabel crawlingLabel1 = new JLabel("Wyszukiwanie:");
    constraints = new GridBagConstraints();
    constraints.anchor = GridBagConstraints.EAST;
    constraints.insets = new Insets(5, 5, 0, 0);
    layout.setConstraints(crawlingLabel1, constraints);
    searchPanel.add(crawlingLabel1);

    crawlingLabel2 = new JLabel();
    crawlingLabel2.setFont(
      crawlingLabel2.getFont().deriveFont(Font.PLAIN));
    constraints = new GridBagConstraints();
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.gridwidth = GridBagConstraints.REMAINDER;
    constraints.insets = new Insets(5, 5, 0, 5);
    layout.setConstraints(crawlingLabel2, constraints);
    searchPanel.add(crawlingLabel2);

    JLabel crawledLabel1 = new JLabel("Wyszukane adresy:");
    constraints = new GridBagConstraints();
    constraints.anchor = GridBagConstraints.EAST;
    constraints.insets = new Insets(5, 5, 0, 0);
    layout.setConstraints(crawledLabel1, constraints);
    searchPanel.add(crawledLabel1);

    crawledLabel2 = new JLabel();
    crawledLabel2.setFont(
      crawledLabel2.getFont().deriveFont(Font.PLAIN));
    constraints = new GridBagConstraints();
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.gridwidth = GridBagConstraints.REMAINDER;
    constraints.insets = new Insets(5, 5, 0, 5);
    layout.setConstraints(crawledLabel2, constraints);
    searchPanel.add(crawledLabel2);

    JLabel toCrawlLabel1 = new JLabel("Adresw do przeszukania :");
    constraints = new GridBagConstraints();
    constraints.anchor = GridBagConstraints.EAST;
    constraints.insets = new Insets(5, 5, 0, 0);
    layout.setConstraints(toCrawlLabel1, constraints);
    searchPanel.add(toCrawlLabel1);

    toCrawlLabel2 = new JLabel();
    toCrawlLabel2.setFont(
      toCrawlLabel2.getFont().deriveFont(Font.PLAIN));
    constraints = new GridBagConstraints();
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.gridwidth = GridBagConstraints.REMAINDER;
    constraints.insets = new Insets(5, 5, 0, 5);
    layout.setConstraints(toCrawlLabel2, constraints);
    searchPanel.add(toCrawlLabel2);

    JLabel progressLabel = new JLabel("Postp wyszukiwania:");
    constraints = new GridBagConstraints();
    constraints.anchor = GridBagConstraints.EAST;
    constraints.insets = new Insets(5, 5, 0, 0);
    layout.setConstraints(progressLabel, constraints);
    searchPanel.add(progressLabel);

    progressBar = new JProgressBar();
    progressBar.setMinimum(0);
    progressBar.setStringPainted(true);
    constraints = new GridBagConstraints();
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.gridwidth = GridBagConstraints.REMAINDER;
    constraints.insets = new Insets(5, 5, 0, 5);
    layout.setConstraints(progressBar, constraints);
    searchPanel.add(progressBar);

    JLabel matchesLabel1 = new JLabel("Znalezione dopasowania:");
    constraints = new GridBagConstraints();
    constraints.anchor = GridBagConstraints.EAST;
    constraints.insets = new Insets(5, 5, 10, 0);
    layout.setConstraints(matchesLabel1, constraints);
    searchPanel.add(matchesLabel1);

    matchesLabel2 = new JLabel();
    matchesLabel2.setFont(
      matchesLabel2.getFont().deriveFont(Font.PLAIN));
    constraints = new GridBagConstraints();
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.gridwidth = GridBagConstraints.REMAINDER;
    constraints.insets = new Insets(5, 5, 10, 5);
    layout.setConstraints(matchesLabel2, constraints);
    searchPanel.add(matchesLabel2);

    // Ustawienie tablicy wynikw.
    table =
      new JTable(new DefaultTableModel(new Object[][]{},
        new String[]{"URL"}) {
      public boolean isCellEditable(int row, int column)
      {
        return false;
      }
    });

    // Ustawienie panelu wynikw.
    JPanel matchesPanel = new JPanel();
    matchesPanel.setBorder(
      BorderFactory.createTitledBorder("Wyniki"));
    matchesPanel.setLayout(new BorderLayout());
    matchesPanel.add(new JScrollPane(table),
      BorderLayout.CENTER);

    // Dodanie paneli do okna.
    getContentPane().setLayout(new BorderLayout());
    getContentPane().add(searchPanel, BorderLayout.NORTH);
    getContentPane().add(matchesPanel, BorderLayout.CENTER);
  }

  // Wyjcie z programu.
  private void actionExit() {
    System.exit(0);
  }

  // Obsuga klikni wyszukiwania i zatrzymania wyszukiwania.
  private void actionSearch() {
    // Jeli kliknito Stop, wyczenie wyszukiwania.
    if (crawling) {
      crawling = false;
      return;
    }

    ArrayList errorList = new ArrayList();

    // Sprawdzanie, czy wpisano adres pocztkowy.
    String startUrl = startTextField.getText().trim();
    if (startUrl.length() < 1) {
      errorList.add("Brak adresu pocztkowego.");
    }
    // Sprawdzanie poprawnoci adresu pocztkowego.
    else if (verifyUrl(startUrl) == null) {
      errorList.add("Bdny adres pocztkowy.");
    }

    // Sprawdzenie, czy maksymalna liczba stron jest pusta lub jest liczb.
    int maxUrls = 0;
    String max = ((String) maxComboBox.getSelectedItem()).trim();
    if (max.length() > 0) {
      try {
        maxUrls = Integer.parseInt(max);
      } catch (NumberFormatException e) {
      }
      if (maxUrls < 1) {
        errorList.add("Bdna warto maksymalnej liczby stron.");
      }
    }

    // Sprawdzenie, czy wpisano nazw pliku dziennika.
    String logFile = logTextField.getText().trim();
    if (logFile.length() < 1) {
      errorList.add("Brakujca nazwa pliku dziennika.");
    }

    // Sprawdzenie wpisania poszukiwanego tekstu.
    String searchString = searchTextField.getText().trim();
    if (searchString.length() < 1) {
      errorList.add("Brak poszukiwanego tekstu.");
    }

    // Poka bd, jeli wystpi i powr.
    if (errorList.size() > 0) {
      StringBuffer message = new StringBuffer();

      // czenie bdw w jednym komunikacie.
      for (int i = 0; i < errorList.size(); i++) {
        message.append(errorList.get(i));
        if (i + 1 < errorList.size()) {
          message.append("\n");
        }
      }

      showError(message.toString());
      return;
    }

    // Usu "www", jeli wystpuje, z adresu pocztkowego.
    startUrl = removeWwwFromUrl(startUrl); 

    // Rozpoczcie wyszukiwania.
    search(logFile, startUrl, maxUrls, searchString);
  }

  private void search(final String logFile, final String startUrl,
    final int maxUrls, final String searchString)
  {
    // Rozpoczcie wyszukiwania w nowym wtku.
    Thread thread = new Thread(new Runnable() {
      public void run() {
        // Wywietlenie kursora zajtoci w trakcie wyszukiwania.
        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

        // Wyczenie pl.
        startTextField.setEnabled(false);
        maxComboBox.setEnabled(false);
        limitCheckBox.setEnabled(false);
        logTextField.setEnabled(false);
        searchTextField.setEnabled(false);
        caseCheckBox.setEnabled(false);

        // Zmiana tekstu przycisku na "Stop."
        searchButton.setText("Stop");

        // Reset statystyk.
        table.setModel(new DefaultTableModel(new Object[][]{},
          new String[]{"URL"}) {
          public boolean isCellEditable(int row, int column)
          {
            return false;
          }
        });
        updateStats(startUrl, 0, 0, maxUrls); 

        // Otwarcie pliku do zapisu wynikw.
        try {
          logFileWriter = new PrintWriter(new FileWriter(logFile));
        } catch (Exception e) {
          showError("Bd otwarcia pliku wynikw.");
          return;
        }

        // Wczenie znacznika wyszukiwania.
        crawling = true;

        // Przeprowadzenie rzeczywistego wyszukiwania.
        crawl(startUrl, maxUrls, limitCheckBox.isSelected(),
          searchString, caseCheckBox.isSelected());

        // Wyczenie wyszukiwania.
        crawling = false;

        // Zamknicie pliku.
        try {
          logFileWriter.close();
        } catch (Exception e) {
          showError("Bd zamykania pliku.");
        }

        // Oznacz wyszukiwanie jako zakoczone.
        crawlingLabel2.setText("Wykonano");

        // Wczenie opcji wyszukiwania.
        startTextField.setEnabled(true);
        maxComboBox.setEnabled(true);
        limitCheckBox.setEnabled(true);
        logTextField.setEnabled(true);
        searchTextField.setEnabled(true);
        caseCheckBox.setEnabled(true);
        
        // Przeczenie znowu na tekst szukania."
        searchButton.setText("Szukaj");

        // Powrt do domylnego kursora.
        setCursor(Cursor.getDefaultCursor());

        // Poka tekst, jeli nie znaleziono wynikw.
        if (table.getRowCount() == 0) {
          JOptionPane.showMessageDialog(SearchCrawler.this,
            "Poszukiwanego tekstu nie znaleziono. Poszukaj innego.",
            "Tekstu nie znaleziono",
            JOptionPane.WARNING_MESSAGE);
        }
      }
    });
    thread.start();
  }

  // Wywietlenie okna dialogowego z bdem.
  private void showError(String message) {
    JOptionPane.showMessageDialog(this, message, "Bd",
      JOptionPane.ERROR_MESSAGE);
  }

  // Aktualizacja statystyk.
  private void updateStats(
    String crawling, int crawled, int toCrawl, int maxUrls)
  {
    crawlingLabel2.setText(crawling);
    crawledLabel2.setText("" + crawled);
    toCrawlLabel2.setText("" + toCrawl);

    // Aktualizacja paska postpu.
    if (maxUrls == -1) {
      progressBar.setMaximum(crawled + toCrawl);
    } else {
      progressBar.setMaximum(maxUrls);
    }
    progressBar.setValue(crawled);

    matchesLabel2.setText("" + table.getRowCount());
  }

  // Dodanie wyniku do tabeli i pliku.
  private void addMatch(String url) {
    // Dodanie adresu do tabeli.
    DefaultTableModel model =
      (DefaultTableModel) table.getModel();
    model.addRow(new Object[]{url});

    // Dodanie adresu do pliku.
    try {
      logFileWriter.println(url);
    } catch (Exception e) {
      showError("Bd zapisu do pliku.");
    }
  }

  // Weryfikacja adresu URL.
  private URL verifyUrl(String url) {
    // Tylko adresy HTTP.
    if (!url.toLowerCase().startsWith("http://"))
      return null;

    // Weryfikacja formatu adresu URL.
    URL verifiedUrl = null;
    try {
      verifiedUrl = new URL(url);
    } catch (Exception e) {
      return null;
    }

    return verifiedUrl;
  }

  // Sprawdzenie, czy robot moe przeszuka dany adres URL.
  private boolean isRobotAllowed(URL urlToCheck) {
    String host = urlToCheck.getHost().toLowerCase();

    // Pobranie listy zabronionych stron z bufora.
    ArrayList disallowList =
      (ArrayList) disallowListCache.get(host);

    // Jeli lista nie w buforze, pobierz j i zbuforuj.
    if (disallowList == null) {
      disallowList = new ArrayList();

      try {
        URL robotsFileUrl =
          new URL("http://" + host + "/robots.txt");

        // Otwarcie poczenia do pobrania pliku robots.txt.
        BufferedReader reader =
          new BufferedReader(new InputStreamReader(
            robotsFileUrl.openStream()));

        // Odczytanie pliku i utworzenie listy zabronionych cieek.
        String line;
        while ((line = reader.readLine()) != null) {
          if (line.indexOf("Disallow:") == 0) {
            String disallowPath =
              line.substring("Disallow:".length());

            // Sprawdzenie, czy s komentarze i usunicie ich.
            int commentIndex = disallowPath.indexOf("#");
            if (commentIndex != - 1) {
              disallowPath =
                disallowPath.substring(0, commentIndex);
            }

            // Usunicie pocztkowych i kocowych spacji.
            disallowPath = disallowPath.trim();

            // Dodanie cieki do listy.
            disallowList.add(disallowPath);
          }
        }

        // Dodanie nowej listy do bufora.
        disallowListCache.put(host, disallowList);
      }
      catch (Exception e) {
        /* Zaoenie, i robot moe dziaa, jeli prba pobrania pliku
           robots.txt zakoczya si bdem. */
        return true;
      }
    }

    /* Przejcie przez list zabronionych, aby sprawdzi
       czy znajduje si tam sprawdzany adres URL. */
    String file = urlToCheck.getFile();
    for (int i = 0; i < disallowList.size(); i++) {
      String disallow = (String) disallowList.get(i);
      if (file.startsWith(disallow)) {
        return false;
      }
    }

    return true;
  }

  // Pobranie podanej strony URL.
  private String downloadPage(URL pageUrl) {
     try {
        // Otwarcie poczenia.
        BufferedReader reader =
          new BufferedReader(new InputStreamReader(
            pageUrl.openStream()));

        // Odczyt strony do bufora.
        String line;
        StringBuffer pageBuffer = new StringBuffer();
        while ((line = reader.readLine()) != null) {
          pageBuffer.append(line);
        }
        
        return pageBuffer.toString();
     } catch (Exception e) {
     }

     return null;
  }

  // Usunicie pocztkowych "www" z adresu URL.
  private String removeWwwFromUrl(String url) {
    int index = url.indexOf("://www.");
    if (index != -1) {
      return url.substring(0, index + 3) +
        url.substring(index + 7);
    }

    return (url);
  }

  // Przejcie przez stron i pobranie wszystkich cz.
  private ArrayList retrieveLinks(
    URL pageUrl, String pageContents, HashSet crawledList,
    boolean limitHost)
  {
    // Kompilacja wzorca wyszukiwania.
    Pattern p =
      Pattern.compile("<a\\s+href\\s*=\\s*\"?(.*?)[\"|>]",
        Pattern.CASE_INSENSITIVE);
    Matcher m = p.matcher(pageContents);

    // Tworzenie listy cz.
    ArrayList linkList = new ArrayList();
    while (m.find()) {
      String link = m.group(1).trim();

      // Pomi puste cze.
      if (link.length() < 1) {
        continue;
      }

      // Pomi cza, ktre s tylko zakotwiczeniami.
      if (link.charAt(0) == '#') {
        continue;
      }

      // Pomi cza typu mailto.
      if (link.indexOf("mailto:") != -1) {
        continue;
      }

      // Pomi cza JavaScript.
      if (link.toLowerCase().indexOf("javascript") != -1) {
        continue;
      }

      // Dodanie wprefiksu w razie potrzeby.
      if (link.indexOf("://") == -1) {
        // Obsuga adresw bezwzgldnych.
        if (link.charAt(0) == '/') {
          link = "http://" + pageUrl.getHost() + link;
        // Obsuga adresw wzgldnych.
        } else {
          String file = pageUrl.getFile();
          if (file.indexOf('/') == -1) {
            link = "http://" + pageUrl.getHost() + "/" + link;
          } else {
            String path =
              file.substring(0, file.lastIndexOf('/') + 1);
            link = "http://" + pageUrl.getHost() + path + link;
          }
        }
      }

      // Usunicie zakotwicze z adresu URL.
      int index = link.indexOf('#');
      if (index != -1) {
        link = link.substring(0, index);
      }

      // Usunicie pocztkowych "www" z adresu serwera.
      link = removeWwwFromUrl(link);

      // Sprawdzenie cza i w przypadku bdu, odrzucenie go.
      URL verifiedLink = verifyUrl(link);
      if (verifiedLink == null) {
        continue;
      }

      /* Jeli jest wczone ograniczenie, pomi adres
         jeli nie znajduje si na tym samym serwerze
         co adres pocztkowyURL. */
      if (limitHost &&
          !pageUrl.getHost().toLowerCase().equals(
            verifiedLink.getHost().toLowerCase()))
      {
        continue;
      }

      // Pomi cze, jeli byo ju sprawdzane.
      if (crawledList.contains(link)) {
        continue;
      }

      // Dodanie cza do listy.
      linkList.add(link);
    }

    return (linkList);
  }

  /* Sprawdzenie, czy szukany tekst znajduje si 
     na danej stronie. */
  private boolean searchStringMatches(
    String pageContents, String searchString,
    boolean caseSensitive)
  {
    String searchContents = pageContents;

    /* Jeli wyszukiwanie bez rozrniania wielkosciliter,
       zmie tekst strony na mae litery. */
    if (!caseSensitive) {
      searchContents = pageContents.toLowerCase();
    }

    // Podzia wyszukiwanego tekstu na wyrazy.
    Pattern p = Pattern.compile("[\\s]+");
    String[] terms = p.split(searchString);

    // Sprawdzenie, czy znaleziono ktry z wyrazw.
    for (int i = 0; i < terms.length; i++) {
      if (caseSensitive) {
        if (searchContents.indexOf(terms[i]) == -1) {
          return false;
        }
      } else {
        if (searchContents.indexOf(terms[i].toLowerCase()) == -1) {
          return false;
        }
      }
    }

    return true;
  }

  // Przeprowadzenie rzeczywistego przechodzenia przez strony
  // w poszukiwaniu tekstu.
  public void crawl(
    String startUrl, int maxUrls, boolean limitHost,
    String searchString, boolean caseSensitive)
  {
    // Ustawienie listy adresw.
    HashSet crawledList = new HashSet();
    LinkedHashSet toCrawlList = new LinkedHashSet();

    // Dodanie adresu poczatkowego do listy.
    toCrawlList.add(startUrl);

    /* Przeprowadzenie przechodzenia przez kolejne strony
       w ptli a do wyczerpania listy adresw. */
    while (crawling && toCrawlList.size() > 0)
    {
      /* Sprawdzenie, czy nie osignito maksymalnej liczby stron
          do przejrzenia.*/
      if (maxUrls != -1) {
        if (crawledList.size() == maxUrls) {
          break;
        }
      }

      // Pobranie adresu na dole listy.
      String url = (String) toCrawlList.iterator().next();

      // Usunicie adresu z listy.
      toCrawlList.remove(url);

      // Konwersja adresu na obiekt URL.
      URL verifiedUrl = verifyUrl(url);

      // Pominicie adresu URL, jeli agenci nie powinni mie do niego dostpu.
      if (!isRobotAllowed(verifiedUrl)) {
        continue;
      }

      // Aktualizacja statystyk.
      updateStats(url, crawledList.size(), toCrawlList.size(),
        maxUrls);

      // Dodanie adresu do listy sprawdzonych stron.
      crawledList.add(url);

      // Pobranie strony z podanego adresu URL.
      String pageContents = downloadPage(verifiedUrl);

      /* Jeli strona zostaa pobrana poprawnie, sprawd jej cza
         oraz zawieranie szukanego tekstu. */
      if (pageContents != null && pageContents.length() > 0)
      {
        // Pobranie listy poprawnych cz znajdujacych si na stronie.
        ArrayList links =
          retrieveLinks(verifiedUrl, pageContents, crawledList,
            limitHost);

        // Dodanie cz do listy adresw do przejrzenia.
        toCrawlList.addAll(links);

        /* Sprawdzenie, czy na stronie znajduje si
           szukany tekst. */
        if (searchStringMatches(pageContents, searchString,
             caseSensitive))
        {
          addMatch(url);
        }
      }

      // Aktualizacja statystyk.
      updateStats(url, crawledList.size(), toCrawlList.size(),
        maxUrls);
    }
  }

  // Uruchomienie agenta wyszukiwania.
  public static void main(String[] args) {
    SearchCrawler crawler = new SearchCrawler();
    crawler.show();
  }
}
