import java.awt.*;
import java.awt.event.*;
import java.net.*;
import java.util.*;
import javax.mail.*;
import javax.mail.internet.*;
import javax.swing.*;
import javax.swing.event.*;

// Klient e-mail.
public class EmailClient extends JFrame
{
  // Model danych tabeli komunikatw.
  private MessagesTableModel tableModel;

  // Tabela komunikatw.
  private JTable table;

  // Obszar tekstowy do wywietlania komunikatw.
  private JTextArea messageTextArea;

  /* Panel dzielony przechowujcy tabel oraz podgld komunikatu. */
  private JSplitPane splitPane;

  // Przyciski zarzdzajce wybranym komunikatem.
  private JButton replyButton, forwardButton, deleteButton;

  // Aktualnie zaznaczony komunikat w tabeli.
  private Message selectedMessage;

  // Znacznik usuwania komunikatu.
  private boolean deleting;

  // To jest sesja JavaMail.
  private Session session;

  // Konstruktor klasy.
  public EmailClient()
  {
    // UStawienie tytuu.
    setTitle("Klient e-mail");

    // Rozmiar okna.
    setSize(640, 480);

    // Obsuga zamknicia okna.
    addWindowListener(new WindowAdapter() {
      public void windowClosing(WindowEvent e) {
        actionExit();
      }
    });

    // Ustawienie menu.
    JMenuBar menuBar = new JMenuBar();
    JMenu fileMenu = new JMenu("Plik");
    fileMenu.setMnemonic(KeyEvent.VK_P);
    JMenuItem fileExitMenuItem = new JMenuItem("Wyjcie",
      KeyEvent.VK_W);
    fileExitMenuItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        actionExit();
      }
    });
    fileMenu.add(fileExitMenuItem);
    menuBar.add(fileMenu);
    setJMenuBar(menuBar);

    // Panel przyciskw.
    JPanel buttonPanel = new JPanel();
    JButton newButton = new JButton("Nowa wiadomo");
    newButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        actionNew ();
      }
    });
    buttonPanel.add(newButton);

    // Ustawienie tabeli komunikatw.
    tableModel = new MessagesTableModel();
    table = new JTable(tableModel);
    table.getSelectionModel().addListSelectionListener(new
      ListSelectionListener() {
      public void valueChanged(ListSelectionEvent e) {
        tableSelectionChanged();
      }
    });
    // Moliwe wybranie tylko jednego wiersza w danym momencie.
    table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

    // UStawienie panelu e-maile.
    JPanel emailsPanel = new JPanel();
    emailsPanel.setBorder(
      BorderFactory.createTitledBorder("Listy e-mail"));
    messageTextArea = new JTextArea();
    messageTextArea.setEditable(false);
    splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT,
      new JScrollPane(table), new JScrollPane(messageTextArea));
    emailsPanel.setLayout(new BorderLayout());
    emailsPanel.add(splitPane, BorderLayout.CENTER);

    // Drugi panel przyciskw.
    JPanel buttonPanel2 = new JPanel();
    replyButton = new JButton("Odpowiedz");
    replyButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        actionReply();
      }
    });
    replyButton.setEnabled(false);
    buttonPanel2.add(replyButton);
    forwardButton = new JButton("Przelij dalej");
    forwardButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        actionForward();
      }
    });
    forwardButton.setEnabled(false);
    buttonPanel2.add(forwardButton);
    deleteButton = new JButton("Usu");
    deleteButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        actionDelete();
      }
    });
    deleteButton.setEnabled(false);
    buttonPanel2.add(deleteButton);

    // Dodanie paneli do okna.
    getContentPane().setLayout(new BorderLayout());
    getContentPane().add(buttonPanel, BorderLayout.NORTH);
    getContentPane().add(emailsPanel, BorderLayout.CENTER);
    getContentPane().add(buttonPanel2, BorderLayout.SOUTH);
  }

  // Wyjcie z programu.
  private void actionExit() {
    System.exit(0);
  }

  // Wykonanie nowego komunikatu.
  private void actionNew () {
    sendMessage(MessageDialog.NEW, null);
  }

  // Wywoywane przy zmianie zaznaczonego wiersza.
  private void tableSelectionChanged() {
    /* Jeli wiersz nie jest usuwany, ustaw zaznaczon wiadomo
       i wywietl j. */
    if (!deleting) {
      selectedMessage =
        tableModel.getMessage(table.getSelectedRow());
      showSelectedMessage();
      updateButtons();
    }
  }

  // Odpowied na wiadomos.
  private void actionReply() {
    sendMessage(MessageDialog.REPLY, selectedMessage);
  }

  // Przesanie wiadomoci dalej.
  private void actionForward() {
    sendMessage(MessageDialog.FORWARD, selectedMessage);
  }

  // Usunicie wiadomoci.
  private void actionDelete() {
    deleting = true;

    try {
      // Usunicie wiadomoci z serwera.
      selectedMessage.setFlag(Flags.Flag.DELETED, true);
      Folder folder = selectedMessage.getFolder();
      folder.close(true);
      folder.open(Folder.READ_WRITE);
    } catch (Exception e) {
      showError("Nie mona usun wiadomoci.", false);
    }

    // Usunicie wiadomoci z tabeli.
    tableModel.deleteMessage(table.getSelectedRow());

    // Aktualizacja okna.
    messageTextArea.setText("");
    deleting = false;
    selectedMessage = null;
    updateButtons();
  }

  // Wysanie wiadomoci.
  private void sendMessage(int type, Message message) {
    // Wywietl okno dialogwe, aby pobra wartoci.
    MessageDialog dialog;
    try {
      dialog = new MessageDialog(this, type, message);
      if (!dialog.display()) {
        // Powr, jeli wyczono okno.
        return;
      }
    } catch (Exception e) {
      showError("Nie mona wysa wiadomoci.", false);
      return;
    }

    try {
      // Utwrz wiadomo z danych z okna.
      Message newMessage = new MimeMessage(session);
      newMessage.setFrom(new InternetAddress(dialog.getFrom()));
      newMessage.setRecipient(Message.RecipientType.TO,
        new InternetAddress(dialog.getTo()));
      newMessage.setSubject(dialog.getSubject());
      newMessage.setSentDate(new Date());
      newMessage.setText(dialog.getContent());

      // Wylij wiadomo.
      Transport.send(newMessage);
    } catch (Exception e) {
      showError("Nie mona wysa wiadomoci.", false);
    }
  }

  // Poka zaznaczon wiadomo w panelu.
  private void showSelectedMessage() {
    // Poka kursor oczekiwania w trakcie pokazywanai wiadomoci.
    setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
    try {
      messageTextArea.setText(
        getMessageContent(selectedMessage));
      messageTextArea.setCaretPosition(0);
    } catch (Exception e) {
      showError("Nie mona pobra wiadomoci.", false);
    } finally {
      // Powrt do domylnego kursora.
      setCursor(Cursor.getDefaultCursor());
    }
  }

  /* Uaktualnienie stanw przyciskw w zaleznoci od
     aktualnie zaznaczonego komunikatu w tabeli. */
  private void updateButtons() {
    if (selectedMessage != null) {
      replyButton.setEnabled(true);
      forwardButton.setEnabled(true);
      deleteButton.setEnabled(true);
    } else {
      replyButton.setEnabled(false);
      forwardButton.setEnabled(false);
      deleteButton.setEnabled(false);
    }
  }

  // Poka okno apliakcji na ekranie.
  public void show() {
    super.show();

    // Aktualizacja panelu i jego podzielenie p na p.
    splitPane.setDividerLocation(.5);
  }

  // Poczenie z serwerem.
  public void connect() {
    // Wywietlenie okna poczenia.
    ConnectDialog dialog = new ConnectDialog(this);
    dialog.show();

    // Tworzenie poczenia URL z podanych danych.
    StringBuffer connectionUrl = new StringBuffer();
    connectionUrl.append(dialog.getType() + "://");
    connectionUrl.append(dialog.getUsername() + ":");
    connectionUrl.append(dialog.getPassword() + "@");
    connectionUrl.append(dialog.getServer() + "/");

    /* Wywietlenie okna dialogowego pobierania wiadomoci. */
    final DownloadingDialog downloadingDialog =
      new DownloadingDialog(this);
    SwingUtilities.invokeLater(new Runnable() {
      public void run() {
        downloadingDialog.show();
      }
    });

    // Ustanowienie sesji JavaMail i poczenie z serwerem.
    Store store = null;
    try {
      // Inicjalizacja sesji JavaMail z serwerem SMTP.
      Properties props = new Properties();
      props.put("mail.smtp.host", dialog.getSmtpServer());
      session = Session.getDefaultInstance(props, null);

      // Poczenie z serwerem e-mail.
      URLName urln = new URLName(connectionUrl.toString());
      store = session.getStore(urln);
      store.connect();
    } catch (Exception e) {
      // Zamknicie okna dialogowego.
      downloadingDialog.dispose();

      // Wywietlenie bdu.
      showError("Bd poczenia.", true);
    }

    // Pobranie nagwkw wiadomoci.
    try {
      // Otwarcie katalogu "INBOX".
      Folder folder = store.getFolder("INBOX");
      folder.open(Folder.READ_WRITE);

      // Pobranie listy komunikatw z folderu.
      Message[] messages = folder.getMessages();

      // Pobranie nagwkw wiadomoci.
      FetchProfile profile = new FetchProfile();
      profile.add(FetchProfile.Item.ENVELOPE);
      folder.fetch(messages, profile);

      // Umieszczenie wiadomoci w tabeli.
      tableModel.setMessages(messages);
    } catch (Exception e) {
      // Zamknicie okna.
      downloadingDialog.dispose();

      // Bd pobierania.
      showError("Bd pobierania wiadomoci.", true);
    }

    // Zamknicie okna pobierania.
    downloadingDialog.dispose();
  }

  // Wywietlenie okna bdu i ewentualne wyczenie programu.
  private void showError(String message, boolean exit) {
    JOptionPane.showMessageDialog(this, message, "Bd",
      JOptionPane.ERROR_MESSAGE);
    if (exit)
      System.exit(0);
  }

  // Pobranie zawartoci wiadomoci.
  public static String getMessageContent(Message message)
    throws Exception {
    Object content = message.getContent();
    if (content instanceof Multipart) {
      StringBuffer messageContent = new StringBuffer();
      Multipart multipart = (Multipart) content;
      for (int i = 0; i < multipart.getCount(); i++) {
        Part part = (Part) multipart.getBodyPart(i);
          if (part.isMimeType("text/plain")) {
            messageContent.append(part.getContent().toString());
          }
        }
      return messageContent.toString();
    } else {
      return content.toString();
    }
  }

  // Uruchomienie programu.
  public static void main(String[] args) {
    EmailClient client = new EmailClient();
    client.show();

    // Wywietlenie okna poczenia.
    client.connect();
  }
}
