import java.awt.*;
import java.awt.event.*;
import java.net.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;

// Zarzdca pobierania.
public class DownloadManager extends JFrame
  implements Observer
{
  // Dodanie pola tekstowego pobierania.
  private JTextField addTextField;

  // Model danych tabeli pobierania.
  private DownloadsTableModel tableModel;
  
  // Tabela z pobraniami.
  private JTable table;

  // Przyciski zarzdzania poszczeglnymi pobraniami.
  private JButton pauseButton, resumeButton;
  private JButton cancelButton, clearButton;

  // Aktualnie zaznaczone pobieranie.
  private Download selectedDownload;

  // Informacja o tym, czy usunito zaznaczenie z elementu tabeli.
  private boolean clearing;

  // Konstruktor DownloadManager.
  public DownloadManager()
  {
    // Tytu aplikacji.
    setTitle("Meneder pobierania");

    // Ustaw rozmiar okna.
    setSize(640, 480);

    // Obsuga zdarzenia zamknicia okna.
    addWindowListener(new WindowAdapter() {
      public void windowClosing(WindowEvent e) {
        actionExit();
      }
    });

    // Ustawienie menu.
    JMenuBar menuBar = new JMenuBar();
    JMenu fileMenu = new JMenu("Plik");
    fileMenu.setMnemonic(KeyEvent.VK_P);
    JMenuItem fileExitMenuItem = new JMenuItem("Wyjcie",
      KeyEvent.VK_W);
    fileExitMenuItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        actionExit();
      }
    });
    fileMenu.add(fileExitMenuItem);
    menuBar.add(fileMenu);
    setJMenuBar(menuBar);

    // Wykonanie panelu pobierania.
    JPanel addPanel = new JPanel();
    addTextField = new JTextField(30);
    addPanel.add(addTextField);
    JButton addButton = new JButton("Dodaj adres");
    addButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        actionAdd();
      }
    });
    addPanel.add(addButton);

    // Tabela pobra.
    tableModel = new DownloadsTableModel();
    table = new JTable(tableModel);
    table.getSelectionModel().addListSelectionListener(new
      ListSelectionListener() {
      public void valueChanged(ListSelectionEvent e) {
        tableSelectionChanged();
      }
    });
    // Moliwo zaznaczenia w danym momencie tylko jednego wiersza.
    table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
  
    // Ustawienie ProgressBar jako renderera kolumny pustpu.
    ProgressRenderer renderer = new ProgressRenderer(0, 100);
    renderer.setStringPainted(true); // pokazanie tekstu postpu
    table.setDefaultRenderer(JProgressBar.class, renderer);

    // Ustawienie takiej wysokoci wiersza, aby zmieci JProgressBar.
    table.setRowHeight(
      (int) renderer.getPreferredSize().getHeight());

    // Ustawienie panelu Pliki.
    JPanel downloadsPanel = new JPanel();
    downloadsPanel.setBorder(
      BorderFactory.createTitledBorder("Pliki"));
    downloadsPanel.setLayout(new BorderLayout());
    downloadsPanel.add(new JScrollPane(table),
      BorderLayout.CENTER);

    // Ustawienie panelu przyciskw.
    JPanel buttonsPanel = new JPanel();
    pauseButton = new JButton("Zatrzymaj");
    pauseButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        actionPause();
      }
    });
    pauseButton.setEnabled(false);
    buttonsPanel.add(pauseButton);
    resumeButton = new JButton("Wznw");
    resumeButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        actionResume();
      }
    });
    resumeButton.setEnabled(false);
    buttonsPanel.add(resumeButton);
    cancelButton = new JButton("Anuluj");
    cancelButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        actionCancel();
      }
    });
    cancelButton.setEnabled(false);
    buttonsPanel.add(cancelButton);
    clearButton = new JButton("Wyczy");
    clearButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        actionClear();
      }
    });
    clearButton.setEnabled(false);
    buttonsPanel.add(clearButton);

    // Dodanie paneli do wywietlania.
    getContentPane().setLayout(new BorderLayout());
    getContentPane().add(addPanel, BorderLayout.NORTH);
    getContentPane().add(downloadsPanel, BorderLayout.CENTER);
    getContentPane().add(buttonsPanel, BorderLayout.SOUTH);
  }

  // Wyjcie z programu.
  private void actionExit() {
    System.exit(0);
  }

  // Dodanie nowego pliku.
  private void actionAdd() {
    URL verifiedUrl = verifyUrl(addTextField.getText());
    if (verifiedUrl != null) {
      tableModel.addDownload(new Download(verifiedUrl));
      addTextField.setText(""); // zresetowanie pola dodawania
    } else {
      JOptionPane.showMessageDialog(this,
        "Bdny adres URL", "Bd",
        JOptionPane.ERROR_MESSAGE);
    }
  }

  // Waryfikacja adresu URL.
  private URL verifyUrl(String url) {
    // Tylko adresy HTTP.
    if (!url.toLowerCase().startsWith("http://"))
      return null;

    // Weryfikacja formatu URL.
    URL verifiedUrl = null;
    try {
      verifiedUrl = new URL(url);
    } catch (Exception e) {
      return null;
    }

    // Upewnienie si, i adres wskazuje plik.
    if (verifiedUrl.getFile().length() < 2)
      return null;

    return verifiedUrl;
  }

  // Wywoywane przy zmianie zaznaczonego wiersza.
  private void tableSelectionChanged() {
    /* Usunicie si z powiadamiania z
       ostatnio zaznaczonego wiersza. */
    if (selectedDownload != null)
      selectedDownload.deleteObserver(DownloadManager.this);

    /* Jeli to nie usuwanie wiersza,
       ustaw, otrzymywanie powiadomie od tego wiersza. */
    if (!clearing) {
      selectedDownload =
        tableModel.getDownload(table.getSelectedRow());
      selectedDownload.addObserver(DownloadManager.this);
      updateButtons();
    }
  }

  // Wstrzymanie wybranego pliku.
  private void actionPause() {
    selectedDownload.pause();
    updateButtons();
  }

  // Wznowienie pobierania pliku.
  private void actionResume() {
    selectedDownload.resume();
    updateButtons();
  }

  // Anulowanie pobierania.
  private void actionCancel() {
    selectedDownload.cancel();
    updateButtons();
  }

  // Usunicie pobierania.
  private void actionClear() {
    clearing = true;
    tableModel.clearDownload(table.getSelectedRow());
    clearing = false;
    selectedDownload = null;
    updateButtons();
  }

  /* Aktualizacja aktualnego stanu przyciskw w zalenoci
     od stanu zaznaczonego pobierania. */
  private void updateButtons() {
    if (selectedDownload != null) {
      int status = selectedDownload.getStatus();
      switch (status) {
        case Download.DOWNLOADING:
          pauseButton.setEnabled(true);
          resumeButton.setEnabled(false);
          cancelButton.setEnabled(true);
          clearButton.setEnabled(false);
          break;
        case Download.PAUSED:
          pauseButton.setEnabled(false);
          resumeButton.setEnabled(true);
          cancelButton.setEnabled(true);
          clearButton.setEnabled(false);
          break;
        case Download.ERROR:
          pauseButton.setEnabled(false);
          resumeButton.setEnabled(true);
          cancelButton.setEnabled(false);
          clearButton.setEnabled(true);
          break;
        default: // CAY lub ANULOWANY
          pauseButton.setEnabled(false);
          resumeButton.setEnabled(false);
          cancelButton.setEnabled(false);
          clearButton.setEnabled(true);
      }
    } else {
      // Nie jest zaznaczone adne pobieranie.
      pauseButton.setEnabled(false);
      resumeButton.setEnabled(false);
      cancelButton.setEnabled(false);
      clearButton.setEnabled(false);
    }
  }

  /* Aktualizacja po otrzymaniu informacji o dowolnych zmianach
     w klasie Download. */
  public void update(Observable o, Object arg) {
    // Aktualizacja przyciskw w przypadku zmiany zaznaczonego pobierania.
    if (selectedDownload != null && selectedDownload.equals(o))
      updateButtons();
  }

  // Uruchomienie programu.
  public static void main(String[] args) {
    DownloadManager manager = new DownloadManager();
    manager.show();
  }
}
