// Prosty interpreter jzyka SBasic. 
 
import java.io.*; 
import java.util.*; 
 
// Klasa wyjtkw dla bdw interpretera.   
class InterpreterException extends Exception {   
  String errStr; // opisuje bd  
  
  public InterpreterException(String str) {  
    errStr = str;  
  }    
   
  public String toString() {   
    return errStr;  
  }   
}   
   
// Waciowa klasa interpretera. 
class SBasic {   
  final int PROG_SIZE = 10000; // maksymalny rozmiar programu 
 
  // Rodzaje tokenw.  
  final int NONE = 0;  
  final int DELIMITER = 1;  
  final int VARIABLE = 2;  
  final int NUMBER = 3;  
  final int COMMAND = 4;  
  final int QUOTEDSTR = 5; 
  
  // Rodzaje bdw.  
  final int SYNTAX = 0;  
  final int UNBALPARENS = 1;  
  final int NOEXP = 2;  
  final int DIVBYZERO = 3;  
  final int EQUALEXPECTED = 4;  
  final int NOTVAR = 5;  
  final int LABELTABLEFULL = 6;  
  final int DUPLABEL = 7;  
  final int UNDEFLABEL = 8;  
  final int THENEXPECTED = 9;  
  final int TOEXPECTED = 10;  
  final int NEXTWITHOUTFOR = 11;  
  final int RETURNWITHOUTGOSUB = 12;  
  final int MISSINGQUOTE = 13; 
  final int FILENOTFOUND = 14;  
  final int FILEIOERROR = 15;  
  final int INPUTIOERROR = 16; 
 
  // Wewntrzna reprezentacja sw kluczowych SBasic. 
  final int UNKNCOM = 0; 
  final int PRINT = 1; 
  final int INPUT = 2; 
  final int IF = 3; 
  final int THEN = 4; 
  final int FOR = 5; 
  final int NEXT = 6; 
  final int TO = 7; 
  final int GOTO = 8; 
  final int GOSUB = 9; 
  final int RETURN = 10; 
  final int END = 11; 
  final int EOL = 12; 
 
  // Token oznaczajcy koniec programu.  
  final String EOP = "\0";  
  
  // Kody operatorw podwjnych, na przykad <=. 
  final char LE = 1; 
  final char GE = 2; 
  final char NE = 3; 
 
  // Tablica zmiennych.   
  private double vars[]; 
  
  // Ta klasa czy sowa kluczowe z ich tokenami. 
  class Keyword {  
    String keyword; // posta tekstowa  
    int keywordTok; // reprezentacja wewntrzna  
 
    Keyword(String str, int t) { 
      keyword = str; 
      keywordTok = t; 
    } 
  } 
 
  /* Tabela sw kluczowych wraz z ich reprezentacj wewntrzn. 
     Wszystkie sowa kluczowe pisane s z maych liter. */ 
  Keyword kwTable[] = {  
    new Keyword("print", PRINT), // w tej tabeli.  
    new Keyword("input", INPUT), 
    new Keyword("if", IF), 
    new Keyword("then", THEN), 
    new Keyword("goto", GOTO), 
    new Keyword("for", FOR), 
    new Keyword("next", NEXT), 
    new Keyword("to", TO), 
    new Keyword("gosub", GOSUB), 
    new Keyword("return", RETURN), 
    new Keyword("end", END) 
  }; 
 
  private char[] prog; // odnosi si do tablicy programu 
  private int progIdx; // aktualny indeks w programie 
 
  private String token; // aktualny token   
  private int tokType;  // typ aktualnego tokenu   
 
  private int kwToken; // wewntrzna reprezentacja sowa kluczowego 
 
  // Obsuga ptli FOR. 
  class ForInfo { 
    int var; // zmienna licznika  
    double target; // warto docelowa  
    int loc; // indeks w kodzie rdowym w celu uzyskania zaptlenia  
  } 
 
  // stos dla ptli FOR. 
  private Stack fStack; 
 
  // Definiuje wpisy tablicy etykiet. 
  class Label { 
    String name; // etykieta  
    int loc; // indeks pooenia etykiety w pliku rdowym  
    public Label(String n, int i) { 
      name = n; 
      loc = i; 
    } 
  } 
 
  // Mapa dla etykiet. 
  private TreeMap labelTable; 
 
  // Stos dla polece gosub. 
  private Stack gStack; 
 
  // operatory relacji 
  char rops[] = { 
    GE, NE, LE, '<', '>', '=', 0 
  }; 
 
  /* Utworzenie tekstu zawierajcego operatory relacji 
     aby uproci ich pniejsze sprawdzanie. */ 
  String relops = new String(rops); 
 
  // Konstruktor SBasic. 
  public SBasic(String progName)  
      throws InterpreterException { 
 
    char tempbuf[] = new char[PROG_SIZE]; 
    int size; 
 
    // Wczytanie programu do wykonania. 
    size = loadProgram(tempbuf, progName); 
 
    if(size != -1) { 
      // Utworzenie tablicy o odpowiednim rozmiarze dla programu. 
      prog = new char[size]; 
 
      // Skopiowanie programu do tablicy. 
      System.arraycopy(tempbuf, 0, prog, 0, size); 
    } 
  } 
 
  // Wczytanie programu. 
  private int loadProgram(char[] p, String fname) 
    throws InterpreterException 
  { 
    int size = 0; 
 
    try { 
      FileReader fr = new FileReader(fname); 
 
      BufferedReader br = new BufferedReader(fr); 
 
      size = br.read(p, 0, PROG_SIZE); 
 
      fr.close(); 
    } catch(FileNotFoundException exc) { 
      handleErr(FILENOTFOUND); 
    } catch(IOException exc) { 
      handleErr(FILEIOERROR); 
    }    
 
    // Jeli plik koczy si znakiem EOF, cofnij. 
    if(p[size-1] == (char) 26) size--; 
 
    return size; // zwr rozmiar programu 
  } 
 
  // Wykonaj program. 
  public void run() throws InterpreterException { 
 
    // Inicjalizacja dla nowego przebiegu. 
    vars = new double[26];   
    fStack = new Stack(); 
    labelTable = new TreeMap(); 
    gStack = new Stack(); 
    progIdx = 0; 
 
    scanLabels(); // znajd etykiety w programie  
 
    sbInterp(); // wykonaj program 
 
  } 
 
  // Punkt wejcia w interpreter SBasic. 
  private void sbInterp() throws InterpreterException 
  { 
 
    // Jest to gwna ptla interpretera. 
    do { 
      getToken(); 
      // Sprawdzanie przypisania. 
      if(tokType==VARIABLE) { 
        putBack(); // zwrcenie zmiennej do strumienia wejciowego  
        assignment(); // obsuga instrukcji przypisania  
      } 
      else // czy sowo kluczowe 
        switch(kwToken) { 
          case PRINT: 
            print(); 
            break; 
          case GOTO: 
            execGoto(); 
            break; 
          case IF: 
            execIf(); 
            break; 
          case FOR: 
            execFor(); 
            break; 
          case NEXT: 
            next(); 
            break; 
          case INPUT: 
            input(); 
            break; 
          case GOSUB: 
            gosub(); 
            break; 
          case RETURN: 
            greturn(); 
            break; 
          case END: 
            return; 
        } 
    } while (!token.equals(EOP)); 
  } 
 
  // Znajdowanie wszystkich etykiet.  
  private void scanLabels() throws InterpreterException 
  { 
    int i; 
    Object result; 
 
    // Sprawdzenie, czy pierwszy token w pliku to etykieta. 
    getToken(); 
    if(tokType==NUMBER)  
      labelTable.put(token, new Integer(progIdx)); 
 
    findEOL(); 
 
    do {      
      getToken(); 
      if(tokType==NUMBER) {// musi to by numer wiersza 
        result = labelTable.put(token, 
                                new Integer(progIdx)); 
        if(result != null) 
          handleErr(DUPLABEL); 
      } 
 
      // Jeli nie linia pusta, znajd nastpn.  
      if(kwToken != EOL) findEOL(); 
    } while(!token.equals(EOP)); 
    progIdx = 0; // ustaw indeks na pocztek programu 
  } 
 
  // Znajd pocztek nastpnego wiersza. 
  private void findEOL() 
  { 
    while(progIdx < prog.length && 
          prog[progIdx] != '\n') ++progIdx; 
    if(progIdx < prog.length) progIdx++; 
  } 
 
  // Przypisanie wartoci do zmiennej. 
  private void assignment() throws InterpreterException 
  { 
    int var; 
    double value; 
    char vname; 
 
    // Pobranie nazwy zmiennej. 
    getToken(); 
    vname = token.charAt(0); 
 
    if(!Character.isLetter(vname)) { 
      handleErr(NOTVAR); 
      return; 
    } 
 
    // Konwersja na indeks z tablicy zmiennych. 
    var = (int) Character.toUpperCase(vname) - 'A'; 
    
    // Pobranie znaku rwnoci. 
    getToken(); 
    if(!token.equals("=")) { 
      handleErr(EQUALEXPECTED); 
      return; 
    } 
 
    // Pobranie wartoci do przypisania. 
    value = evaluate(); 
 
    // Przypisanie wartoci. 
    vars[var] = value; 
  } 
 
  // Wykonanie prostszej wersji instrukcji PRINT. 
  private void print() throws InterpreterException 
  { 
    double result; 
    int len=0, spaces; 
    String lastDelim = ""; 
 
    do { 
      getToken(); // pobranie nastpnego elementu  
      if(kwToken==EOL || token.equals(EOP)) break; 
 
      if(tokType==QUOTEDSTR) { // czy tekst 
        System.out.print(token); 
        len += token.length(); 
        getToken(); 
      } 
      else { // czy wyraenie 
        putBack(); 
        result = evaluate(); 
        getToken(); 
        System.out.print(result); 
 
        // Dodanie dugoci wyjcia do cznej wartoci. 
        Double t = new Double(result); 
        len += t.toString().length(); // zapamitanie dugoci 
      } 
      lastDelim = token; 
 
      // Jeli przecinek, przejd do nastpnego zatrzymania. 
      if(lastDelim.equals(",")) { 
        // oblicz liczb spacji do nastpnego przesuniecia 
        spaces = 8 - (len % 8);  
        len += spaces; // dodaj spacje do dugoci 
        while(spaces != 0) {  
          System.out.print(" "); 
          spaces--; 
        } 
      } 
      else if(token.equals(";")) { 
        System.out.print(" "); 
        len++; 
      } 
      else if(kwToken != EOL && !token.equals(EOP)) 
        handleErr(SYNTAX);  
    } while (lastDelim.equals(";") || lastDelim.equals(",")); 
 
    if(kwToken==EOL || token.equals(EOP)) { 
      if(!lastDelim.equals(";") && !lastDelim.equals(",")) 
        System.out.println(); 
    } 
    else handleErr(SYNTAX);  
  } 
 
  // Wykonanie instrukcji GOTO.  
  private void execGoto() throws InterpreterException 
  { 
    Integer loc; 
 
    getToken(); // pobranie nazwy etykiety  
 
    // Znalezienie etykiety w tablicy. 
    loc = (Integer) labelTable.get(token); 
 
    if(loc == null) 
      handleErr(UNDEFLABEL); // nie znaleziono etykiety 
    else // przeniesnienie wykonywania programu w nowe miejsce 
      progIdx = loc.intValue(); 
  } 
 
  // Wykonanie instrukcji IF. 
  private void execIf() throws InterpreterException 
  { 
    double result; 
 
    result = evaluate(); // pobranie wartoci wyraenia 
 
    /* Jeli wynik to true (niezerowy), 
       do nastpnego wiersza programu. */ 
    if(result != 0.0) {  
      getToken(); 
      if(kwToken != THEN) { 
        handleErr(THENEXPECTED); 
        return; 
      } // w przeciwnym razie wykonana zostanie docelowa instrukcja  
    } 
    else findEOL(); // znajd pocztek nastpnego wiersza  
  } 
 
  // Wykonaj ptl FOR.  
  private void execFor() throws InterpreterException 
  { 
    ForInfo stckvar = new ForInfo(); 
    double value; 
    char vname; 
 
    getToken(); // odczytaj zmienn sterujc  
    vname = token.charAt(0); 
    if(!Character.isLetter(vname)) { 
      handleErr(NOTVAR); 
      return; 
    } 
 
    // Zapisz indeks zmiennej sterujcej. 
    stckvar.var = Character.toUpperCase(vname) - 'A'; 
 
    getToken(); // odczytaj znak rwnoci 
    if(token.charAt(0) != '=') { 
      handleErr(EQUALEXPECTED); 
      return; 
    } 
 
    value = evaluate(); // pobierz warto pocztkow  
 
    vars[stckvar.var] = value; 
 
    getToken(); // odczytaj i pomi tekst TO 
    if(kwToken != TO) handleErr(TOEXPECTED); 
 
    stckvar.target = evaluate(); // pobierz warto docelow 
 
    /* Jeli ptl mona wykona przynajmniej raz, 
       umie j na stosie. */ 
    if(value >= vars[stckvar.var]) {  
      stckvar.loc = progIdx; 
      fStack.push(stckvar); 
    } 
    else // w przeciwnym razie pomi kod ptli 
      while(kwToken != NEXT) getToken(); 
  } 
 
  // Wykonanie polecenia NEXT.  
  private void next() throws InterpreterException 
  { 
    ForInfo stckvar; 
 
    try { 
      // Pobranie informacji dla tej ptli For. 
      stckvar = (ForInfo) fStack.pop(); 
      vars[stckvar.var]++; // inkrenetacja zmiennej sterujcej 
   
      // Jeli koniec, wyjcie. 
      if(vars[stckvar.var] > stckvar.target) return;  
 
      // W przeciwnym razie zapamietanie informacji. 
      fStack.push(stckvar); 
      progIdx = stckvar.loc;  // ptla  
    } catch(EmptyStackException exc) { 
      handleErr(NEXTWITHOUTFOR); 
    } 
  } 
 
  // Wykonanie prostej postaci INPUT.  
  private void input() throws InterpreterException 
  { 
    int var; 
    double val = 0.0; 
    String str; 
 
    BufferedReader br = new 
      BufferedReader(new InputStreamReader(System.in)); 
 
    getToken(); // sprawdzenie, czy wystpuje pytanie  
    if(tokType == QUOTEDSTR) { 
      // jeli tak, wywietl je i sprawd istnienie znaku przecinka  
      System.out.print(token); 
      getToken(); 
      if(!token.equals(",")) handleErr(SYNTAX); 
      getToken(); 
    } 
    else System.out.print("? "); // w przeciwnym razie wywietl znak ? 
 
    // pobierz zmienn wejciow  
    var =  Character.toUpperCase(token.charAt(0)) - 'A'; 
 
    try { 
      str = br.readLine(); 
      val = Double.parseDouble(str); // odczytaj warto 
    } catch (IOException exc) { 
      handleErr(INPUTIOERROR); 
    } catch (NumberFormatException exc) { 
      /* Mona zechcie obsuy ten bd inaczej 
         ni inne bdy interpretera. */ 
      System.out.println("Bdne wejcie."); 
    } 
 
    vars[var] = val; // zapisz wejcie  
  } 
 
  // Wykonanie instrukcji GOSUB.  
  private void gosub() throws InterpreterException 
  { 
    Integer loc; 
 
    getToken(); 
 
    // Znajd etykiet do wywoania. 
    loc = (Integer) labelTable.get(token);  
 
    if(loc == null) 
      handleErr(UNDEFLABEL); // etykieta nie jest zdefiniowana  
    else { 
      // Zapisz miejsce powrotu. 
      gStack.push(new Integer(progIdx)); 
 
      // Rozpocznij wykonywanie programu w miejscu loc. 
      progIdx = loc.intValue(); 
    } 
  } 
 
  // Powrt z  GOSUB. 
  private void greturn() throws InterpreterException 
  { 
    Integer t; 
 
    try { 
      // Przywrcenie indeksu programu. 
      t = (Integer) gStack.pop(); 
      progIdx = t.intValue(); 
    } catch(EmptyStackException exc) { 
      handleErr(RETURNWITHOUTGOSUB); 
    } 
 
  } 
 
  // **************** Analizator wyrae **************** 
 
  // Punkt wejcia do analizatora.   
  private double evaluate() throws InterpreterException 
  {   
    double result = 0.0;   
 
    getToken();   
    if(token.equals(EOP))  
      handleErr(NOEXP); // brak wyraenia   
  
    // Przeanalizuj i wylicz wyraenie.  
    result = evalExp1();   
 
    putBack();   
 
    return result;   
  }   
     
  // Przetwarzanie operatorw relacji.  
  private double evalExp1() throws InterpreterException 
  { 
    double l_temp, r_temp, result; 
    char op; 
 
    result = evalExp2(); 
    // Jeli na kocu programu, powr. 
    if(token.equals(EOP)) return result; 
 
    op = token.charAt(0);   
 
    if(isRelop(op)) { 
      l_temp = result; 
      getToken(); 
      r_temp = evalExp1(); 
      switch(op) { // przeprowadzenie dziaa relacji  
        case '<': 
          if(l_temp < r_temp) result = 1.0; 
          else result = 0.0; 
          break; 
        case LE: 
          if(l_temp <= r_temp) result = 1.0; 
          else result = 0.0; 
          break; 
        case '>': 
          if(l_temp > r_temp) result = 1.0; 
          else result = 0.0; 
          break; 
        case GE: 
          if(l_temp >= r_temp) result = 1.0; 
          else result = 0.0; 
          break; 
        case '=': 
          if(l_temp == r_temp) result = 1.0; 
          else result = 0.0; 
          break; 
        case NE: 
          if(l_temp != r_temp) result = 1.0; 
          else result = 0.0; 
          break; 
      } 
    } 
    return result; 
  } 
  
  // Dodanie lub odjcie elementw.   
  private double evalExp2() throws InterpreterException  
  {   
    char op;   
    double result;  
    double partialResult;   
  
    result = evalExp3();   
  
    while((op = token.charAt(0)) == '+' || op == '-') {   
      getToken();   
      partialResult = evalExp3();   
      switch(op) {   
        case '-':   
          result = result - partialResult;   
          break;   
        case '+':   
          result = result + partialResult;   
          break;   
      }   
    }   
    return result;  
  }   
     
  // Pomnoenie lub podzielenie elementw.   
  private double evalExp3() throws InterpreterException  
  {   
    char op;   
    double result;  
    double partialResult;   
     
    result = evalExp4();   
  
    while((op = token.charAt(0)) == '*' ||   
           op == '/' || op == '%') {   
      getToken();   
      partialResult = evalExp4();   
      switch(op) {   
        case '*':   
          result = result * partialResult;   
          break;   
        case '/':   
          if(partialResult == 0.0)   
            handleErr(DIVBYZERO);   
          result = result / partialResult;   
          break;   
        case '%':   
          if(partialResult == 0.0)   
            handleErr(DIVBYZERO);   
          result = result % partialResult;   
          break;   
      }   
    }   
    return result;  
  }   
     
  // Wykonanie potgowania.   
  private double evalExp4() throws InterpreterException  
  {   
    double result;  
    double partialResult;  
    double ex;   
    int t;   
     
    result = evalExp5();   
  
    if(token.equals("^")) {   
      getToken();   
      partialResult = evalExp4();   
      ex = result;   
      if(partialResult == 0.0) {   
        result = 1.0;   
      } else   
        for(t=(int)partialResult-1; t > 0; t--)   
          result = result * ex;   
    }   
    return result;  
  }   
     
  // Obliczenie jednoargumentowych + lub -.   
  private double evalExp5() throws InterpreterException  
  {   
    double result;  
    String  op;   
  
    op = "";   
    if((tokType == DELIMITER) &&   
        token.equals("+") || token.equals("-")) {   
      op = token;   
      getToken();   
    }   
    result = evalExp6();   
  
    if(op.equals("-")) result = -result;  
  
    return result;   
  }   
     
  // Obsuga wyraenia w nawiasach.   
  private double evalExp6() throws InterpreterException  
  {   
    double result;  
  
    if(token.equals("(")) {   
      getToken();   
      result = evalExp2();   
      if(!token.equals(")"))   
        handleErr(UNBALPARENS);   
      getToken();   
    }   
    else result = atom();   
  
    return result;  
  }   
     
  // Pobranie wartoci zmiennej lub wyraenia.   
  private double atom() throws InterpreterException   
  {   
    double result = 0.0;  
 
    switch(tokType) {   
      case NUMBER:   
        try {   
          result = Double.parseDouble(token);   
        } catch (NumberFormatException exc) {   
          handleErr(SYNTAX);   
        }   
        getToken();   
        break;  
      case VARIABLE:   
        result = findVar(token);   
        getToken();   
        break;   
      default:   
        handleErr(SYNTAX);   
        break;   
    }   
    return result;  
  }   
     
   // Zwrcenie wartoci zmiennej.   
  private double findVar(String vname) 
    throws InterpreterException  
  {   
    if(!Character.isLetter(vname.charAt(0))){   
      handleErr(SYNTAX);   
      return 0.0;   
    }   
    return vars[Character.toUpperCase(vname.charAt(0))-'A'];   
  }   
   
  // Zwrcenie tokenu do strumienia wejciowego.   
  private void putBack()     
  {   
    if(token == EOP) return;  
    for(int i=0; i < token.length(); i++) progIdx--;   
  }   
   
  // Obsuga bdw.   
  private void handleErr(int error) 
    throws InterpreterException  
  {   
    String[] err = {   
      "Bd skadni",   
      "Brak jednego z nawiasw",   
      "Brak wyraenia",   
      "Dzielenie przez zero", 
      "Oczekiwany znak rwnoci", 
      "Nie jest to zmienna", 
      "Pena tablica etykiet", 
      "Duplikacja etykiety", 
      "Niezdefiniowana etykieta", 
      "Oczekiwane THEN", 
      "Oczekiwane TO", 
      "NEXT bez FOR", 
      "RETURN bez GOSUB", 
      "Potrzebny cudzysw zamykajcy", 
      "Pliku nie znaleziono", 
      "Bd we-wy w trakcie wczytywania pliku", 
      "Bd we-wy w instrukcji INPUT" 
    };   
   
    throw new InterpreterException(err[error]);   
  }   
     
  // Pobranie nastpnego tokenu.   
  private void getToken() throws InterpreterException 
  {   
    char ch; 
 
    tokType = NONE;   
    token = "";   
    kwToken = UNKNCOM; 
 
    // Sprawdzenie koca programu.   
    if(progIdx == prog.length) { 
      token = EOP;  
      return;  
    }  
 
    // Pominicie biaych spacji.  
    while(progIdx < prog.length &&   
          isSpaceOrTab(prog[progIdx])) progIdx++; 
   
    // Jeli na kocu byy spacje, koczymy program.  
    if(progIdx == prog.length) {  
      token = EOP;  
      tokType = DELIMITER; 
      return;  
    }  
 
    if(prog[progIdx] == '\r') { // obsuga crlf 
      progIdx += 2; 
      kwToken = EOL; 
      token = "\r\n"; 
      return; 
    } 
 
    // Sprawdzenie operatorw relacji. 
    ch = prog[progIdx]; 
    if(ch == '<' || ch == '>') { 
      if(progIdx+1 == prog.length) handleErr(SYNTAX); 
 
      switch(ch) { 
        case '<': 
          if(prog[progIdx+1] == '>') { 
            progIdx += 2;; 
            token = String.valueOf(NE); 
          } 
          else if(prog[progIdx+1] == '=') { 
            progIdx += 2; 
            token = String.valueOf(LE); 
          } 
          else { 
            progIdx++; 
            token = "<"; 
          } 
          break; 
        case '>': 
          if(prog[progIdx+1] == '=') { 
            progIdx += 2;; 
            token = String.valueOf(GE); 
          } 
          else { 
            progIdx++; 
            token = ">"; 
          } 
          break; 
      } 
      tokType = DELIMITER; 
      return; 
    } 
   
    if(isDelim(prog[progIdx])) { 
      // Jest operatorem.   
      token += prog[progIdx];   
      progIdx++;   
      tokType = DELIMITER;   
    }   
    else if(Character.isLetter(prog[progIdx])) {  
      // Jest zmienn lub sowem kluczowym. 
      while(!isDelim(prog[progIdx])) {   
        token += prog[progIdx];   
        progIdx++;   
        if(progIdx >= prog.length) break;   
      }   
 
      kwToken = lookUp(token); 
      if(kwToken==UNKNCOM) tokType = VARIABLE;   
      else tokType = COMMAND; 
    } 
    else if(Character.isDigit(prog[progIdx])) { 
      // Jest liczb. 
      while(!isDelim(prog[progIdx])) {   
        token += prog[progIdx];   
        progIdx++;   
        if(progIdx >= prog.length) break;   
      }   
      tokType = NUMBER;   
    }   
    else if(prog[progIdx] == '"') { 
      // Jest tekstem w cudzysowie. 
      progIdx++; 
      ch = prog[progIdx]; 
      while(ch !='"' && ch != '\r') { 
        token += ch; 
        progIdx++; 
        ch = prog[progIdx]; 
      } 
      if(ch == '\r') handleErr(MISSINGQUOTE); 
      progIdx++;  
      tokType = QUOTEDSTR; 
    } 
    else { // nieznany znak koczy program 
      token = EOP;  
      return;  
    }  
  }   
     
  // Zwraca true, jeli c jest przerywnikiem.   
  private boolean isDelim(char c)   
  {   
    if((" \r,;<>+-/*%^=()".indexOf(c) != -1))   
      return true;   
    return false;   
  }   
 
  // Zwraca true, jeli c jest spacj lub znakiem tabulacji. 
  boolean isSpaceOrTab(char c)  
  { 
    if(c == ' ' || c =='\t') return true; 
    return false; 
  } 
 
  // Zwraca true, jeli c jest operatorem relacji. 
  boolean isRelop(char c) { 
    if(relops.indexOf(c) != -1) return true; 
    return false; 
  } 
 
  /* Szuka wewntrznej reprezentacji tokenu w tablicy. */ 
  private int lookUp(String s) 
  { 
    int i; 
 
    // Konwersja na mae litery. 
    s = s.toLowerCase(); 
   
    // Sprawdzenie, czy token jest sowem kluczowym. 
    for(i=0; i < kwTable.length; i++) 
      if(kwTable[i].keyword.equals(s)) 
        return kwTable[i].keywordTok; 
    return UNKNCOM; // nieznane sowo kluczowe 
  } 
}

