﻿// searchPage.js
(function () {
  "use strict";

  WinJS.Binding.optimizeBindingReferences = true;

  var appModel = Windows.ApplicationModel;
  var appViewState = Windows.UI.ViewManagement.ApplicationViewState;
  var nav = WinJS.Navigation;
  var ui = WinJS.UI;
  var utils = WinJS.Utilities;
  var searchPageURI = "/pages/Search/searchPage.html";

  ui.Pages.define(searchPageURI, {
    _filters: [],
    _lastSearch: "",

    // Ta funkcja jest wywoływana, gdy użytkownik otworzy tę stronę.
    // Umieszcza dane aplikacyjne w elementach strony.
    ready: function (element, options) {
      var listView = element.querySelector(".resultslist").winControl;
      listView.itemTemplate = element.querySelector(".itemtemplate");
      listView.oniteminvoked = this._itemInvoked;
      this._handleQuery(element, options);
      listView.element.focus();
    },

    // Ta funkcja aktualizuje układ strony w odpowiedzi na zmiany w viewState.
    updateLayout: function (element, viewState, lastViewState) {
      /// <param name="element" domElement="true" />

      var listView = element.querySelector(".resultslist").winControl;
      if (lastViewState !== viewState) {
        if (lastViewState === appViewState.snapped || viewState === appViewState.snapped) {
          var handler = function (e) {
            listView.removeEventListener("contentanimating", handler, false);
            e.preventDefault();
          }
          listView.addEventListener("contentanimating", handler, false);
          var firstVisible = listView.indexOfFirstVisible;
          this._initializeLayout(listView, viewState);
          if (firstVisible >= 0 && listView.itemDataSource.list.length > 0) {
            listView.indexOfFirstVisible = firstVisible;
          }
        }
      }
    },

    // Ta funkcja filtruje przeszukiwane dane za pomocą zadanego filtru.
    _applyFilter: function (filter, originalResults) {
      if (filter.results === null) {
        filter.results = originalResults.createFiltered(filter.predicate);
      }
      return filter.results;
    },

    // Ta funkcja odpowiada na wybranie nowego filtru przez użytkownika.
    // Aktualizuje listę wyboru i wyświetlane wyniki.
    _filterChanged: function (element, filterIndex) {
      var filterBar = element.querySelector(".filterbar");
      var listView = element.querySelector(".resultslist").winControl;

      utils.removeClass(filterBar.querySelector(".highlight"), "highlight");
      utils.addClass(filterBar.childNodes[filterIndex], "highlight");

      element.querySelector(".filterselect").selectedIndex = filterIndex;

      listView.itemDataSource = this._filters[filterIndex].results.dataSource;
    },

    _generateFilters: function () {
      this._filters = [];
      this._filters.push({ results: null, text: "Wszystkie", predicate: function (item) { return true; } });

      // Dodanie grup do każdego kanału
      Data.groups.forEach(function (feed) {
        // FIltrowanie według kanału
        this._filters.push({
          results: null,
          text: feed.title,
          predicate: function (item) {
            // Jeżeli kanał odpowiada grupie elementów, to jest ona w tym filtrze
            return item.group === feed;
          }
        });
      }, this);

    },

    // Ta funkcja wykonuje każdy krok wymagany podczas wyszukiwania
    _handleQuery: function (element, args) {
      var originalResults;
      this._lastSearch = args.queryText;
      WinJS.Namespace.define("searchPage", { markText: WinJS.Binding.converter(this._markText.bind(this)) });
      this._initializeLayout(element.querySelector(".resultslist").winControl, Windows.UI.ViewManagement.ApplicationView.value);
      this._generateFilters();
      originalResults = this._searchData(args.queryText);
      if (originalResults.length === 0) {
        document.querySelector('.filterarea').style.display = "none";
      } else {
        document.querySelector('.resultsmessage').style.display = "none";
      }
      this._populateFilterBar(element, originalResults);
      this._applyFilter(this._filters[0], originalResults);
    },

    // Ta funkcja aktualizuje obiekt ListView z nowym układem.
    _initializeLayout: function (listView, viewState) {
      /// <param name="listView" value="WinJS.UI.ListView.prototype" />

      if (viewState === appViewState.snapped) {
        listView.layout = new ui.ListLayout();
        document.querySelector(".titlearea .pagetitle").textContent = '“' + this._lastSearch + '”';
        document.querySelector(".titlearea .pagesubtitle").textContent = "";
      } else {
        listView.layout = new ui.GridLayout();

        // Do zrobienia: zmienić "App Name" na nazwę aplikacji
        document.querySelector(".titlearea .pagetitle").textContent = "RssReader";
        document.querySelector(".titlearea .pagesubtitle").textContent = "Wyniki wyszukiwania “" + this._lastSearch + '”';
      }
    },

    _itemInvoked: function (args) {
      args.detail.itemPromise.done(function itemInvoked(item) {
        // TODO: Przejdź do elementu, który został wywołany.
        navigateToItem(item.data);
      });
    },


    // Ta funkcja koloruje wyszukiwane słowo. Odwołanie w /pages/Search/searchPage.html
    // jako część szablonu elementów ListView
    _markText: function (text) {
      return text.replace(this._lastSearch, "<mark>" + this._lastSearch + "</mark>");
    },

    // Ta funkcja generuje listę wyboru filtru
    _populateFilterBar: function (element, originalResults) {
      var filterBar = element.querySelector(".filterbar");
      var listView = element.querySelector(".resultslist").winControl;
      var li, option, filterIndex;

      filterBar.innerHTML = "";
      for (filterIndex = 0; filterIndex < this._filters.length; filterIndex++) {
        this._applyFilter(this._filters[filterIndex], originalResults);

        li = document.createElement("li");
        li.filterIndex = filterIndex;
        li.tabIndex = 0;
        li.textContent = this._filters[filterIndex].text + " (" + this._filters[filterIndex].results.length + ")";
        li.onclick = function (args) { this._filterChanged(element, args.target.filterIndex); }.bind(this);
        li.onkeyup = function (args) {
          if (args.key === "Enter" || args.key === "Spacebar")
            this._filterChanged(element, args.target.filterIndex);
        }.bind(this);
        utils.addClass(li, "win-type-interactive");
        utils.addClass(li, "win-type-x-large");
        filterBar.appendChild(li);

        if (filterIndex === 0) {
          utils.addClass(li, "highlight");
          listView.itemDataSource = this._filters[filterIndex].results.dataSource;
        }

        option = document.createElement("option");
        option.value = filterIndex;
        option.textContent = this._filters[filterIndex].text + " (" + this._filters[filterIndex].results.length + ")";
        element.querySelector(".filterselect").appendChild(option);
      }

      element.querySelector(".filterselect").onchange = function (args) { this._filterChanged(element, args.currentTarget.value); }.bind(this);
    },

    // Ta funkcja wypełnia lilstę WinJS.Binding.List wynikami wyszukiwania
    // według zadanego zapytania
    _searchData: function (queryText) {
      var originalResults;
      // Tylko wtedy, gdy są elementy do wyszukania
      if (window.Data && window.Data.items && window.Data.items.length > 0) {
        var searchExpression = new RegExp(queryText, "i"); // Use query to do a regex search
        originalResults = Data.items.createFiltered(function (item) {
          return searchExpression.test(item.title);
        });
      } else {
        originalResults = new WinJS.Binding.List();
      }
      return originalResults;
    }
  });

  WinJS.Application.addEventListener("activated", function (args) {
    if (args.detail.kind === appModel.Activation.ActivationKind.search) {
      args.setPromise(ui.processAll().then(function () {
        if (!nav.location) {
          nav.history.current = { location: Application.navigator.home, initialState: {} };
        }

        return nav.navigate(searchPageURI, { queryText: args.detail.queryText });
      }));
    }
  });

  // zapytanie wysłane
  appModel.Search.SearchPane.getForCurrentView().onquerysubmitted = function (args) { nav.navigate(searchPageURI, args); };

  // obsłuż nawigację
  function navigateToItem(item) {
    // określ liczbę elementów w grupie
    var items = Data.getItemsFromGroup(item.group);
    var itemIndex = items.indexOf(item);

    // jeżeli element został znaleziony, przejdź do kanału i elementu
    if (item) {
      WinJS.Navigation.navigate("/pages/split/split.html", { groupKey: item.group.key, selectedIndex: itemIndex });
    }
  }

  // sugestie
  appModel.Search.SearchPane.getForCurrentView().onsuggestionsrequested = function (args) {
    var req = args.request;

    // określ żądanie i teksty alternatywne
    var alternatives = args.linguisticDetails.queryTextAlternatives;
    var query = args.queryText;
    var queries = [query];
    alternatives.forEach(function (alt) { queries.push(alt) });

    // pobierz sugestie i dołącz je do systemu
    var suggestions = Data.getSuggestedResults(queries, 5);
    req.searchSuggestionCollection.appendSearchSeparator("RSS Items");
    suggestions.forEach(function (item) {
      //req.searchSuggestionCollection.appendQuerySuggestion(item.title);
      req.searchSuggestionCollection.appendResultSuggestion(item.title,
        item.description,
        item.id, // element do pokazania, gdy zostanie wybrana sugestia
        Data.appIconStreamReference,
        item.title); // wszystkie parametry z wyjątkiem ostatniego
    });
  };

  appModel.Search.SearchPane.getForCurrentView().onresultsuggestionchosen = function (args) {
    // Jeżeli użytkownik wybierze sugestię, idź do odpowiedniej grupy blogów
    // z tagiem elementu jako jego identyfikatorem
    navigateToItem(Data.resolveItemReference(args.tag));
  };

})();
