/**
  * @brief Wypisz "Hello World" na szeregowe we/wy
  * przy użyciu bufora cyklicznego.
  *
  * @note Wprowadzono tu sytuację wyścigu w celu zademonstrowania,
  * jak nie należy pisać takiego programu.
  */

#include <stdbool.h>
#include "stm32f0xx_nucleo.h"
#include "stm32f0xx.h"

const char hello[] = "Hello World!\r\n";   // Wysyłany komunikat.
int current; // Znak w wysyłanym komunikacie.

UART_HandleTypeDef uartHandle;      // Inicjacja interfejsu UART.

#define BUFFER_SIZE 8   // Rozmiar bufora danych.

struct circularBuffer {
    uint32_t putIndex;      // Miejsce, w którym umieocimy nastepny znak.
    uint32_t getIndex;      // Miejsce, z którego pobierzemy nastepny znak.
    uint32_t nCharacters;   // Liczba znaków w buforze.
    char data[BUFFER_SIZE]; // Dane znajdujące się w buforze.
};

// Prosty, klasyczny bufor dla układu USART2.
volatile struct circularBuffer buffer = {0,0,0, {'\0'}};

/**
  * @brief Funkcja ta jest wykonywana w razie wystąpienia błędu.
  *
  * Jedyne, co robi, to miganie diodą LED.
  */
void Error_Handler(void)
{
    /* Włącz diodę LED2. */
    HAL_GPIO_WritePin(LED2_GPIO_PORT, LED2_PIN, GPIO_PIN_SET);

    while (true)
    {
    // Przełącz stan diody LED2.
        HAL_GPIO_TogglePin(LED2_GPIO_PORT, LED2_PIN);
        HAL_Delay(1000);        // Poczekaj 1 sekundę.
    }
}

/**
 * Obsłuż przerwanie USART2.
 *
 * Tajemniczo wywoływana przez system przerwan mikroukładu.
 * Jej nazwy nie wolno zmieniać ze względu na kod startowy,
 * który wypełnia wektor przerwań.
 */
void USART2_IRQHandler(void)
{
    if ((uartHandle.Instance->ISR & USART_ISR_TXE) != 0) {
        if (buffer.nCharacters == 0) {
            // Wyłącz przerwanie.
            uartHandle.Instance->CR1 &= ~(USART_CR1_TXEIE);
            return;
        }
        uartHandle.Instance->TDR = buffer.data[buffer.getIndex];     // Wyślij znak do UART.
        ++buffer.getIndex;
        if (buffer.getIndex == BUFFER_SIZE)
            buffer.getIndex = 0;

        --buffer.nCharacters;

        if (buffer.nCharacters == 0)
            uartHandle.Instance->CR1 &= ~(USART_CR1_TXEIE);
        return;
    }
    // Ponieważ jedynym włączonym przez nas przerwaniem było TXE, nigdy nie powinniśmy 
    // tu trafić. Jeśli włączymy inne przerwania, będziemy musieli umieścić
    // tutaj kod ich obsługi.
}

/**
 * Umieść znak w buforze komunikacji szeregowej.
 *
 * @param ch Znak do wysłania.
 */
void myPutchar(const char ch)
{
    // Poczekaj, aż będzie miejsce.
    while (buffer.nCharacters == BUFFER_SIZE)
        continue;
    buffer.data[buffer.putIndex] = ch;
    ++buffer.putIndex;
    if (buffer.putIndex == BUFFER_SIZE)
        buffer.putIndex = 0;

    // Dodaliśmy kolejny znak.
    ++buffer.nCharacters;
    // To by było na tyle.
    
    // Włącz (ponownie) przerwanie.
    uartHandle.Instance->CR1 |= USART_CR1_TXEIE;
}

/**
 * Nasza wersja funkcji puts.
 *
 * Wyprowadza na wyjście dokładnie taki łańcuch znaków, jaki został podany.
 *
 * @param str Łańcuch znaków do wysłania.
 *
 * @note Zakłada się, że wskaźnik str nie ma wartości NULL.
 */
void myPuts(const char* str)
{
    for (/* zmienna str jest ustawiona */; *str != '\0'; ++str)
        myPutchar(*str);
}

/**
 * Zainicjuj LED2 (po to, by można było zasygnalizować błąd miganiem na czerwono).
 */
void led2_Init(void)
{
    // Inicjacja zegara LED.
    LED2_GPIO_CLK_ENABLE();

    GPIO_InitTypeDef GPIO_LedInit;      // Inicjacja LED.
    // Zainicjuj LED.
    GPIO_LedInit.Pin = LED2_PIN;
    GPIO_LedInit.Mode = GPIO_MODE_OUTPUT_PP;
    GPIO_LedInit.Pull = GPIO_PULLUP;
    GPIO_LedInit.Speed = GPIO_SPEED_FREQ_HIGH;
    HAL_GPIO_Init(LED2_GPIO_PORT, &GPIO_LedInit);
}

/**
 * Zainicjuj interfejs UART2 w celu wyprowadzania danych.
 */
void uart2_Init(void)
{
    // Inicjacja interfejsu UART.
    //  UART2 -- ten połączony z interfejsem USB programatora ST-LINK.
    uartHandle.Instance = USART2;
    uartHandle.Init.BaudRate = 9600;                    // Prędkość 9600 bodów.
    uartHandle.Init.WordLength = UART_WORDLENGTH_8B;    // 8 bitów na znak.
    uartHandle.Init.StopBits = UART_STOPBITS_1;         // 1 bit stopu.
    uartHandle.Init.Parity = UART_PARITY_NONE;          // Brak kontroli parzystości.
    uartHandle.Init.Mode = UART_MODE_TX_RX;             // Wysyłanie i odbiór.
    uartHandle.Init.HwFlowCtl = UART_HWCONTROL_NONE;    // Brak sprzętowego sterowania przepływem.

    // Poddaj strumień przychodzący nadpróbkowaniu.
    uartHandle.Init.OverSampling = UART_OVERSAMPLING_16;

    // Nie stosuj jednej próbki na bit.
    uartHandle.Init.OneBitSampling = UART_ONE_BIT_SAMPLE_DISABLE;

    // Żadnych zaawansowanych funkcji.
    uartHandle.AdvancedInit.AdvFeatureInit = UART_ADVFEATURE_NO_INIT;
    /*
     * Dla tych z Was, którzy łączą się przez emulator terminala: powyższe parametry
     * przekładają się na format transmisji 9600,8,N,1.
     */

    if (HAL_UART_Init(&uartHandle) != HAL_OK)
    {
        Error_Handler();
    }
}

int main(void)
{
    HAL_Init(); // Zainicjuj sprzęt.
    led2_Init();
    uart2_Init();
    // Poinformuj mikroukład, że chcemy uaktywnić wektor przerwania
    // interfejsu USART2.
    NVIC_EnableIRQ(USART2_IRQn);

    // Kontynuuj wysyłanie komunikatu przez długi czas.
    for (;;) {
        myPuts(hello);
        HAL_Delay(500);
    }
}

/**
 * Tajemna funkcja wywoływana przez warstwę HAL w celu faktycznego
 * izainicjowania interfejsu UART. W tym przypadku musimy
 * przestawić piny UART na tryb alternatywny, by działały jako
 * linie UART, a nie GPIO..
 *
 * @note: : Działa tylko z interfejsem UART2, tym połączonym z konwerterem
 * USB/port szeregowy.
 *
 * @param uart Dane UART.
 */
void HAL_UART_MspInit(UART_HandleTypeDef* uart)
{
    GPIO_InitTypeDef GPIO_InitStruct;
    if(uart->Instance == USART2)
    {
        /* Włączenie zegara urządzenia peryferyjnego. */
        __HAL_RCC_USART2_CLK_ENABLE();

        /*
         * Konfigurowanie bitów GPIO interfejsu USART2.
         * PA2     ------> USART2_TX
         * PA3     ------> USART2_RX
         */
        GPIO_InitStruct.Pin = GPIO_PIN_2|GPIO_PIN_3;
        GPIO_InitStruct.Mode = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull = GPIO_NOPULL;
        GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_LOW;
        // Funkcja alternatywna -- związana z UART.
        GPIO_InitStruct.Alternate = GPIO_AF1_USART2;
        HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
    }

}

/**
 * Tajemna funkcja wywoływana przez warstwę HAL w celu przywrócenia
 * urządzenia UART do stanu niezainicjowanego.  Nigdy tego nie robimy, ale umieściliśmy tu tę
 * funkcję gwoli kompletności.
 *
 * @note:  Działa tylko z interfejsem UART2, tym połączonym z konwerterem
 * USB/port szeregowy.
 *
 * @param uart The UART information
 */
void HAL_UART_MspDeInit(UART_HandleTypeDef* uart)
{
    if(uart->Instance == USART2)
    {
        /* Wyłączenie  zegara urządzenia peryferyjnego. */
        __HAL_RCC_USART2_CLK_DISABLE();

        /*
         * Konfigurowanie bitów GPIO interfejsu USART2.
         * PA2     ------> USART2_TX
         * PA3     ------> USART2_RX
         */
        HAL_GPIO_DeInit(GPIOA, GPIO_PIN_2|GPIO_PIN_3);
    }
}
