package com.corejdo.examples.model;

import java.util.*;
import javax.jdo.*;

/**
 * @author mccammon
 *
 */
public class Publisher implements InstanceCallbacks {

  private String name;
  private Set books;

  private final static Publisher[] EMPTY_ARRAY = new Publisher[0];
  
  /**
   * <p>Wyszukuje wydawcw o podanej nazwie.</p>
   * <p>Zwraca tablic instancji klasy Publisher posiadajcych odpowiedni 
   * nazw. Jeli nie istnieje adna taka instancja, to metoda zwraca pust
   * tablic.</p>
   * 
   * @param pm instancja PersistenceManager uywana do realizacji zapytania
   * @param name poszukiwana nazwa wydawcy
   * @return tablica instancji klasy Publisher posiadajcych okrelon nazw; jeli
   * nie istnieje adna taka instancja, to metoda zwraca pust tablic.
   */
  public static Publisher[] findByName(PersistenceManager pm, String name) {

    Publisher[] publishers = EMPTY_ARRAY;

    Query query = pm.newQuery(Publisher.class, "this.name == name");

    query.declareParameters("String name");

    Collection result = (Collection) query.execute(name);

    publishers = (Publisher[]) result.toArray(publishers);

    query.close(result);

    return publishers;
  }

  /**
   * <p>Tworzy instancj o podanej nazwie.<p>
   * 
   * @param name nazwa
   */
  public Publisher(String name) {

    this.name = name;
  }
  
  protected Publisher() {}

  /**
   * <p>Zwraca nazw wydawcy.</p>
   * 
   * @return nazwa
   */
  public String getName() {
    
    return name;
  }

  /**
   * <p>Okrela nazw wydawcy.</p>
   * 
   * @param name nazwa
   */
  public void setName(String name) {
    
    this.name = name;
  }

  /**
   * <p>Zwraca kolekcj ksiek wydanych przez wydawc.</p>
   * <p>Kolekcja ta nie moe by modyfikowana.Modyfikacj kolekcji ksiek 
   * wydanych przez wydawc umoliwiaj metody {@link #addBook} i
   * {@link #removeBook}.</p>
   * 
   * @return kolekcja instancji klasy Book; jeli wydawca nie wyda 
   * jeszcze adnych ksiek, to zwracana jest pusta kolekcja.
   */
  public Collection getBooks() {
    
    return (
      books != null
        ? Collections.unmodifiableSet(books)
        : Collections.EMPTY_SET);
  }

  /**
   * <p>Dodaje ksik do kolekcji danego wydawcy.</p>
   * 
   * @param book ksika
   */
  public void addBook(Book book) {
    
    if (books == null) {

      books = new HashSet();
    }

    books.add(book);

    book.setPublisher(this);
  }
  
  /**
   * <p>Usuwa ksik z kolekcji danego wydawcy.</p>
   * 
   * @param book ksika
   */
  public void removeBook(Book book) {
   
    if (books != null) {
      
      books.remove(book);
      
      book.setPublisher(null);
      
      if (books.isEmpty()) {
        
        books = null;
      }
    } 
  }
  
  /**
   * <p>Nie jest uywana.</p>
   * 
   * @see javax.jdo.InstanceCallbacks#jdoPostLoad()
   */
  public void jdoPostLoad() {}

  /**
   * <p>Nie jest uywana.</p>
   * 
   * @see javax.jdo.InstanceCallbacks#jdoPreClear()
   */
  public void jdoPreClear() {}

  /**
   * @see javax.jdo.InstanceCallbacks#jdoPreDelete()
   */
  public void jdoPreDelete() {

    if (books != null) {

      Iterator iter = books.iterator();
      
      while (iter.hasNext()) {
        
        Book book = (Book) iter.next();
        
        book.setPublisher(null);
      }
      
      books = null;
    }
  }

  /**
   * <p>Nie jest uywana.</p>
   * 
   * @see javax.jdo.InstanceCallbacks#jdoPreStore()
   */
  public void jdoPreStore() {}
}
