package com.corejdo.casestudy.usecase;

import javax.jdo.JDOUserException;
import javax.jdo.PersistenceManager;

import com.corejdo.casestudy.model.User;
import com.corejdo.casestudy.tools.CmdLine;
/**
 * Abstrakcyjna klasa bazowa wszystkich przypadkw zastosowa.
 * Tworzy instancj PersistenceManager, rozpoczyna transakcj i zatwierdza j
 * lub odwouje. Klasy pochodne musz jedynie dostarcza implementacji
 * metody <tt>_execute()</tt>.
 *
 * @author Heiko Bobzin, Keiron McCammon, Sameer Tyagi, Michael Vorburger
 * @version 1.0
 */

public abstract class AbstractUseCase
{
    private Context context;

    public AbstractUseCase() {
    }

    final Context getContext()
    {
        if (context == null) {
            throw new RuntimeException("Brak kontekstu.");
        }
        return context;
    }

    final void setContext(Context context)
    {
        this.context = context;
    }

    final PersistenceManager getPM()
    {
        return getContext().getPM();
    }

    /**
     * Tworzy kontekst transakcji, rozpoczyna transakcj i zatwierdza j
     * po pomylnym wykonaniu metody <tt>_execute</tt>.
     * Jeli klasa pochodna zgosia wyjtek, to transakcja jest odwoywana.
     */
    public final void execute()
    {
        if (!isAllowed()) throw new UseCaseException("Niedozwolony.");
        getPM().currentTransaction().begin();
        boolean active = true;
        try {
            _execute();
            getPM().currentTransaction().commit();
            active = false;
        }
        finally {
            if (active) {
                try {
                    getPM().currentTransaction().rollback();
                    active = false;
                }
                catch (JDOUserException ex)
                {
                    ex.printStackTrace();
                }
            }
        }
    }

    /**
     * Kontrola uprawnie uytkownika. Klasy pochodne przesaniaj t implementacj
     * i zwracaj warto false, jeli dostp nie jest dozwolony.
     */
    public boolean isAllowed()
    {
        return true;
    }

    /**
     * Implementuje podstawowe operacje dla danego przypadku zastosowania.
     * Jest wywoywana przez metod <tt>excecute()</tt> w kontekcie transakcji.
     */
    protected abstract void _execute();

    /**
     * Wersja uruchamiana z wiersza polece.
     */
    protected abstract void run(String args[]);

    /**
     * Metoda wywoywana przez metod main klasy pochodnej
     * w celu inicjacji dziaania w wierszu polece.
     * @param args
     */
    void runCmdLine(String args[])
    {
        this.context = Main.mainContext;
        CmdLine.init(args);
        run(args);
    }

    final User getUser()
    {
        User user = this.getContext().user;
        if (user == null) {
            throw new UseCaseException("Najpierw musisz si zalogowa.");
        }
        return user;
    }

}

