package com.corejdo.examples.model;

import java.util.*;
import javax.jdo.*;

/**
 * @author mccammon
 *
 */
public class Book implements InstanceCallbacks {

  private String title;
  private String isbn;
  private Date publicationDate;
  private Set authors;
  private Publisher publisher;

  /* Pusta tablica zwracana przez metod findByTitle() */ 
  private final static Book[] EMPTY_ARRAY = new Book[0];
  
  /**
   * <p>Wyszukuje wszystkie ksiki o podanym tytule.</p>
   * <p>Do realizacji zapytania uywa dostarczonej instancji PersistenceManager 
   * oraz podanego tytuu ksiki. Zwraca tablic instancji klasy Book 
   * posiadajcych okrelony tytu. Jeli w bazie nie ma instancji o takim tytule,
   * to metoda zwraca pust tablic.</p>
   * 
   * @param pm instancja PersistenceManager uywana do realizacji zapytania
   * @param title tytu poszukiwanej ksiki
   * @return tablica instancji klasy Book o podanym tytule; jeli nie istnieje adna taka
   * instancja, to metoda zwraca pust tablic.
   */
  public static Book[] findByTitle(PersistenceManager pm, String title) {

    Book[] books = EMPTY_ARRAY;

    Query query = pm.newQuery(Book.class, "this.title == title");

    query.declareParameters("String title");

    Collection result = (Collection) query.execute(title);

    books = (Book[]) result.toArray(books);
    
    query.close(result);

    return books;
  }

  /**
   * <p>Tworzy instancj klasy Book o okrelonym tytule, indeksie isbn 
   * i dacie wydania.</p>
   * <p>Po utworzeniu instancja klasy Book powinna zosta powizana 
   * z instancjami klas reprezentujcych autorw i wydawcw.</p>
   * 
   * @param title tytu ksiki
   * @param isbn indeks isbn
   * @param publicationDate data wydania
   */
  public Book(
    String title,
    String isbn,
    Date publicationDate) {

    this.title = title;
    this.isbn = isbn;
    this.publicationDate = publicationDate;
  }

  /**
   * <p>Tworzy instancj klasy Book o okrelonym tytule, indeksie isbn </p>
   * <p>Po utworzeniu instancja klasy Book powinna zosta powizana 
   * z instancjami klas reprezentujcych autorw i wydawcw.</p>
   * 
   * @param title tytu
   * @param isbn indeks isbn
   */
  public Book(String title, String isbn) {
    
    this.title = title;
    this.isbn = isbn;
  }
  
  protected Book() {}

  /**
   * <p>Zwraca kolekcj autorw, ktrzy napisali ksik.</p>
   * <p>Kolekcja ta nie moe by modyfikowana.</p>
   * 
   * @return kolekcja instancji klasy Author; jeli ksika reprezentowana przez 
   * instancj  klasy Book nie posiada jeszcze autorw, to metoda zwraca pust kolekcj.
   */
  public Collection getAuthors() {

    return (
      authors != null
        ? Collections.unmodifiableSet(authors)
        : Collections.EMPTY_SET);
  }

  /**
   * <p>Zwraca indeks ISBN.</p>
   * 
   * @return indeks ISBN
   */
  public String getISBN() {

    return isbn;
  }

  /**
   * <p>Nadaje warto indeksowi ISBN.</p>
   * 
   * @param indeks ISBN
   */
  public void setISBN(String isbn) {

    this.isbn = isbn;
  }

  /**
   * <p>Zwraca dat wydania.</p>
   * 
   * @return data wydania
   */
  public Date getPublicationDate() {

    return publicationDate;
  }

  /**
   * <p>Okrela dat wydania.</p>
   * 
   * @param publicationDate data wydania
   */
  public void setPublicationDate(Date publicationDate) {

    this.publicationDate = publicationDate;
  }

  /**
   * <p>Zwraca tytu ksiki.</p>
   * 
   * @return tytu
   */
  public String getTitle() {

    return title;
  }

  /**
   * <p>Nadaje tytu ksice.<p>
   * 
   * @param title tytu
   */
  public void setTitle(String title) {

    this.title = title;
  }

  /**
   * <p>Zwraca wydawc ksiki.</p>
   * 
   * @return wydawca lub warto null jeli ksika nie posiada jeszcze wydawcy
   */
  public Publisher getPublisher() {

    return publisher;
  }

  /**
   * <p>Okrela wydawc ksiki.</p>
   * 
   * @param publisher wydawca
   */
  public void setPublisher(Publisher publisher) {

    this.publisher = publisher;
  }

  /**
   * <p>Dodaje autora do kolekcji autorw ksiki.</p>
   * 
   * @param author autor
   */
  void addAuthor(Author author) {

    if (authors == null) {
      
      authors = new HashSet();
    }
    
    authors.add(author);
  }

  /**
   * <p>Usuwa autora z kolekcji autorw ksiki.</p>
   * 
   * @param author autor
   */
  void removeAuthor(Author author) {

    if (authors != null) {
      
      authors.remove(author);
      
      if (authors.isEmpty()) {
        
        authors = null;
      }
    }    
  }
  
  /**
   * <p>Nie jest uywana.</p>
   * 
   * @see javax.jdo.InstanceCallbacks#jdoPostLoad()
   */
  public void jdoPostLoad() {}

  /**
   * <p>Nie jest uywana.</p>
   * 
   * @see javax.jdo.InstanceCallbacks#jdoPreClear()
   */
  public void jdoPreClear() {}

  /**
   * <p>Ponisza metoda wywoywana jest zwrotnie i uywana w celu jawnego usunicia instancji powizanych
   * z usuwan instancj.</p>
   * <p>Gwarantuje, e ksika zostaje usunita ze zbiorw ksiek wszystkich jej autorw
   * znajdujcych si w kolekcji autorw ksiki.</p>
   * 
   * @see javax.jdo.InstanceCallbacks#jdoPreDelete()
   */
  public void jdoPreDelete() {
    
    if (authors != null) {
      
      Iterator iter = authors.iterator();
      
      while (iter.hasNext()) {
        
        Author author = (Author) iter.next();
        
        author.removeBook(this);
      }
    }
    
    if (publisher != null) {
      
      publisher.removeBook(this);
    }
  }

  /**
   * <p>Nie jest uywana.</p>
   * 
   * @see javax.jdo.InstanceCallbacks#jdoPreStore()
   */
  public void jdoPreStore() {}  
}
