package com.corejdo.examples.model;

import java.util.*;
import javax.jdo.*;

/**
 * @author mccammon
 *
 */
public class Author implements InstanceCallbacks {

  protected String name;
  protected Address address;
  protected Set books;

  /* Pusta tablica zwracana przez metod findByName() */ 
  private final static Author[] EMPTY_ARRAY = new Author[0];

  /**
   * <p>Wyszukuje autorw o podanym nazwisku.</p>
   * <p>Uywa dostarczonej instancji PersistenceManager do realizacji zapytania
   * o autorw o podanym nazwisku. Zwraca tablic instancji klasy Author
   * posiadajcych dan warto pola name. Jeli w bazie nie ma takich instancji, to metoda zwraca
   * pust tablic.</p>
   * 
   * @param pm instancja PersistenceManager uywana do wykonania zapytania
   * @param name nazwisko poszukiwanych autorw
   * @return tablica instancji klasy Author posiadajcych odpowiedni warto pola name; jeli
   * brak takich instancji klasy Author, to metoda zwraca pust tablic.
   */
  public static Author[] findByName(PersistenceManager pm, String name) {

    Author[] authors = EMPTY_ARRAY;

    Query query = pm.newQuery(Author.class, "this.name == name");

    query.declareParameters("String name");

    Collection result = (Collection) query.execute(name);

    authors = (Author[]) result.toArray(authors);

    query.close(result);

    return authors;
  }

  /**
   * <p>Tworzy instancj klasy Author o podanym nazwisku i adresie.</p>
   * 
   * @param name nazwisko
   * @param address adres
   */
  public Author(String name, Address address) {

    this.name = name;
    this.address = address;
  }
  
  /**
   * <p>Tworzy instancj klasy Author o podanym nazwisku.</p>
   * 
   * @param name nazwisko
   */
  public Author(String name) {
    
    this.name = name;
  }

  protected Author() {}

  /**
   * <p>Zwraca nazwisko autora.</p>
   * 
   * @return nazwisko
   */
  public String getName() {

    return name;
  }

  /**
   * <p>Okrela nazwisko autora.</p>
   * 
   * @param name nazwisko
   */
  public void setName(String name) {

    this.name = name;
  }

  /**
   * <p>Zwraca instancj klasy Address reprezentujc adres autora.</p>
   * 
   * @return instancja klasy Address lub warto null jeli adres nie zosta okrelony
   */
  public Address getAddress() {

    return address;
  }

  /**
   * <p>Zwraca kolekcj ksiek napisanych przez autora.</p>
   * <p>Kolekcja ta nie moe by modyfikowana. Kolekcja instancji klasy Book dla danego
   * autora moe by modyfikowana za pomoc metod {@link #addBook addBook()} i
   * {@link #removeBook removeBook()}.</p>
   * 
   * @return kolekcja instancji Book (nie moe by modyfikowana); jeli autor nie 
   * posiada adnego dorobku, to metoda zwraca pust kolekcj.
   */
  public Collection getBooks() {

    return (
      books != null
        ? Collections.unmodifiableSet(books)
        : Collections.EMPTY_SET);
  }

  /**
   * <p>Dodaje ksik do kolekcji ksiek napisanych przez autora.</p>
   * <p>Autor zostanie rwnie umieszczony na licie autorw danej ksiki.</p>
   * 
   * @param book ksika
   */
  public void addBook(Book book) {

    if (books == null) {

      books = new HashSet();
    }

    books.add(book);

    book.addAuthor(this);
  }

  /**
   * <p>Usuwa ksik z kolekcji ksiek napisanych przez autora.</p>
   * <p>Autor zostaje rwnie usunity z listy autorw ksiki.</p>
   * 
   * @param book ksika
   */
  public void removeBook(Book book) {

    if (books != null) {

      books.remove(book);

      book.removeAuthor(this);
      
      if (books.isEmpty()) {
        
        books = null;
      }
    }
  }

  /**
   * <p>Nie jest uywana.</p>
   * 
   * @see javax.jdo.InstanceCallbacks#jdoPostLoad()
   */
  public void jdoPostLoad() {}

  /**
   * <p>Nie jest uywana.</p>
   * 
   * @see javax.jdo.InstanceCallbacks#jdoPreClear()
   */
  public void jdoPreClear() {}

  /**
   * <p>Ponisza metoda wywoywana jest zwrotnie i wykorzystywana do niejawnego usunicia instancji
   * powizanych z usuwan instancj.</p>
   * <p>Usuwa zwizan z autorem instancj klasy Address oraz usuwa autora
   * z listy autorw kadej ksiki znajdujcej si w kolekcji ksiek napisanych
   * przez autora.</p>
   * 
   * @see javax.jdo.InstanceCallbacks#jdoPreDelete()
   */
  public void jdoPreDelete() {

    JDOHelper.getPersistenceManager(this).deletePersistent(address);

    if (books != null) {

      Iterator iter = books.iterator();

      while (iter.hasNext()) {

        Book book = (Book) iter.next();

        book.removeAuthor(this);
      }
      
      books = null;
    }
  }

  /**
   * <p>Nie jest uywana.</p>
   * 
   * @see javax.jdo.InstanceCallbacks#jdoPreStore()
   */
  public void jdoPreStore() {}
}
