package helion.rozdzial6;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import javax.microedition.rms.RecordComparator;
import javax.microedition.rms.RecordEnumeration;
import javax.microedition.rms.RecordFilter;
import javax.microedition.rms.RecordListener;
import javax.microedition.rms.RecordStore;
import javax.microedition.rms.RecordStoreException;

// Klasa implementujaca magazyn ksiazek
// skatalogowanych wedlug numerow ISBN
public class BookStore implements RecordComparator, RecordFilter {

    // Nazwa magazynu rekordow, ktory ma przechowywac informacje o ksiazkach
    private static final String STORE_NAME = "BookStore";

    // Magazyn rekordow
    private RecordStore store;

    // Numer ISBN wykorzystywany w operacji filtrowania
    private String searchISBN;

    // Tworzy obiekt klasy BookStore i otwiera go
    public BookStore() {
        try {
            store = RecordStore.openRecordStore(STORE_NAME, true);
        } catch (RecordStoreException ex) {
            System.err.println(ex);
        }
    }

    // Zamyka obiekt klasy BookStore
    public void close() throws RecordStoreException {
        if (store != null) {
            store.closeRecordStore();
        }
    }

    // Pobiera liczbe ksiazke przechowywanych w obiekcie BookStore
    public int getBookCount() throws RecordStoreException {
        if (store != null) {
            return store.getNumRecords();
        }
        return 0;
    }

    // Dodaje RecordListener do obiektu klasy BookStore
    public void addRecordListener(RecordListener l) {
        if (store != null) {
            store.addRecordListener(l);
        }
    }

    // Usuwa obiekt RecordListener
    public void removeRecordListener(RecordListener l) {
        if (store != null) {
            store.removeRecordListener(l);
        }
    }

    // Pobiera posortowana liste wszystkich ksiazek z magazynu.
    public RecordEnumeration getBooks() throws RecordStoreException {
        if (store != null) {
            return store.enumerateRecords(null, this, false);
        }
        return null;
    }


    // Odczytuje BookInfo z rekordu
    // o podanym identyfikatorze
    public BookInfo getBookInfo(String isbn) throws RecordStoreException,
                                                    IOException {
        BookInfo bookInfo = null;
        searchISBN = isbn;

        // Znajdz ksiazke o podanym numerze ISBN
        RecordEnumeration enum = store.enumerateRecords(
                                        this, null, false);

        // Jezeli znalazles, sprawdz jej identyfikator i pobierz obiekt BookInfo
        if (enum.numRecords() > 0) {
            int id = enum.nextRecordId();
            bookInfo = getBookInfo(id);
        }

        // Zwolnienie wyliczenia
        enum.destroy();

        return bookInfo;
    }

    // Odczytuje BookInfo z rekordu
    // o podanym identyfikatorze
    public BookInfo getBookInfo(int id) throws RecordStoreException,
                                                    IOException {
        byte[] bytes = store.getRecord(id);
        DataInputStream is = new DataInputStream(
                            new ByteArrayInputStream(bytes));

        String isbn = is.readUTF();
        BookInfo info = new BookInfo(isbn);
        info.id = id;
        info.title = is.readUTF();
        info.ranking = is.readInt();
        info.reviews = is.readInt();
        info.lastRanking = is.readInt();
        info.lastReviews = is.readInt();

        return info;
    }

    // Dodaje wpis do magazynu lub modyfikuje istniejace dane,
    // jezeli dotycza one ksiazki o takim samym numerze ISBN.
    public void saveBookInfo(BookInfo bookInfo)
                                throws IOException, RecordStoreException {
        if (store != null) {
            searchISBN = bookInfo.getIsbn();
            RecordEnumeration enum = store.enumerateRecords(
                                        this, null, false);
            if (enum.numRecords() > 0) {
              // Znaleziono pasujacy rekord. Ustaw id
              // obiektu BookInfo tak, aby pasowal do istniejacego rekordu
                bookInfo.id = enum.nextRecordId();
                byte[] bytes = toByteArray(bookInfo);
                store.setRecord(bookInfo.id, bytes, 0, bytes.length);
            } else {
                // Utworz nowy rekord
                bookInfo.id = store.getNextRecordID();
                byte[] bytes = toByteArray(bookInfo);
                store.addRecord(bytes, 0, bytes.length);
            }

            // Na zakonczenie usun wyliczenie RecordEnumeration
            enum.destroy();
        }
    }

    // Usuwa opis danej ksiazki z magazynu rekordow
    public void deleteBook(BookInfo bookInfo) throws RecordStoreException {
        if (store != null) {
            store.deleteRecord(bookInfo.id);
        }
    }

    // Implementacja obiektu RecordComparator
    public int compare(byte[] book1, byte[] book2) {
        try {
            DataInputStream stream1 =
                new DataInputStream(new ByteArrayInputStream(book1));
            DataInputStream stream2 =
                new DataInputStream(new ByteArrayInputStream(book2));

            // Dopasowuj po numerach ISBN, ale sortuj po tytulach
            String isbn1 = stream1.readUTF();
            String isbn2 = stream2.readUTF();
            if (isbn1.equals(isbn2)) {
                return RecordComparator.EQUIVALENT;
            }
            String title1 = stream1.readUTF();
            String title2 = stream2.readUTF();
            int result = title1.compareTo(title2);
            if (result == 0) {
                return RecordComparator.EQUIVALENT;
            }
            return result < 0 ? RecordComparator.PRECEDES :
                                RecordComparator.FOLLOWS;
        } catch (IOException ex) {
            return RecordComparator.EQUIVALENT;
        }
    }

    // Implementacja obiektu RecordFilter
    public boolean matches(byte[] book) {
        if (searchISBN != null) {
            try {
                DataInputStream stream =
                    new DataInputStream(new ByteArrayInputStream(book));

                // Dopasowanie na podstawie numeru ISBN
                return searchISBN.equals(stream.readUTF());
            } catch (IOException ex) {
                System.err.println(ex);
            }
        }

        // Domyslnie przyjmujemy brak dopasowania
        return false;
    }

    // Zapisuje rekord w postaci tablicy bajtow.
    private byte[] toByteArray(BookInfo bookInfo) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        DataOutputStream os = new DataOutputStream(baos);

        os.writeUTF(bookInfo.isbn);
        os.writeUTF(bookInfo.title == null ? "" : bookInfo.title);
        os.writeInt(bookInfo.ranking);
        os.writeInt(bookInfo.reviews);
        os.writeInt(bookInfo.lastRanking);
        os.writeInt(bookInfo.lastReviews);

        return baos.toByteArray();
    }
}