package decoratorexample.controller;

import java.io.*;
import java.net.*;
import java.util.*;
import java.text.*;

import javax.servlet.*;
import javax.servlet.http.*;
import javax.servlet.*;

import javax.xml.parsers.*;
import javax.xml.transform.*;
import javax.xml.transform.dom.*;
import javax.xml.transform.stream.*;
import org.w3c.dom.*;


/**
 * Filtr wykonujcy operacje XSLT na wybranym pliku
 * przed umieszczeniem go na wyjciu.
 */

public class XSLTFilter implements Filter {
    
    /** Obiekt konfiguracji filtra */
    private FilterConfig filterConfig = null;
    
    /**
     * Konstruktor
     */
    public XSLTFilter() {
    }
 
     /**
     * Wywoywana przez kontener podczas inicjacji filtra
     */
    public void init(FilterConfig filterConfig) {
        this.filterConfig = filterConfig;
    }
    
    /**
     * Wykonuje waciwe filtrowanie.  Obudowuje odpowied obiektem XSLT.
     */
    public void doFilter(ServletRequest request, 
                         ServletResponse response,
                         FilterChain chain)
        throws IOException, ServletException 
    {    
        ServletContext sc = filterConfig.getServletContext();
     
        sc.log("DoFilter");
        
        
        
        // obudowuje odpowied
        XSLTResponseWrapper xsltResponse = 
            new XSLTResponseWrapper((HttpServletResponse)response, 
                                    "/SimpleTransform.xml");
        HttpServletRequest httpRequest = (HttpServletRequest) request;
        
        // przekazuje do nastpnego filtra
        chain.doFilter(httpRequest, xsltResponse);
        
        // zapisuje rzeczywist odpowied
        sc.log("zapis odpowiedzi");
        xsltResponse.writeResponse();
    }
    
    /**
     * Wywoywana przez kontener podczas usuwania filtra.
     */
    public void destroy() {
    } 
   
    /**
     * Klasa pomocnicza umoliwiajca adaptacj klasy ByteArrayOutputStream 
     * jako klasy pochodnej klasy ServletOutputStream
     */
    class CacheOutputStream extends ServletOutputStream {
        
        private ByteArrayOutputStream bos;
       
        public CacheOutputStream() {
            bos = new ByteArrayOutputStream();
        }
        
        /** 
         * Zapisuje pojedynczy bajt do strumienia wyjcia
         */
        public void write(int b) {
            bos.write(b);
        }
        
        /**
         * Zapisuje tablic bajtw do strumienia wyjcia
         */
        public void write(byte[] b, int offset, int len) {
            bos.write(b, offset, len);
        }
 
        /**
         * Dokonuje konwersji strumienia wyjcia na tablic bajtw
         */
        public byte[] toByteArray() {
            return bos.toByteArray();
        }
    }
    
    
    class XSLTResponseWrapper extends HttpServletResponseWrapper {
        /** oryginalna odpowied */
        private HttpServletResponse orig;
        
        private PrintWriter writer;
        private ServletOutputStream stream;
        private ServletOutputStream outStream;
        
        /** nazwa przeksztacenia XML */
        private String transform;
        
        /** czy docelowa strona jest w jzyku XML? */
        private boolean isXML = false;
        
        /** 
         * Konstruktor przechowuje oryginaln odpowied i przeksztacenie
         */
        public XSLTResponseWrapper(HttpServletResponse response, 
                                   String transform)
        {
            super(response);
            orig = response;
            this.transform = transform;
        }
      
        /**
         * Tworzy strumie wyjciowy.  Jeli odpowied jest zakodowana, to przechowuje
         * j w celu pniejszego przeksztacenia.  W przeciwnym razie uywa oryginalnego strumienia.
         */
        public ServletOutputStream createOutputStream() throws IOException {
            if (containsHeader("XML-Encoded")) {
                isXML = true;
                outStream = new CacheOutputStream();
                return outStream;
            } else {
                return orig.getOutputStream();
            }
        }
        
        /**
         * Zwraca strumie wyjcia.  Zgasza wyjtek, jeli wczeniej wywoana
         * zostaa metoda getWriter().
         */
        public ServletOutputStream getOutputStream() throws IOException {
            if (stream != null) {
                return stream;
            }
            
            if (writer != null) {
                throw new IOException("Wczeniej wywoano getWriter()");
            }
            
            stream = createOutputStream();
            return stream;
        }
        
        /** 
         * Zwraca obiekt zapisu.  Zgasza wyjtek, jeli wczeniej wywoana
         * zostaa metoda getOutputStream().
         */
        public PrintWriter getWriter() throws IOException {
            if (writer != null) {
                return writer;
            }
            
            if (stream != null) {
                throw new IOException("Wczeniej wywoano getOutputStream()");
            }
            
            writer = new PrintWriter(new OutputStreamWriter(createOutputStream()));
            return writer;
        }
        
        /** 
         * Wywoywana przez filtr w celu wykonania waciwego przeksztacenia XML.
         * Zwraca natychmiast sterowanie, jeli dane nie s w formacie XML.
         */
        public void writeResponse() throws IOException {
            if (!isXML) { 
                return;
            }
            
            ServletContext sc = filterConfig.getServletContext();
            
            if (outStream == null) {
                throw new IOException("Brak strumienia");
            }
            
            // odczyt danych z bufora
            byte[] data = ((CacheOutputStream) outStream).toByteArray();
            InputStream is = new ByteArrayInputStream(data);
            
            sc.log("Przeksztacenie");
            
            // waciwe przeksztacenie
            try {
                DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
                DocumentBuilder db = dbf.newDocumentBuilder();
            
                // wczytuje dane odpowiedzi z bufora
                Document doc = db.parse(is);
          
                // stosuje przeksztacenie
                TransformerFactory tf = TransformerFactory.newInstance();
                StreamSource stylesheet = 
                    new StreamSource(sc.getResourceAsStream(transform));
                Transformer transformer = tf.newTransformer(stylesheet); 
                DOMSource source = new DOMSource(doc);
                
                // wysya wynik do oryginalnego strumienia wyjcia
                StreamResult result = new StreamResult(orig.getOutputStream());
                transformer.transform(source, result);
            } catch( Exception ex ) {
                sc.log("Bd podczas przeksztacenia", ex);
            } 
        }
    }             
}



