<?php
//namespace net\xmp\phpbook;

class HTMLPage {

    protected $title;
    protected $content;
    protected $template;

    function __construct($title, $template='./template.html') {
        $this->title = $title;
        $this->content = '';
        $this->template = $template;
    }

    /**
     * Displays the created HTML page.
     */
    function printPage() {
        global $page;
        $page = $this;
        require $this->template;
    }

    /**
     * Returns single data fields of HTMLPage.
     *
     * @param string $name Name of the data field
     * @param boolean $escape Indicates if the value for HTML output is masked
     */
    function get($name, $escape=true) {
        if ($escape) {
            return $this->escape($this->$name);
        } else {
            return $this->$name;
        }
    }

    /**
     * Adds a HTML element with text to the page.
     *
     * @param string $name Name of the HTML element
     * @param string $content Content of the element
     * @param boolean $keepWhiteSpace Leave white space as is
     */
    function addElement($name, $content, $keepWhiteSpace=false) {
        $this->content .= "<$name>" . $this->escape($content, $keepWhiteSpace) . "</$name>";
    }

    /**
     * Adds unfiltered HTML to the page.
     *
     * @param string $html HTML to be added
     */
    function addHTML($html) {
        $this->content .= $html;
    }

    /**
     * Adds a table to the page.
     *
     * @param array $content Two-dimensional array:
     *                       First line contains headings  (<th>)
     *                       The following lines contain data (<td>)
     * @param array $rawHTML Indicates which columns are added as unfiltered HTML
     */
    function addTable($content, $rawHTML=array()) {
        $this->content .= "<table>\n";
        // Adds a header
        $this->content .= "<thead><tr>\n";
        foreach ($content[0] as $cell) {
            $this->addElement('th', $cell);
        }
        $this->content .= "</tr></thead>\n";
        // Adds data
        $this->content .= '<tbody>';
        for ($i = 1; $i < count($content); $i++) {
            $this->content .= '<tr>';
            for ($j = 0; $j < count($content[$i]); $j++) {
                if (empty($rawHTML[$j])) {
                    $this->addElement('td', $content[$i][$j]);
                } else {
                    $this->addHTML('<td>' . $content[$i][$j] . '</td>');
                }
            }
            $this->content .= "</tr>\n";
        }
        $this->content .= "</tbody></table>\n";
    }

    /**
     * Adds a list to the page.
     *
     * @param array $items List entries. If the entry is an array the first part is
     *                     highlighted and the second part is displayed normally
     * @param boolean $rawHTML Indicates if the entries are added unfiltered.
     */
    function addList($items, $rawHTML=false) {
        $this->content .= "<ul>\n";
        foreach ($items as $item) {
            if (!is_array($item)) {
                $rawHTML ? $this->addHTML('<li>' . $item . '</li>')
                         : $this->addElement('li', $item);
            } else {
                $this->content .= '<li>';
                $this->addElement('b', $item[0] . ': ');
                $this->content .= $rawHTML ? $item[1] : $this->escape($item[1]);
                $this->content .= '</li>';
            }
        }
        $this->content .= "</ul>\n";
    }

    /**
     * Masks the text displayed on the HTML page
     * Replaces critical characters with entities and
     * deletes all characters less than chr(32) depending on white space option
     *
     * @param string $txt Text
     * @param boolean $keepWhiteSpace Leave white space as is
     */
    function escape($txt, $keepWhiteSpace=false) {
        if ($keepWhiteSpace) {
            $txt = preg_replace('/[\\x00-\\x08\\x0b-\\x1f]/', '', $txt);
            return filter_var($txt, FILTER_SANITIZE_SPECIAL_CHARS);
        } else {
            $txt = preg_replace('/\s+/', ' ', $txt);
            return filter_var($txt, FILTER_SANITIZE_SPECIAL_CHARS, FILTER_FLAG_STRIP_LOW);
        }
    }
}
