package com.packtpub.hibernatesearch.startup;

import java.util.Arrays;
import java.util.HashSet;

import javax.servlet.ServletContextEvent;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;
import org.hibernate.search.FullTextSession;
import org.hibernate.search.Search;
import org.hibernate.service.ServiceRegistry;
import org.hibernate.service.ServiceRegistryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.packtpub.hibernatesearch.domain.App;
import com.packtpub.hibernatesearch.domain.CustomerReview;
import com.packtpub.hibernatesearch.domain.Device;

/**
 * Klasa pomocnicza dla instancji uruchamianej jako wze podrzdny w klastrze.  Wycznie wze nadrzdny moe bezporednio 
 * aktualizowa nadrzdne indeksy Lucene.  Wzy podrzdne mog aktualizowa indeksy wycznie przesyajc wiadomoci przez
 * JMS do wza nadrzdnego, gdzie s nastpnie przetwarzane.
 * 
 * Dziki implementacji interfejsu ServletContextListener, kontener servletw autoamtycznie wywoa metod "contextInitialized" podczas
 * startu.  Wewtrz metody utworzymy i zindeksujemy okoo poowy danych testowych (tak aby zademonstowa utworzenie drugiej poowy 
 * przez wze podrzdny). 
 * 
 * Poprzednie wersje tej klasy (nazywane "StartupDataLoader") miay adnotacj @WebListener. Niemniej jednak w tej wersji 
 * polegamy na pliku "web.xml" definiujcym waciwy listener... poniewa nie jest wskazane, eby aplikacja jednoczenie 
 * uruchomia MasterNodeInitializer *oraz* SlaveNodeInitializer.
 */
public class SlaveNodeInitializer implements javax.servlet.ServletContextListener {
	
	Logger logger = LoggerFactory.getLogger(SlaveNodeInitializer.class);
	
	/**
	 * Nie powinna by uywana bezporednio. Uyj openSession().
	 */
	private static SessionFactory sessionFactory;
	
	/**
	 * Konstruowanie nowej SessionFactory Hibernate dla kadego dania HTTP powodowaoby problemy wydajnociowe. Poniewa
	 * servlety Javy musz by wspbiene, nie moemy zadeklarowa by SessionFactory bya publiczn statyczn zmienn.
	 * Ta metoda udostpnia wspbieny dostp do SessionFactory, dziki czemu metoda zapeniajca baz danych przykadowymi danymi 
	 * i servlet wyszukujcy mog otwiera poczenia do bazy danych bardziej wydajnie. 
	 * 
	 * @return Session
	 */
	public static synchronized Session openSession() {
		if(sessionFactory == null) {
			Configuration configuration = new Configuration();
			configuration.configure("/hibernate-slave.cfg.xml");
			ServiceRegistry serviceRegistry = new ServiceRegistryBuilder().applySettings(configuration.getProperties()).buildServiceRegistry();
			sessionFactory = configuration.buildSessionFactory(serviceRegistry);
		}
		return sessionFactory.openSession();
	}
	
	public void contextInitialized(ServletContextEvent event) {
		
		event.getServletContext().setAttribute("mode", "slave");

		// W celach demonstracyjnych utworzomy poow danych podczas startu wza nadrzdnego... a drug poow podczas startu wza 
		// podrzdnego.  Po kilku sekundach oba wzy odwie lokalne kopie swoich indeksw uywajc nadrzdnego indeksu...  
		// i wszystkie encje App bd wyszukiwalne na wszystkich wzach.  
		FullTextSession fullTextSession = Search.getFullTextSession( openSession() );
		fullTextSession.beginTransaction();
		
		//
		// Pobierz referencje do 5 urzdze, ktre powinny by ju utworzone w bazie danych przez wze nadrzdny
		//
		Device xPhone = (Device) fullTextSession.createQuery( "from Device as device where device.name = ?" ).setString(0, "xPhone").uniqueResult();
		Device xTablet = (Device) fullTextSession.createQuery( "from Device as device where device.name = ?" ).setString(0, "xTablet").uniqueResult();
		Device solarSystem = (Device) fullTextSession.createQuery( "from Device as device where device.name = ?" ).setString(0, "Solar System Phone").uniqueResult();
		Device flame = (Device) fullTextSession.createQuery( "from Device as device where device.name = ?" ).setString(0, "Flame Book Reader").uniqueResult();
		Device pc = (Device) fullTextSession.createQuery( "from Device as device where device.name = ?" ).setString(0, "Pecet").uniqueResult();
		
		//
		// Utwrz i zapisz 6 z 12 aplikacji razem z ich urzdzeniami i komentarzami
		//		
		App frustratedFlamingos = new App(
				"Sfrustrowane Flamingi", 
				"flamingo.jpg", 
				"Maa aplikacja pozwalajca na ciskanie wielkimi ptakami na lewo i prawo bez powodu. Chyba nie zastanawiasz si czemu s sfrustrowane?",
				"Gry",
				0.99f);
		frustratedFlamingos.setSupportedDevices( new HashSet<Device>(Arrays.asList(new Device[] { xPhone, xTablet, solarSystem, flame, pc })) );
		CustomerReview frustratedFlamingosReview = new CustomerReview("Procarz", 4, "LOL, uwielbiam katapulotwa flamingi w krowy! Troch mnie wkurza, e reklama troch zasania obszar gry.");
		frustratedFlamingos.setCustomerReviews( new HashSet<CustomerReview>(Arrays.asList(new CustomerReview[] { frustratedFlamingosReview })) );
		fullTextSession.save(frustratedFlamingos);
		logger.info("Zapisuj " + frustratedFlamingos.getName());
		
		App grype = new App(
				"Wideokonferencje Grype", 
				"laptop.jpg", 
				"Dzwo za darmo lokalnie i za granic, rwnie z transmisj obrazu, uywajc domowego poczenia internetowego i naszego produktu. W sumie nasza aplikacja zadziaa jeszcze lepiej jeli skorzystasz z poczenia internetowego Twojego pracodawcy!",
				"Internet",
				3.99f);
		grype.setSupportedDevices( new HashSet<Device>(Arrays.asList(new Device[] { xPhone, xTablet, solarSystem, pc })) );
		CustomerReview grypeReview = new CustomerReview("biurowy.luzak", 4, "Szkoda, e dodali obsug wideo do najnowszej wersji. Wczeniej nie musiaem si ubiera.");
		grype.setCustomerReviews( new HashSet<CustomerReview>(Arrays.asList(new CustomerReview[] { grypeReview })) );
		fullTextSession.save(grype);
		logger.info("Zapisuj " + grype.getName());
		
		App eReader = new App(
				"Czytnik E-Bookw", 
				"book.jpg", 
				"Nasza aplikacja sprawi, e bdziesz mg czyta ksiki na komputerze albo na dowolnym urzdzeniu mobilnym.  Polecamy \"Hibernate Search by Example\".",
				"Media",
				1.99f);
		eReader.setSupportedDevices( new HashSet<Device>(Arrays.asList(new Device[] { xPhone, xTablet, solarSystem, flame, pc })) );
		CustomerReview eReaderReview = new CustomerReview("StevePerkins", 5, "Ksika 'Hibernate Search by Example' jest super!  Dziki za rekomendacje!");
		eReader.setCustomerReviews( new HashSet<CustomerReview>(Arrays.asList(new CustomerReview[] { eReaderReview })) );
		fullTextSession.save(eReader);
		logger.info("Zapisuj " + eReader.getName());
		
		App domeBrowser = new App(
				"Przegldarka pod kopu", 
				"orangeswirls.jpg", 
				"Ta cudowna aplikacja pozwala nam ledzi Twoj aktywno w Internecie. Moemy odgadn gdzie mieszkasz, co jade dzisiaj na niadanie, albo jakie s Twoje najwiksze sekrety. Aha, w aplikacj jest jeszcze wbudowana przegldarka stron WWW.",
				"Internet",
				0);
		domeBrowser.setSupportedDevices( new HashSet<Device>(Arrays.asList(new Device[] { solarSystem, flame, pc })) );
		CustomerReview domeBrowserReview = new CustomerReview("oni.wiedza", 1, "Odinstalowaem t aplikacj.  Jeeli udao im si sfaszowa ldowanie na ksiycu, z pewnoci wykorzystaj histori mojej przegldarki przeciwko mnie.");
		domeBrowser.setCustomerReviews( new HashSet<CustomerReview>(Arrays.asList(new CustomerReview[] { domeBrowserReview })) );
		fullTextSession.save(domeBrowser);
		logger.info("Zapisuj " + domeBrowser.getName());
		
		App athenaRadio = new App(
				"Internetowe radio Atena", 
				"jamming.jpg", 
				"Suchaj swoich ulubionych piosenek w radiu internetowym! Kiedy polubisz jaki utwr, nasza aplikacja zagra Ci wicej podobnych kompozycji. A przynajmniej zagra wicej piosenek ... eby by szczerym, czasem wcale nie s takie podobne  :(",
				"Media",
				3.99f);
		athenaRadio.setSupportedDevices( new HashSet<Device>(Arrays.asList(new Device[] { xPhone, xTablet, solarSystem, flame, pc })) );
		CustomerReview athenaRadioReview = new CustomerReview("lskinner", 5, "Chciaem'Free Bird', dostaem 'Free Bird'.  Czego chcie wicej?");
		athenaRadio.setCustomerReviews( new HashSet<CustomerReview>(Arrays.asList(new CustomerReview[] { athenaRadioReview })) );
		fullTextSession.save(athenaRadio);
		logger.info("Zapisuj " + athenaRadio.getName());
		
		App mapJourney = new App(
				"Mapa podry", 
				"compass.jpg", 
				"Potrzebujesz wskazwek, by dotrze do celu?  Nasza aplikacja GPS na pewno wygeneruje wystarczajco duo propozycji, by dotar do celu!  Prdzej czy pniej.",
				"Podre",
				0.99f);
		mapJourney.setSupportedDevices( new HashSet<Device>(Arrays.asList(new Device[] { xPhone, solarSystem, pc })) );
		CustomerReview mapJourneyReview = new CustomerReview("Zagubiony", 3, "Nic szczeglnego... ale cigle o niebo lepsze od map Orange.");
		mapJourney.setCustomerReviews( new HashSet<CustomerReview>(Arrays.asList(new CustomerReview[] { mapJourneyReview })) );
		fullTextSession.save(mapJourney);
		logger.info("Zapisuj " + mapJourney.getName());
		
		//
		// Zamknij i wyczy sesj Hibernate
		//
		fullTextSession.getTransaction().commit();
		fullTextSession.close();
		
	} 

	/**
	 * Ta metoda jest wywoywana gdy mechanizm servletw jest wyczany. Zamyka Hibernate SessionFactory jeeli jeszcze jest otwarta.
	 */
	public void contextDestroyed(ServletContextEvent event) {
		if(!sessionFactory.isClosed()) {
			sessionFactory.close();
		}
	}
	
}
