package com.packtpub.hibernatesearch.startup;

import java.util.Arrays;
import java.util.HashSet;

import javax.servlet.ServletContextEvent;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;
import org.hibernate.search.FullTextSession;
import org.hibernate.search.Search;
import org.hibernate.service.ServiceRegistry;
import org.hibernate.service.ServiceRegistryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.packtpub.hibernatesearch.domain.App;
import com.packtpub.hibernatesearch.domain.CustomerReview;
import com.packtpub.hibernatesearch.domain.Device;

/**
 * Klasa pomocnicza dla instancji uruchamianej jako wze nadrzdny w klastrze. Wycznie wze nadrzdny moe bezporednio 
 * aktualizowa nadrzdne indeksy Lucene.  Wzy podrzdne mog aktualizowa indeksy wycznie przesyajc wiadomoci przez
 * JMS do wza nadrzdnego, gdzie s nastpnie przetwarzane.
 * 
 * Dziki implementacji interfejsu ServletContextListener, kontener servletw autoamtycznie wywoa metod "contextInitialized" podczas
 * startu.  Wewtrz metody utworzymy i zindeksujemy okoo poowy danych testowych (tak aby zademonstowa utworzenie drugiej poowy 
 * przez wze podrzdny).  Klasa tworzy osobny wtek monitorujcy kolejk JMS i wykonujcy aktualizacje Lucene na podstawie 
 * wiadomoci przesyanych przez wzy podrzdne. 
 * 
 * Poprzednie wersje tej klasy (nazywane "StartupDataLoader") miay adnotacj @WebListener. Niemniej jednak w tej wersji 
 * polegamy na pliku "web.xml" definiujcym waciwy listener... poniewa nie jest wskazane, eby aplikacja jednoczenie 
 * uruchomia MasterNodeInitializer *oraz* SlaveNodeInitializer.
 */
public class MasterNodeInitializer implements javax.servlet.ServletContextListener {
	
	Logger logger = LoggerFactory.getLogger(MasterNodeInitializer.class);
	
	/**
	 * Nie powinna by uywana bezporednio. Uyj openSession().
	 */
	private static SessionFactory sessionFactory;
	
	/**
	 * Konstruowanie nowej SessionFactory Hibernate dla kadego dania HTTP powodowaoby problemy wydajnociowe. Poniewa
	 * servlety Javy musz by wspbiene, nie moemy zadeklarowa by SessionFactory bya publiczn statyczn zmienn.
	 * Ta metoda udostpnia wspbieny dostp do SessionFactory, dziki czemu metoda zapeniajca baz danych przykadowymi danymi 
	 * i servlet wyszukujcy mog otwiera poczenia do bazy danych bardziej wydajnie. 
	 * 
	 * @return Session
	 */
	public static synchronized Session openSession() {
		if(sessionFactory == null) {
			Configuration configuration = new Configuration();
			configuration.configure();
			ServiceRegistry serviceRegistry = new ServiceRegistryBuilder().applySettings(configuration.getProperties()).buildServiceRegistry();
			sessionFactory = configuration.buildSessionFactory(serviceRegistry);			
		}
		return sessionFactory.openSession();
	}
	
	public void contextInitialized(ServletContextEvent event) {
		
		event.getServletContext().setAttribute("mode", "master");

		// W celach demonstracyjnych utworzomy poow danych podczas startu wza nadrzdnego... a drug poow podczas startu wza 
		// podrzdnego.  Po kilku sekundach oba wzy odwie lokalne kopie swoich indeksw uywajc nadrzdnego indeksu...  
		// i wszystkie encje App bd wyszukiwalne na wszystkich wzach.  
		FullTextSession fullTextSession = Search.getFullTextSession( openSession() );
		fullTextSession.beginTransaction();
		
		//
		// Utwrz 5 urzdze
		//
		Device xPhone = new Device("Orange", "xPhone", null);
		Device xTablet = new Device("Orange", "xTablet", null);
		Device solarSystem = new Device("Song-Sung", "Solar System Phone", null);
		Device flame = new Device("Jungle", "Flame Book Reader", null);
		Device pc = new Device(null, "Pecet", null);
		
		//
		// Utwrz i zapisz 6 z 12 aplikacji razem z ich urzdzeniami i komentarzami
		//
		App theCloud = new App(
				"Chmura!", 
				"cloud.jpg", 
				"Aplikacja Chmura! jest miejscem, w ktrym dziej si cuda. Firmy dziaaj w chmurach. Programici nie potrzebuj administratorw, jedzenia i picia w zasadzie te nie. Moesz oglda telewizj na tablecie, siedza na kanapie; nie musisz w ogle patrze na telewizor! cignij aplikacj Chmura! z chmury i dowiadcz tej cudownej potgi!",
				"Biznes",
				7.99f);
		theCloud.setSupportedDevices( new HashSet<Device>(Arrays.asList(new Device[] { xPhone, xTablet })) );
		CustomerReview theCloudReview1 = new CustomerReview("fanboy1984", 5, "Ta aplikacja sprawia, e mj xPhone jest jeszcze bardziej stylowy i trendy!");
		CustomerReview theCloudReview2 = new CustomerReview("anty.hipster", 1, "Nie rozumiem o co chodzi z t 'Chmur'.  Dla mnie to raczej marketingowy bekot...");
		theCloud.setCustomerReviews( new HashSet<CustomerReview>(Arrays.asList(new CustomerReview[] { theCloudReview1, theCloudReview2 })) );
		fullTextSession.save(theCloud);
		logger.info("Zapisuj " + theCloud.getName());

		App salesCloser = new App(
				"Zamykacz transakcji", 
				"pointing.jpg", 
				"Wysokoenergetyczna aplikacja wspomagajca energicznych sprzedawcw. Twrz pene energii kalendarze i arkusze. Gdy jeste w miecie i energicznie nawizujesz kontakty, chcesz prezentowa swe naenergetyzowane foldery, pokazujce, e te masz w sobie mnstwo energii.",
				"Biznes",
				5.99f);
		salesCloser.setSupportedDevices( new HashSet<Device>(Arrays.asList(new Device[] { xPhone, solarSystem })) );
		CustomerReview salesCloserReview = new CustomerReview("GdzieJestKasa", 5, "wietna aplikacja!  Jeeli uywae kiedy 'Sales Commandera 2000', szybko poapiesz si w tym interfejsie.");
		salesCloser.setCustomerReviews( new HashSet<CustomerReview>(Arrays.asList(new CustomerReview[] { salesCloserReview })) );
		fullTextSession.save(salesCloser);
		logger.info("Zapisuj " + salesCloser.getName());

		App football = new App(
				"Midzynarodowy turniej pikarski", 
				"ball.jpg", 
				"Ta maa aplikacja oferuje Ci rado gry w pik non, poza tym, e grasz na ekranie dotykowym zamiast biega i kopa albo w ogle si rusza. Poza tym daje Ci podobne wraenia.",
				"Gry",
				1.99f);
		football.setSupportedDevices( new HashSet<Device>(Arrays.asList(new Device[] { xTablet, flame })) );
		CustomerReview footballReview = new CustomerReview("PrawdziwyAmerykanin", 2, "Podpucha... To nie jest amerykaski futbol!");
		football.setCustomerReviews( new HashSet<CustomerReview>(Arrays.asList(new CustomerReview[] { footballReview })) );
		fullTextSession.save(football);
		logger.info("Zapisuj " + football.getName());
		
		App crystal = new App(
				"Kolejna gra o krysztaach", 
				"brilliant.jpg", 
				"Olniewajca aplikacja, w ktrej czysz krysztay tego samego koloru, by znikny. Co jak Tetris. W sumie przypomina tuzin innych gier, w ktrych czysz krysztay tego samego koloru.",
				"Gry",
				0.99f);
		crystal.setSupportedDevices( new HashSet<Device>(Arrays.asList(new Device[] { flame, pc })) );
		CustomerReview crystalReview = new CustomerReview("KolejnyGracz", 3, "czemu ta gra jest dostpna tylko na dwch urzdzeniach?  Z tuzin klonw tej gry jest dostpny dla wszystkich urzdze.  Powinnicie deaktywowa t aplikacj do czasu, gdy wszystkie urzdzenia bd wspierane...");
		crystal.setCustomerReviews( new HashSet<CustomerReview>(Arrays.asList(new CustomerReview[] { crystalReview })) );
		crystal.setActive(false);
		fullTextSession.save(crystal);
		logger.info("Zapisuj " + crystal.getName());
		
		App pencilSharpener = new App(
				"Temperwka", 
				"pencil.jpg", 
				"Ostrz owki, wkadajc je do gniazda Bluetooth i wciskajc przycisk. Ta aplikacja wyciska ze sprztu wszystko!",
				"Biznes",
				2.99f);
		pencilSharpener.setSupportedDevices( new HashSet<Device>(Arrays.asList(new Device[] { xPhone, solarSystem })) );
		CustomerReview pencilSharpenerReview1 = new CustomerReview("brakuje.cyferek", 1, "Ta aplikacja jest niebezpieczna!  Myl o pozwie.");
		CustomerReview pencilSharpenerReview2 = new CustomerReview("Prawnik", 5, "@brakuje.cyferek  Napisz do mnie.  Pogadamy...");
		pencilSharpener.setCustomerReviews( new HashSet<CustomerReview>(Arrays.asList(new CustomerReview[] { pencilSharpenerReview1, pencilSharpenerReview2 })) );
		fullTextSession.save(pencilSharpener);
		logger.info("Zapisuj " + pencilSharpener.getName());
		
		App staplerTracker = new App(
				"Namierzacz zszywaczy", 
				"stapler.jpg", 
				"Czy kto cigle poycza sobie Twj zszywacz? To typowy problem w biurach. Nasza aplikacja biznesowa sprawi, e ju nigdy nie bdziesz musia si zastanawia, gdzie podziewa si Twj zszywacz.",
				"Biznes",
				0.99f);
		staplerTracker.setSupportedDevices( new HashSet<Device>(Arrays.asList(new Device[] { pc })) );
		CustomerReview staplerTrackerReview = new CustomerReview("mike.bolton", 3, "'PC LOAD LETTER'?  O co w ogle chodzi?!?");
		staplerTracker.setCustomerReviews( new HashSet<CustomerReview>(Arrays.asList(new CustomerReview[] { staplerTrackerReview })) );
		fullTextSession.save(staplerTracker);
		logger.info("Zapisuj " + staplerTracker.getName());
		
		//
		// Zamknij i wyczy sesj Hibernate
		//
		fullTextSession.getTransaction().commit();
		fullTextSession.close();
		
		//
		// Utwrz wtek by ledzi zawarto kolejki JMS  Klasa "QueueMonitor" jest odpowiedzialna za monitorowanie 
		// i odbieranie wiadomoci, podczas gdy klasa "QueueController" odpowiada za wykonywanie akutalizacji indeksw 
		// Lucene na podstawie danych w wiadomoci.
		//
		QueueController queueController = new QueueController(sessionFactory.openSession());
		Thread queueMonitor = new Thread(new QueueMonitor(queueController)); 
		queueMonitor.start();
			
	} 

	/**
	 * Ta metoda jest wywoywana gdy mechanizm servletw jest wyczany. Zamyka Hibernate SessionFactory jeeli jeszcze jest otwarta.
	 */
	public void contextDestroyed(ServletContextEvent event) {
		if(!sessionFactory.isClosed()) {
			sessionFactory.close();
		}
	}
	
}
