package org.hibernate.auction.persistence;

import net.sf.hibernate.*;
import org.hibernate.auction.model.*;
import org.hibernate.auction.user.UserSession;

import java.math.BigDecimal;
import java.sql.*;
import java.util.Currency;

/**
 * Wasne odwzorowanie typu dla typu wartoci MonetaryAmount.
 * <p>
 * Pamitaj, e ten typ odwzorowania dotyczy starszej bazy danych stosujcej tylko jedn
 * kolumn numeryczn do przechowywania kwot. Kady obiekt <tt>MonetaryAmount</tt>
 * zostanie skonwertowany na dolary (uywajc kodu konwertujcego)
 * i zapisany do bazy danych. Kada warto odczytana z bazy danych
 * zostanie skonwertowana na walut odpowiedni dla uytkownika (lokalnowtkowy obiekt UserSession).
 *
 * @author Christian Bauer <christian@hibernate.org>
 */
public class MonetaryAmountSimpleUserType
		implements UserType {

	private static final int[] SQL_TYPES = {Types.NUMERIC};

	public int[] sqlTypes() { return SQL_TYPES; }

	public Class returnedClass() { return MonetaryAmount.class; }

	public boolean isMutable() { return false; }

	public Object deepCopy(Object value) {
		return value; // MonetaryAmount jest niezmienna
	}

	public boolean equals(Object x, Object y) {
		if (x == y) return true;
		if (x == null || y == null) return false;
		return x.equals(y);
	}

	public Object nullSafeGet(ResultSet resultSet,
							  String[] names,
							  Object owner)
			throws HibernateException, SQLException {

		if (resultSet.wasNull()) return null;
		BigDecimal valueInUSD = resultSet.getBigDecimal(names[0]);
		Currency userCurrency =
						(Currency) UserSession.get("currency");
		return new MonetaryAmount(valueInUSD, userCurrency);
	}

	public void nullSafeSet(PreparedStatement statement,
							Object value,
							int index)
			throws HibernateException, SQLException {

		if (value == null) {
			statement.setNull(index, Types.NUMERIC);
		} else {
			MonetaryAmount anyCurrency = (MonetaryAmount)value;
			MonetaryAmount amountInUSD =
			  MonetaryAmount.convert( anyCurrency,
									  Currency.getInstance("USD") );
			statement.setBigDecimal(index, amountInUSD.getValue());
		}
	}
}
