package org.hibernate.auction.persistence;

import net.sf.hibernate.*;
import net.sf.hibernate.cfg.Configuration;
import org.apache.commons.logging.*;
import org.hibernate.auction.exceptions.InfrastructureException;

import javax.naming.*;

/**
 * Podstawowa klasa pomocnicza dla Hibernate, obsuguje klasy SessionFactory, Session i Transaction.
 * <p>
 * Uywa statycznego inicjalizatora dla klasy SessionFactory i przechowuje obiekty
 * Session i Transactions w zmiennych lokalnowtkowych. Wszystkie wyjtki zostay oroczone
 * nieweryfikowalnym wyjtkiem InfrastructureException.
 *
 * @author christian@hibernate.org
 */
public class HibernateUtil {

	private static Log log = LogFactory.getLog(HibernateUtil.class);

	private static Configuration configuration;
	private static SessionFactory sessionFactory;
	private static final ThreadLocal threadSession = new ThreadLocal();
	private static final ThreadLocal threadTransaction = new ThreadLocal();
	private static final ThreadLocal threadInterceptor = new ThreadLocal();

	// Utwrz pocztkowy SessionFactory z domylnych plikw konfiguracyjnych.
	static {
		try {
			configuration = new Configuration();
			sessionFactory = configuration.configure().buildSessionFactory();
			// Moemy te doczy si do JNDI:
			// configuration.configure().buildSessionFactory()
		} catch (Throwable ex) {
			// Musimy wyapywa Throwable, bo w przeciwnym razie nie zapalibymy
			// NoClassDefFoundError ai innych podklas Error.
			log.error("Nieudane utworzenie SessionFactory.", ex);
			throw new ExceptionInInitializerError(ex);
		}
	}

	/**
	 * Zwraca SessionFactory uywan w metodach statycznych.
	 *
	 * @return SessionFactory
	 */
	public static SessionFactory getSessionFactory() {
		/* Zamiast zmiennej statycznej uyj JNDI:
		SessionFactory sessions = null;
		try {
			Context ctx = new InitialContext();
			String jndiName = "java:hibernate/HibernateFactory";
			sessions = (SessionFactory)ctx.lookup(jndiName);
		} catch (NamingException ex) {
			throw new InfrastructureException(ex);
		}
		return sessions;
		*/
		return sessionFactory;
	}

	/**
	 * Zwraca oryginaln konfiguracj Hibernate.
	 *
	 * @return Configuration
	 */
	public static Configuration getConfiguration() {
		return configuration;
	}

	/**
	 * Ponownie tworzy SessionFactory na podstawie statycznej Configuration.
	 *
	 */
	 public static void rebuildSessionFactory()
		throws InfrastructureException {
		synchronized(sessionFactory) {
			try {
				sessionFactory = getConfiguration().buildSessionFactory();
			} catch (Exception ex) {
				throw new InfrastructureException(ex);
			}
		}
	 }

	/**
	 * Ponownie tworzy SessionFactory na podstawie wskazanej konfiguracji.
	 *
	 * @param cfg
	 */
	 public static void rebuildSessionFactory(Configuration cfg)
		throws InfrastructureException {
		synchronized(sessionFactory) {
			try {
				sessionFactory = cfg.buildSessionFactory();
				configuration = cfg;
			} catch (Exception ex) {
				throw new InfrastructureException(ex);
			}
		}
	 }

	/**
	 * Pobiera sesj lokalnowtkow..
	 * <p/>
	 * Jeli sesja nie istnieje, otwiera now.
	 *
	 * @return Session
	 */
	public static Session getSession()
		throws InfrastructureException {
		Session s = (Session) threadSession.get();
		try {
			if (s == null) {
				log.debug("Potwieram now sesj dla wtku.");
				if (getInterceptor() != null) {
					log.debug("Uywam przechwytywania: " + getInterceptor().getClass());
					s = getSessionFactory().openSession(getInterceptor());
				} else {
					s = getSessionFactory().openSession();
				}
				threadSession.set(s);
			}
		} catch (HibernateException ex) {
			throw new InfrastructureException(ex);
		}
		return s;
	}

	/**
	 * Zamyka sesj lokalnowtkow.
	 */
	public static void closeSession()
		throws InfrastructureException {
		try {
			Session s = (Session) threadSession.get();
			threadSession.set(null);
			if (s != null && s.isOpen()) {
				log.debug("Zamyka sesj wtku.");
				s.close();
			}
		} catch (HibernateException ex) {
			throw new InfrastructureException(ex);
		}
	}

	/**
	 * Uruchamia transakcj bazodanow.
	 */
	public static void beginTransaction()
		throws InfrastructureException {
		Transaction tx = (Transaction) threadTransaction.get();
		try {
			if (tx == null) {
				log.debug("Uruchamia transakcj bazodanow dla wtku.");
				tx = getSession().beginTransaction();
				threadTransaction.set(tx);
			}
		} catch (HibernateException ex) {
			throw new InfrastructureException(ex);
		}
	}

	/**
	 * Zatwierdza transakcj bazodanow.
	 */
	public static void commitTransaction()
		throws InfrastructureException {
		Transaction tx = (Transaction) threadTransaction.get();
		try {
			if ( tx != null && !tx.wasCommitted()
							&& !tx.wasRolledBack() ) {
				log.debug("Zatwierdza transakcj bazodanow dla wtku.");
				tx.commit();
			}
			threadTransaction.set(null);
		} catch (HibernateException ex) {
			rollbackTransaction();
			throw new InfrastructureException(ex);
		}
	}

	/**
	 * Wycofuje transakcj bazodanow.
	 */
	public static void rollbackTransaction()
		throws InfrastructureException {
		Transaction tx = (Transaction) threadTransaction.get();
		try {
			threadTransaction.set(null);
			if ( tx != null && !tx.wasCommitted() && !tx.wasRolledBack() ) {
				log.debug("Prbuj wycofa transakcj bazodanow dla wtku.");
				tx.rollback();
			}
		} catch (HibernateException ex) {
			throw new InfrastructureException(ex);
		} finally {
			closeSession();
		}
	}

	/**
	 * Ponownie docza sesj do aktualnego wtku.
	 *
	 * @param session Sesja do podczenia.
	 */
	public static void reconnect(Session session)
		throws InfrastructureException {
		try {
			session.reconnect();
			threadSession.set(session);
		} catch (HibernateException ex) {
			throw new InfrastructureException(ex);
		}
	}

	/**
	 * Odcza i zwraca sesj aktualnego wtku.
	 *
	 * @return Session the disconnected Session
	 */
	public static Session disconnectSession()
		throws InfrastructureException {

		Session session = getSession();
		try {
			threadSession.set(null);
			if (session.isConnected() && session.isOpen())
				session.disconnect();
		} catch (HibernateException ex) {
			throw new InfrastructureException(ex);
		}
		return session;
	}

	/**
	 * Rejestruje przechytywanie dla aktualnej sesji.
	 * <p>
	 * Kada nowa sesja zostaje otwarta z przechwytywaniem od momentu rejestracji.
	 * Nie ma wpywu na sesj otwart przed rejestracj.
	 */
	public static void registerInterceptor(Interceptor interceptor) {
		threadInterceptor.set(interceptor);
	}

	private static Interceptor getInterceptor() {
		Interceptor interceptor =
			(Interceptor) threadInterceptor.get();
		return interceptor;
	}

}

