local nmap = require "nmap"
local stdnse = require "stdnse"
local string = require "string"
local table = require "table"
local nsedebug = require "nsedebug"

_ENV = stdnse.module("dicom", stdnse.seeall)

local MIN_SIZE_ASSOC_REQ = 68 -- Minimalna wielkość zapytania ASSOCIATE
local MIN_SIZE_ASSOC_RESP = 8 -- Winimalna wielkość odpowiedzi
local MAX_SIZE_PDU = 128000 -- Maksymalna wielkość jednostki PDU
local MIN_HEADER_LEN = 6 -- Minimalna długość nagłówka DICOM
local PDU_NAMES = {}
local PDU_CODES = {}

-- Tablica zawierająca kody i nazwy jednostek PDU
PDU_CODES =
{ 
  ASSOCIATE_REQUEST  = 0x01,
  ASSOCIATE_ACCEPT   = 0x02,
  ASSOCIATE_REJECT   = 0x03,
  DATA               = 0x04,
  RELEASE_REQUEST    = 0x05,
  RELEASE_RESPONSE   = 0x06,
  ABORT              = 0x07
}
-- Tablica zawierająca nazwy identyfikatorów UID i ich wartości
UID_VALUES =
{
  VERIFICATION_SOP = "1.2.840.10008.1.1", -- Klasa weryfikacyjna SOP
  APPLICATION_CONTEXT = "1.2.840.10008.3.1.1.1", -- Nazwa kontekstu aplikacji DICOM
  IMPLICIT_VR = "1.2.840.10008.1.2", -- Niejawny kod little-endian: domyślna składnia transportowa
  FIND_QUERY = "1.2.840.10008.5.1.4.1.2.2.1" -- Model zapytania / odbierania informacji
}

-- Nazwy typów jednostek PDU będą przechowywane jako klucze
for i, v in pairs(PDU_CODES) do
  PDU_NAMES[v] = i
end

---
-- start_connection(host, port) Funkcja tworząca gniazdo do komunikacji z usługą DICOM.
--
-- @param host Adres hosta
-- @param port Numer portu
-- @return (status, socket) Jeżeli status ma wartość true, socket zawiera obiekt gniazda.
--                          W przeciwnym razie socket zawiera komunikat o błędzie.
---
function start_connection(host, port)
  local dcm = {}
  local status, err
  dcm['socket'] = nmap.new_socket()

  status, err = dcm['socket']:connect(host, port, "tcp")

  if(status == false) then
    return false, "DICOM: Błąd połączenia z usługą: " .. err
  end

  return true, dcm
end

-- send(dcm, data) Funkcja wysyłająca pakiet DICOM za pomocą utworzonego gniazda.
--
-- @param dcm Obiekt DICOM
-- @param data Dane do wysłania
-- @return status True, jeżeli dane zostały wysłane pomyślnie.
--                W przeciwnym razie funkcja zwraca wartość false i komunikat o błędzie.
function send(dcm, data)
  local status, err
  stdnse.debug2("DICOM: wysyłanie pakietu (liczba bajtów: %d)", #data)
  if dcm["socket"] ~= nil then
    status, err = dcm["socket"]:send(data)
    if status == false then
      return false, err
    end
  else
    return false, "Brak gniazda"
  end
  return true
end

-- receive(dcm) Funkcja odbierająca pakiet DICOM za pomocą utworzonego gniazda.
--
-- @param dcm DICOM object
-- @return (status, data) Jeżeli status ma wartość true, data zawiera dane.
--                        W przeciwnym razie zawiera komunikat o błędzie.
function receive(dcm)
  local status, data = dcm["socket"]:receive()
  if status == false then
    return false, data
  end
  stdnse.debug2("DICOM: odebranie pakietu (liczba bajtów: %d)", #data)
  return true, data
end

---
-- pdu_header_encode(pdu_type, length) Funkcja kodująca nagłówek pakietu DICOM.
--
-- @param pdu_type Typ PDU (liczba całkowita bez znaku)
-- @param length Długość komunikatu DICOM
-- @return (status, dcm) Jeżeli status ma wartość true, funkcja zwraca nagłówek.
--                       W przeciwnym razie zwraca komunikat o błędzie.
---
function pdu_header_encode(pdu_type, length)
  -- Kilka prostych testów poprawności. Nie są sprawdzane zakresy,
  -- aby można było tworzyć zniekształcone pakiety.
  if not(type(pdu_type)) == "number" then
    return false, "Typ PDU musi być liczbą całkowitą bez znaku z zakresu 0-7."
  end
  if not(type(length)) == "number" then
    return false, "Długość komunikatu musi być liczbą całkowitą bez znaku."
  end
  
  local header = string.pack("<B >B I4",
                            pdu_type, -- Typ PDU (1 bajt)
                            0,        -- Zarezerwowane (1 bajt zawsze równy 0x0)
                            length)   -- Długość komunikatu (4 bajty, liczba całkowita 
                                      -- bez znaku, little-endian)
  if #header < MIN_HEADER_LEN then
    return false, "Nagłówek musi się składać z co najmniej 6 bajtów."
  end
  return true, header
end 

---
-- associate(host, port) Funkcja wysyłająca żądanie A-ASSOCIATE
-- do dostawcy usługi DICOM w celu nawiązania połączenia.
--
-- @param host Adres hosta
-- @param port Numer portu
-- @return (status, dcm) Jeżeli status ma wartość true, funkcja zwraca obiekt DICOM.
--                       W przeciwnym razie zwraca komunikat o błędzie.
---
function associate(host, port, calling_aet_arg, called_aet_arg)
  local application_context = ""
  local presentation_context = ""
  local userinfo_context = ""

  local status, dcm = start_connection(host, port)
  if status == false then
    return false, dcm
  end

  application_context = string.pack(">B B I2 c" .. #UID_VALUES["APPLICATION_CONTEXT"],
                          0x10, -- Typ PDU (1 bajt)
                          0x0,  -- Zarezerwowane (1 bajt)
                          #UID_VALUES["APPLICATION_CONTEXT"], -- Długość kontekstu (2 bajty)
                          UID_VALUES["APPLICATION_CONTEXT"]) -- Kontekst aplikacji

  presentation_context = string.pack(">B B I2 B B B B B B I2 c" .. 
                                     #UID_VALUES["VERIFICATION_SOP"] ..
                                     "B B I2 c" .. #UID_VALUES["IMPLICIT_VR"],
                                     0x20, -- Typ kontekstu prezentacji (1 bajt)
                                     0x0,  -- Zarezerwowane (1 bajt)
                                     0x2e, -- Długość kontekstu (2 bajty)
                                     0x1,  -- Identyfikator kontekstu prezentacji (1 bajt)
                                     0x0,0x0,0x0,  -- Zarezerwowane (3 bajty)
                                     0x30, -- Składnia abstrakcyjna (1 bajt)
                                     0x0,  -- Zarezerwowane (1 bajt)
                                     0x11, -- Długość kontekstu (2 bajty)
                                     UID_VALUES["VERIFICATION_SOP"],
                                     0x40, -- Składnia transferowa (1 bajt)
                                     0x0,  -- Zarezerwowane (1 bajt)
                                     0x11, -- Długość kontekstu (2 bajty)
                                     UID_VALUES["IMPLICIT_VR"])

  local implementation_id = "1.2.276.0.7230010.3.0.3.6.2"
  local implementation_version = "OFFIS_DCMTK_362"
  userinfo_context = string.pack(">B B I2 B B I2 I4 B B I2 c" .. #implementation_id .. 
                                 " B B I2 c".. #implementation_version,
                                 0x50,    -- Typ kontekstu (1 bajt)
                                 0x0,     -- Zarezerwowane (1 bajt)
                                 0x3a,    -- Długość kontekstu (2 bajty)
                                 0x51,    -- Typ kontekstu (1 bajt) 
                                 0x0,     -- Zarezerwowane (1 bajt)
                                 0x04,    -- Długość kontekstu (2 bajty)
                                 0x4000,  -- Dane (4 bajty)
                                 0x52,    -- Typ kontekstu (1 bajt)
                                 0x0,     -- Zarezerwowane (1 bajt)
                                 0x1b,    -- Długość kontekstu (2 bajty)
                                 implementation_id, -- ID implementacji (dł. implementation_id)
                                 0x55,    -- Typ kontekstu (1 bajt)
                                 0x0,     -- Zarezerwowane (1 bajt)
                                 #implementation_version,  -- Długość kontekstu (2 bajty)
                                 implementation_version)

  local called_ae_title = called_aet_arg or stdnse.get_script_args("dicom.called_aet") or "ANY-SCP"
  local calling_ae_title = calling_aet_arg or stdnse.get_script_args("dicom.calling_aet") or "NMAP-DICOM"
  if #calling_ae_title > 16 or #called_ae_title > 16 then
    return false, "Tytuł jednostki aplikacji wywoływanej/wywołującej nie może przekraczać 16 znaków."
  end

  -- Wypełnienie niewykorzystanej części bufora spacjami
  called_ae_title = called_ae_title .. string.rep(" ", 16 - #called_ae_title)
  calling_ae_title = calling_ae_title .. string.rep(" ", 16 - #calling_ae_title)

  -- Żądanie A-ASSOCIATE
  local assoc_request = string.pack(">I2 I2 c16 c16 c32 c" .. application_context:len() .. " c" .. 
                                    presentation_context:len() .. " c" .. userinfo_context:len(),
                                    0x1, -- Wersja protokołu (2 bajty)
                                    0x0, -- Zarezerwowane (2 bajty równe 0x0)
                                    called_ae_title, -- Tytuł jedn. apl. wywoływanej (16 bajtów)
                                    calling_ae_title, -- Tytuł jedn. apl. wywołującej (16 bajtów)
                                    0x0, -- Zarezerwowane (32 bajty równe 0x0)
                                    application_context,
                                    presentation_context,
                                    userinfo_context)

  local status, header = pdu_header_encode(PDU_CODES["ASSOCIATE_REQUEST"], #assoc_request)
  -- Sprawdzenie, czy nagłówek został utworzony
  if status == false then
    return false, header
  end
  
  assoc_request = header .. assoc_request
  stdnse.debug2("Długość PDU minus nagłówek:%d", #assoc_request-#header)
  if #assoc_request < MIN_SIZE_ASSOC_REQ then
    return false, string.format("Żądanie A-ASSOCIATE musi się składać z min. %d bajtów. Aktualna wielkość: %d bajtów.", MIN_SIZE_ASSOC_REQ, #assoc_request)
  end

  status, err = send(dcm, assoc_request)
  if status == false then
    return false, string.format("Błąd wysłania żądania A-ASSOCIATE: %s", err)
  end
  status, err = receive(dcm)
  if status == false then
    return false, string.format("Błąd odczytu odpowiedzi A-ASSOCIATE: %s", err)
  end
  if #err < MIN_SIZE_ASSOC_RESP then
    return false, "Zbyt krótka odpowiedź A-ASSOCIATE."
  end

  local resp_type, _, resp_length, resp_version = string.unpack(">B B I4 I2", err)
  stdnse.debug1("Typ PDU: %d, długość: %d, protokół: %d", resp_type, resp_length, resp_version)

  if resp_type == PDU_CODES["ASSOCIATE_ACCEPT"] then
    stdnse.debug1("Komunikat ASSOCIATE ACCEPT")
    return true, dcm
  elseif resp_type == PDU_CODES["ASSOCIATE_REJECT"] then
    stdnse.debug1("Komunikat ASSOCIATE REJECT")
    return false, "Żądanie A-ASSOCIATE odrzucone"
  else
    return false, "Nieznana odpowiedź:" .. resp_type
  end
end -- Koniec funkcji

return _ENV