/*
Copyright  (c)  2000  by  Network ICE Corporation.   Wszystkie  prawa
zastrzezone.  Kod  zrodlowy  niniejszego  programu  jest udostepniany
publicznie  tylko w celach dyskusji.  Jednakze, kod ten nie nalezy do
public domain (or open-source or GPL).  However,  we plan to GPL this
code in the future.

Przyznawana licencja uprawnia do uzywania  niniejszego programu tylko
w przypadkach, gdy pojedynczy uzytkownik monitoruje  swoj wlasny ruch
sieciowy.  Innego  rodzaju  uzycie  wiaze sie z koniecznoscia  zakupu
licencji  na niniejszy  program,  co mozna osiagnac  piszac pod adres
<sales@altivore.com>. Wykorzystanie niniejszego oprogramowania w celu 
monitorowania   danych   nalezacych  do  innych  osob  moze  stanowic
naruszenie obowiazujacego prawa.


ALTIVORE v0.9.3

  Jest to przykladowy  program zawierajacy pewne  funkcje obslugiwane
  przez opracowany przez FBI program  "Carnivore".  Ma on sluzyc jako
  punkt w dyskusji na temat mechanizow programu Carnivore. Nie zostal
  on doglebnie przetestowany i zawiera zapewne wiele bledow.

  Moze sluzyc takze jako "alternatywa" dla dostawcow uslug interneto-
  wych,  ktorzy  nie  zycz  sobie  instalowania  "czarnych skrzynek" 
  programu  FBI. Nakazy sadowe  wymagajace od dostawcy  udostepnienia
  danych nie okreslaja koniecznosci wykorzystania programu Carnivore,
  jesli dostawca jest w stanie dostarczyc dane w inny sposob.

  Program ten moze takze byc przydatny w kwestiach zarzadzania siecia,
  na  przyklad archiwizacji danych lub monitorowania polaczen telefo-
  micznych konsumentow, ktorzy zglaszaja problemy w dziale serwisowym.

SPOSOB WYKORZYSTANIA NINIEJSZEGO PROGRAMU

  Program  niniejszy  musi  zosta  skompilowany  i  skonsolidowany z
  biblioteka libpcap.
  Podobnie biblioteka libpcap musi zostac zainstalowana w  systemie w
  celu poprawnego dzialania.

  Program  niniejszy  byl  kompilowany i poddawany pobieznym testom w
  systemie Windows oraz Linux. Powinien dzialac poprawnie w przypadku
  wikszosci innych systemow po wprowadzeniu drobnych zmian.

  Kompilacja w systemie Windows

  Nalezy pobrac srodowisko programistyczne WINPCAP z pod adresu:
  http://netgroup-serv.polito.it/winpcap/
  Jako include directory wskaz katalog "WPdpack/include" oraz  dolacz
  bilioteki "libpcap.lib", "user32.lib" oraz "wsock32.lib".

  Kompilacja w systemie Linux

  gcc altivore.c -lpcap -Ipcap -o altivore

  Uwaga: libpcap nie dziala poprawnie w systemie RedHat 6.2

RODZAJ PRZECHWYTYWANYCH DANYCH

  Modul niniejszy  stanowi  probe odpowiedzi na prosbe wyrazona przez
  FBI  o  niezalezna   ocene  techniczna  programu  Carnivore,  ktora
  opublikowano 25 sierpnia 2000 r. Zalacznik 1 tego dokumentu opisuje
  kilka scenariuszy wykorzystania Carnivore.

  W  calym dokumencie wyrazenie "Alice" odnosi sie  do  podejrzanego,
  ktorego   korespondencja   elektroniczna   podlega   monitorowaniu.
  Wyrazenie "Bob" odnosi sie do osoby, z ktora komunikuje sie  Alice.
  Ponizszy fragment mozna skopiowac i wkleic do pliku "altivore.ini",
  z ktorego  niniejszy program  korzysta w celu przechowywania danych
  konfiguracyjnych.

  [1.1 email header pen-register]
    ;Nakazuje monitorowanie wszelkich naglowkow wiadomosci
    ;przesylanych do oraz od uzytkownika Alice. Nie podlega
    ;przechwytywaniu pole "Subject:" ("Temat:"), ktore jest 
    ;prawnie postrzegane jako czesc "danych", nie zas 
    ;"call records". Dzialania powinny sie odbywac z poziomu 
    ;serwera pocztowego, przetwarzajacego komunikacje
    ;pocztowa uzytkownika Alice lub w jego poblizu.
    mode = email-headers
    email.address = alice@example.com
    logfile = alice.txt

  [1.2 HTTP pen-register from dial-up user]
    ;Nakazuje monitorowanie adresow IP stron odwiedzanych 
    ;przez uzytkownika. Pewnym utrudnieniem jest fakt, ze
    ;uzytkownik laczy sie przez linie telefoniczna i za 
    ;kazdym razem jest mu przydzielany inny adres IP. 
    ;Monitorowaniu podlega protokol hasel laczy komutowanych
    ;znany jako "RADIUS", co pozwala rozpoczac dzialania 
    ;w momencie zalogowania sie uzytkownika Alice oraz 
    ;znalezienie wykorzystywanego adresu IP. Dzialania powinny 
    ;odbywac sie z poziomu segmentu sieci znajdujacego sie za
    ;zspolem serwerow dial-up oraz w takim miejscu, aby mozliwe
    ;bylo monitorowanie pakietow protokolu RADIUS. Niniejsza
    ;wersja Carnivore pozwala monitorowac jedynie pakiety 
    ;rozliczeniowe (Accounting packets); moze okazac sie, ze
    ;funkcja ta musi zostac uaktywniona, aby program dzialal 
    ;poprawnie.
    mode = server-access
    radius.account = alice@example.com
    server.port = 80
    logfile = alice.csv

  [1.3 FTP pen-register from dial-up user]
    ;Jak powyzej, ale monitorowaniu podlega
    ;ruch FTP zamiast HTTP.
    mode = server-access
    radius.account = alice@example.com
    server.port = 80
    logfile = alice.csv

  [2.1 email content-wiretap]
    ;Zamiast przechwytywania samych naglowkow, w scenariuszu
    ;tym przechwytywana jest cala zawartosc wiadomosci.
    mode = email-content
    email.address = alice@example.com
    tracefile = alice.tcp
    
  [2.2 email content-wiretap]
    ;Przechwytywana jest cala zawartosc wiadomosci adresowanych
    ;do/z okreslonego adresu IP. Odpowiada to dzialaniu darmowego 
    ;produktu o nazwie TCPDUMP. Przyklad: 
    ;    tcpdump -w tracefile.tcp host 192.0.2.189
    mode = ip-content
    ip.address = 192.0.2.189
    tracefile = alice.tcp
    

PROJEKT

  Brak obslugi ponownego skladania/przestawiania
    Program  niniejszy nie  obsluguje defragmentacji pakietow  IP ani
    przestawiania  segmentow  TCP.   W  rezultacie  moze  sie zdarzyc
    pominiecie   pewnych   wiadomosci  lub   przypadkowe   dolaczenie
    segmentow  nalezacych  do  korespondencji  innych osob.  Jest  to
    wazna  kwestia;  problemy  z  fragemntacja stanowia  istotna wade
    wielu  produktow  i prawdopodobnie  jest tak rowniez  w przypadku
    programu Carnivore.
    
  Niewielka obsluga stanowosci serwera SMTP
    Altivore w niewielkim tylko zakresie monitoruje stan serwera SMTP
    (niemozliwa  jest pelna  obsluga stanowosci SMTP  bez rozkladania
    i ponownego  skladania  fragmentow pakietow).  W rezultacie, moze
    okazac sie, ze przechwycono wiadomosc email nie nalezaca so Alice
    (osoby podejrzanej). Przykladowo,  jesli system nie jest w stanie
    okreslic,  gdzie konczy  si  wiadomosc  email,  moze przypadkowo
    przechwycic  kolejne  wiadomosci  przesylane w ramach tego samego 
    polaczenia SMTP.  Najprawdopodobniej  stanowi  to rowniez problem 
    w przypadku programu FBI Carnivore.

  Niekompletna obsluga protokolu RADIUS
    Kod odpowiedzialny za analize protokolu RADIUS byl testowany tylko
    w przypadku kilku dostawcow uslug internetowych. Ma to znaczenie w
    w pewnych sytuacjach,  w ktorych kod nie bedzie dzialal. Jednym ze
    sposobow  ominiecia tego problemu  jest wymuszenie przy  wdrazaniu
    zastosowanie  opcji  Accounting  (rozliczanie)  protokolu  RADIUS.
    W przypadku Altivore  wymagane jest podjecie  pewnych  dodatkowych
    dzialan zwiazanych z dekodowaniem protokolu RADIUS.

  Uwierzytelnianie materialu dowodowego
    Zarzadzanie  materialem  dowodowym  to  powazna  sprawa.  Programy
    Altivore i Carnivore powinny obslugiwac standard MD5, PGP lub X.509
    podpisywania  za  pomoca  kluczy  prywatnych  w  celu calosciowego
    uwierzytelniania plikow. Pozwalaloby to na wykrywanie pozniejszych
    prob dostepu do materialu dowodowego przez osoby niepowolane.

ALTIVORE A NETWORK ICE

  Network  ICE  to  wiodacy  dostawca   oprogramowania  zwiazanego  z
  produktami wykorzystujacymi techologie "sniffingu" ruchu sieciowego
  w poszukiwaniu oznak aktywnosci hakerow w celu zabezpieczenia sieci
  naszych klientow.  Nasza  glowna  zaleta  sa implementowane funkcje
  dekodowania stanu protokolow oraz wydajny sniffing.  Oznacza to, ze
  mozemy monitorowac gigabitowe sieci przy pelnym rozkladzie pakietow
  oraz analizie stanu protokolu aplikacji.

  Z kolei program Carnivore napisano prawdopodobnie przy uzyciu wielu
  technik  stosowanych przez  naszych konkurentow.  Altivore napisano
  takze  przy  uzyciu  tych  technik,   aby  zademonstrowac  problemy
  dotyczace takiego podejscia.  Obsluga stanowosci jest minimalna, co
  ma  na  celu  pokazanie,  dlaczego w  przypadku tej klasy produktow
  wymagane jest badanie biorace pod uwage kwestie stanowosci.

*/
#include <string.h>
#include <malloc.h>
#include <stdlib.h>
#include <ctype.h>
#include <stdarg.h>
#include <stdio.h>
#include <sys/stat.h>
#include <errno.h>

/* Odwolanie do biblioteki libpcap, standardowej biblioteki systemow
 * Windows oraz UNIX sluzacej do sniffingu.*/
#include <pcap.h>


#define HASH_ENTRIES        1024
#define ADD_IF_NOT_FOUND    1
#define IGNORE_IF_NOT_FOUND 0
#define TCP_TO_SERVER       0x100
#define TCP_FROM_SERVER     0x000

/** Maksymalna dlugosc adresu poczty elektronicznej. Czesc adresu
 * przekraczajaca te dlugosc jest ignorowana. */
#define MAX_ADDRESS_LENGTH 1024

/** Maksymalna liczba odbiorcow. Kolejni odbiorcy przekraczajacy te 
 * dlugosc sa ignorowani. */
#define MAX_RECIPIENTS 100

#undef TRUE
#define TRUE 1
#undef FALSE
#define FALSE 0

/** Czytelne wyswietlanie adresow IP */
#define _XIP(a,n) (int)(((a)>>(n))&0xFF)
#define P_IP_ADDR(a) _XIP(a,24), _XIP(a,16), _XIP(a,8), _XIP(a,0)

/**
 * Elementy zwiazane z ekstrakcja danych protokou TCP/IP.
 */
#define ex8(p,f)            ((p)[f]) 
#define ex16(p,f)            ((p)[f] << 8 | (p)[f+1])
#define ex32(p,f)            ((ex16(p,f)<<16) | ex16(p,f+2))
#define IP_VERSION(p,f)            ((ex8(p,f+0) >> 4) & 0x0F)
#define IP_SIZEOF_HDR(p,f)        ((ex8(p,f+0) & 0x0F) * 4)
#define IP_TOTALLENGTH(p,f)        ex16(p,f+2)
#define IP_PROTOCOL(p,f)        ex8(p,f+9)
#define IP_SRC(p,f)                ex32(p,f+12)                
#define IP_DST(p,f)                ex32(p,f+16)
#define TCP_SRC(p,f)            ex16(p,f+0)
#define TCP_DST(p,f)            ex16(p,f+2)
#define TCP_SEQNO(p,f)            ex32(p,f+4)
#define TCP_ACKNO(p,f)            ex32(p,f+8)
#define TCP_FLAGS(p,f)            (ex8(p,f+13)&0x3F)
#define TCP_SIZEOF_HDR(p,f)        (((ex8(p,f+12)>>4) & 0x0f)*4)
#define TCP_FIN                    1
#define TCP_SYN                    2
#define TCP_RST                    4

#define FREE(x) if (x) free(x)

/**
 * Funkcja uzytkowa sluzaca przypisywaniu ciagow znakow. Rozwiazuje
 * kilka problemow zwiazanych z obsluga ciagow, takich jak kopiowanie
 * "ciagow o okreslonej dlugosci" zamiast ciagow zakonczonych znakiem NUL.
 */
static void 
setString(char **r_str, const void *vstr, int offset, int len)
{
    const char *str = vstr; /*kludge: unikanie ostrzezen*/
    if (*r_str)
        free(*r_str);
    if (str == NULL) {
        *r_str = NULL;
        return;
    }
    if (len == -1)
        len = strlen((const char*)str);
    *r_str = (char*)malloc(len+1);
    memcpy(*r_str, str+offset, len);
    (*r_str)[len] = '\0';
}

/** Wersja memcmp() niezalezna od wielkosci liter */
static int 
memcasecmp(const void *lhs, const void *rhs, int length)
{
    int i;
    for (i=0; i<length; i++) {
        if (tolower(((char*)lhs)[i]) != tolower(((char*)rhs)[i]))
            return -1;
    }
    return 0;
}

/** Narzedzie sluzace do porownan niezaleznych od wielkosci liter*/
static int 
startsWith(const char lhs[], const char rhs[])
{
    int len = strlen(lhs);
    if ((int)strlen(rhs) < len)
        len = strlen(rhs);
    return memcmp(lhs, rhs, len) == 0;
}


/** 
 * Implementacja idei tablicy ciagow znakow zakonczonych znakiem
 * nul. Nalezy uzywac funkcji straXXXX().
 */
struct stringarray {
    char **str;
    int length;
    int max;
};
typedef struct stringarray stringarray;


/** stringarray.straAddElement()
 * Dodaje ciag na koniec tablicy ciagow.
 */
void 
straAddElement(stringarray *lhs, const char rhs[])
{
    if (lhs->length + 1 >= lhs->max) {
        int new_max = lhs->max * 2 + 1;
        char **new_array = (char**)malloc(sizeof(char*)*(new_max));
        if (lhs->str) {
            memcpy(    new_array, 
                    lhs->str, 
                    sizeof(new_array[0])*lhs->length);
            free(lhs->str);
        }
        lhs->str = new_array;
        lhs->max = new_max;
    }
    lhs->str[lhs->length] = strdup(rhs);
    lhs->length++;
}


/** 
 * Ponizej zdefiniowano tryby, w ktorych moze pracowac program
 * Carnivore/Altivore. Patrz informacje podane wyzej w celu skonfigurowania
 * tych trybw.
 */
enum {
    /** Przechwytuje naglowki wiadomosci email do pliku tekstowego */
    mode_email_headers = 1,

    /** Przechwytuje jedynie adresy do/z uzytkownika Alice */
    mode_email_addresses,

    /** Rejestruje proby dostepu do serwerow na okreslonym porcie TCP. */
    mode_server_access,

    /** Rejestruje pelna zawartosc wiadomosci email Alice*/
    mode_email_content,

    /** Rejestruje pelne dane sledzace dla wskazanego adresu IP. */
    mode_ip_content
};
#define MODE(carn, m) ((carn)->mode == m)
static const char *modeNames[] = {"unspecified", "email-headers",
        "email-addresses", "server-access", "email-content",
        "ip-content", 0};

int 
parseMode(const char modeName[])
{
    int i;

    for (i=0; modeNames[i]; i++) {
        if (strcmp(modeName, modeNames[i]) == 0)
            return i;
    }
    return 0;
}

struct intlist {
    int list[32];
    int count;
};
typedef struct intlist intlist;

/**
 * Glowny objekt systemu Carnivore.
 */
struct Carnivore
{
    /** Tryb, w ktorym dziala program */
    int mode;

    /** Nazwa kompatybilnego pliku sledzenia sniffera, do ktorego
     * beda kopiowane dane (w przypadku pelnego podsluchu komunikacji).*/
    char *tracefile;
    FILE *fp_trace;
    
    /** Plik dziennika dla informacji tekstowych. */
    char *logfile;

    /** Lista filtrowanych adresow IP. Jest uzywana w przypadku, gdy 
     * nakaz sadowy okresla konkretne adresy IP. DO ZROBIENIA: umozliwienie
     * wykorzystania zakresow oraz wiekszej liczby adresow IP.*/
    intlist ip;

    /** Zawiera liste portow, ktore beda wykorzystywane w celu
     * monitorowania, kiedy nastepuje dostep do serwera okreslonego typu.
     */
    intlist port;

    /** Tabela polaczen TCP/IP sluzaca do zachowywania stanu sesji*/
    struct TcpCnxn *cxTable[HASH_ENTRIES];
    int cxId;

    /** Czy powinnismy zapisac ostatnia ramke do pliku */
    int do_filter;

    /** Czy powinnismy usunac dane polaczenie z naszej listy. */
    int do_remove;

    /** List adresow email. Adresy te sa porownywane z adresami
     * wiadomosci email, ktore sie pojawiaja w celu okreslenia, czy
     * nalezy utworzyc kopie. */
    stringarray email_addresses;

    /** Lista nazw kont systemu RADIUS, ktore nalezy monitorowac w czasie
     * podsluchu IP. */
    stringarray radius_accounts;

    /** Tablica plikow sledzenia, ktore beda odczytywane w celu
     * przetestowania systemu. Musza one byc zapisane w formacie
     * tcpdump/libpcap. */
    stringarray testinput;

    /** Tablica nazw kontrolerow, ktore nalezy otworzyc w trybie
     * odbierania (promiscuous mode). */
    stringarray interfaces;
};
typedef struct Carnivore Carnivore;

/** 
 * Test sluzacy sprawdzeniu, czy filtrowaniu podlega adres IP zrodlowy
 * lub docelowy IP. Jesli filtrowany jest ruch zwiazany wlasnie z tym
 * adresem IP, wowczas bedzie on najprawdopodobniej zapisany do pliku.
 * Trzeba pamietac, ze wykonywane jest liniowe przeszukiwanie tablicy
 * przy zalozeniu, ze filtrowanych jest tylko kilka adresow IP,
 * czesto nawet tylko jeden.
 */
int 
has_integer(intlist *ip, int ip1, int ip2)
{
    int i;
    for (i=0; i<ip->count; i++) {
        if (ip->list[i] == ip1 || ip->list[i] == ip2)
            return 1;
    }
    return 0;
}

/** Dodaje okreslony adres IP do listy adresow, ktore maja podlegac
 * filtrowaniu. Moze to byc adres IP podany w konfiguracji lub taki, 
 * ktory podlega automatycznej konfiguracji przez analizie protokolu
 * RADIUS. */
void 
add_integer(intlist *ip, int ip_address)
{
    if (ip_address == 0)
        return; /*ignore empty IP addresses*/
    if (has_integer(ip, ip_address, ip_address))
        return; /*ignore duplicates*/
    if (ip->count < sizeof(ip->list)/sizeof(int)) {
        ip->list[ip->count] = ip_address;
        ip->count++;
    }
}

/** Usuwa adres IP z listy filtrowania. Funkcja ta jest wywolywana
 * kiedy analiza protokolu RADIUS pozwala stwierdzic, ze monitorowany
 * uzytkownik sie rozlaczyl. */
void 
del_integer(intlist *ip, int ip_address) 
{
    int i;
    for (i=0; i<ip->count; i++) {
        if (ip->list[i] == ip_address) {
            memmove(ip->list+i, ip->list+i+1, 
                            (ip->count - i - 1)*sizeof(int));
            ip->count--;
        }
    }
}



/** matchName()
 * Sprawdza, czy okreslony adres email powinien podlagac filtrowaniu.
 * Jest to prawdopodobnie adres email kogos, kto zgodnie z nakazem
 * sadowym ma byc monitorowany.
 */
int 
matchName(const char addr[], int addr_len, stringarray *list)
{
    int i;
    if (addr == NULL)
        return 0;
    for (i=0; i<list->length; i++) {
        int lhs_len = strlen(list->str[i]);
        if (list->str[i][0] == '*') {
            /*match end of string, e.g. allow specification of
             * "*@suspect.com" to match any emails for a domain*/
            if (addr_len >= lhs_len - 1) {
                const char *new_lhs = list->str[i]+1;
                const char *new_addr = addr+addr_len-lhs_len+1;
                if (memcasecmp(new_lhs, new_addr, lhs_len-1) == 0)
                    return TRUE;
            }
        }
        else if (addr_len == lhs_len 
                && memcasecmp(list->str[i], addr, addr_len) == 0)
            return TRUE;
    }
    return FALSE;
}


/**
 * Wpis okreslajacy polaczenie TCP. Taki wpis istnieje dla kazdego
 * polaczenia, ktore podlega sledzeniu. Zawiera on podstawowe dane
 * dotyczace protokolu TCP info, a takze pewne dane dotyczace
 * nadrzednego protokolu SMTP.
 */
struct TcpCnxn
{
    /** Kazde nowe polaczenie jest identyfikowane przez unikatowy ID */
    int msg_id;
    int server_ip;
    int client_ip;
    int server_port;
    int client_port;
    int server_seqno;
    int client_seqno;
    struct TcpCnxn *next;
    time_t creation_time;
    char *sender;
    int sender_matches;
    char *recipient;
    stringarray recipients;
    
    /** Czy nalezy zapisac wiadomosc elektroniczna zwiazna z tym
     * polaczeniem. */
    int do_filter;

    /** Czy nalezy filtrowac dana ramke. Jest to potrzebne w celu
     * przechwycenia koncowej kropki umieszczonej w wiadomosci email. 
     */
    int filter_one_frame;

    /** Czy nalezy usunac dany wpis o polaczeniu przy najblizszej
     * okazji. */
    int do_remove;

    /** Czy analizowana jest 'koperta', czy tez sama
     * wiadomosc. */
    int state;
};
typedef struct TcpCnxn TcpCnxn;


/**
 * Tworzy dla tabeli wartosc mieszajaca (hash entry). Jest ona
 * oparta jedynie na sdresach IP oraz numberach portow. Konkretny
 * algorytm mieszajacy nie jest wazny, ale powinien dostosowany
 * do wymagan czasowych i dawac jak najlepsze rezultaty. Warto
 * zauwazyc, ze skoro dokonano juz konwersji z postaci (src,dst)
 * na postac (srvr,clnt), to funkcja mieszajaca nie musi byc 
 * symetryczna.
 */
int 
cxHash(TcpCnxn *cx)
{
    int result = 0;
    result = abs((cx->server_ip ^ (cx->client_ip*2)) 
                ^ ((cx->server_port<<16) | cx->client_port));
    return result % HASH_ENTRIES;
}


/**
 * Porownuje dwa obiekty polaczenia w celu sprawdzenia, czy s one
 * takie same. W procesie porownywania wykorzystywany jest tylko 
 * adres IP oraz port TCP.
 */
int 
cxEquals(TcpCnxn *lhs, TcpCnxn *rhs)
{
    if (lhs->server_ip != rhs->server_ip)
        return 0;
    if (lhs->client_ip != rhs->client_ip)
        return 0;
    if (lhs->server_port != rhs->server_port)
        return 0;
    if (lhs->client_port != rhs->client_port)
        return 0;
    return 1;
}


/**
 * Wyszukuje w tabeli obiekt polaczenia TCP. Jesli nie zostaje znaleziony,
 * mozna go dodac (w zaleznosci od parametru).
 * @param carn
 *        Dany obiekt.
 * @param rhs
 *        Kopia obiektu polaczenia, ktory jest wyszukiwany (pobierane sa
          z nich po prostu adresy i porty w celu ich porownania).
 * @param add_if_not_found
 *        Czy nalezy dodac nowy obiekt polaczenia, jesli nie mozna odnalezc
 *        go wsrod istniejacych. Wazna sparwa jest dodawanie obiektow
 *        polaczen tylko w czasie SYN/SYN-ACK w celu unikniecia przypadkowego
 *        zmienienia stanu w trakcie polaczenia.
 */
TcpCnxn *
cxLookup(Carnivore *carn, TcpCnxn *rhs, int add_if_not_found)
{
    int h = cxHash(rhs);
    TcpCnxn **r_cx = &carn->cxTable[h];
    
    for (;;) {
        if (*r_cx == NULL) {
            /* Obiekt polaczenia nie zostal znaleziony. Jesli byl
             * to SYN lub SYN-ACK, wowczas nalezy go dodac. */
            if (add_if_not_found) {
                *r_cx = (TcpCnxn*)malloc(sizeof(TcpCnxn));
                memset(*r_cx, 0, sizeof(**r_cx));
                (*r_cx)->server_ip = rhs->server_ip;
                (*r_cx)->client_ip = rhs->client_ip;
                (*r_cx)->server_port = rhs->server_port;
                (*r_cx)->client_port = rhs->client_port;
                (*r_cx)->server_seqno = rhs->server_seqno;
                (*r_cx)->client_seqno = rhs->client_seqno;
                (*r_cx)->creation_time = time(0);
            }
            return *r_cx;
        }

        if (cxEquals(*r_cx, rhs))
            return *r_cx;
        else
            r_cx = &(*r_cx)->next;
    }
}


/**
 * Przywraca dane dotyczace protokolu SMTP do znanego stanu. Wazne
 * jest, aby zrobic to w jak najdelikatniejszy sposob: trzeba ustawic
 * dane tak, aby uniknac przypadkowego przechwycenia poczty innej osoby.
 */
void 
cxResetMsg(TcpCnxn *cx)
{
    cx->do_filter = FALSE; /*nie pobieraj tych wiadomosci*/
    if (cx->sender) {
        free(cx->sender);
        cx->sender = NULL;
    }
    cx->sender_matches = FALSE;
    if (cx->recipients.length) {
        int i;
        for (i=0; i<cx->recipients.length; i++)
            free(cx->recipients.str[i]);
        free(cx->recipients.str);
        cx->recipients.str = NULL;
        memset(&cx->recipients, 0, sizeof(cx->recipients));
    }
}


/**
 * Usuwa obiekt polaczenia TCP z tabeli. Funkcja jest wywolywana
 * zawsze, gdy osigany jest koniec procesu przetwarzania protokolu
 * SMTP, gdy przerywane jest polaczenie TCP lub osiagniety zostaje
 * limit czasu polaczenia i jest ono usuwane.
 */
void 
cxRemove(Carnivore *carn, TcpCnxn *rhs)
{
    int h = cxHash(rhs);
    TcpCnxn **r_cx = &carn->cxTable[h];
    for (;;) {
        if (*r_cx == NULL)
            break; /*nie znaleziono*/
        else if (cxEquals(*r_cx, rhs)) {
            TcpCnxn *cx = *r_cx;
            *r_cx = cx->next;
            cxResetMsg(cx);
            free(cx);
            break;
        }
        else
            r_cx = &(*r_cx)->next;
    }
}


/** Zapisuje w formacie najpierw mlodszy wartosc integer do bufora */
void 
writeint(unsigned char hdr[], int offset, int x)
{
    hdr[offset+0] = (unsigned char)(x>>0);
    hdr[offset+1] = (unsigned char)(x>>8);
    hdr[offset+2] = (unsigned char)(x>>16);
    hdr[offset+3] = (unsigned char)(x>>24);
}


/**
 * Zapisuje biezacy pakiet do pliku kompatybilnego z TCPDUMP.
 * Warto zauwazyc, ze mozna by uzyc tu wewnetrznego mechanizmu
 * zapisywania biblioteki libpcap, jednak w przyszlosci chcemy 
 * miec mozliwosc stosowania podpisow elektronicznych, wiec
 * na pliku beda wykonywane dziwne dzialania.
 */
void 
carnSavePacket(Carnivore *carn, const unsigned char buf[], 
        int orig_len, time_t timestamp, int usecs)
{
    unsigned char hdr[16];
    int snap_len = orig_len;

    /* We were triggered to save the frame, now turn this off.
     * The SMTP state engine will have to revalidate the next
     * packet in order to make sure we should be saving it. */
    carn->do_filter = FALSE;

    /* Wyjdz z funkcji (bez zapisywania zawartosci), jesli nie
     * dzialamy w odpowiednim trybie.*/
    switch (carn->mode) {
    case mode_email_content:
    case mode_ip_content:
        break;
    default:
        return;
    }
    if (carn->tracefile == NULL)
        return; /*no filename*/

    /*W razie potrzeby otworz plik sledzenia*/
    if (carn->fp_trace == NULL) {
        struct stat s = {0};
        if (stat(carn->tracefile, &s) == 0)    {
            /*Ooops, it already exists. Maybe we crashed before? 
             * We should not put the header on the file if it 
             * already exists */
            carn->fp_trace = fopen(carn->tracefile, "a+b");
        } else {
            /*Create a new one.*/
            carn->fp_trace = fopen(carn->tracefile, "wb");
            if (carn->fp_trace) {
                /*create a file header*/
                static const char *foo = 
                        "\xD4\xC3\xB2\xA1" /*MAGIC*/
                        "\x02\x00\x04\x00" /*major/minor version*/
                        "\x00\x00\x00\x00" /*this timezone (GMT)*/
                        "\x00\x00\x00\x00" /*sig figs */
                        "\xDC\x05\x00\x00" /*snap length*/
                        "\x01\x00\x00\x00"; /*link type*/
                if (fwrite(foo, 1, 24, carn->fp_trace) != 24) {
                    int xxx = errno;
                    fclose(carn->fp_trace);
                    carn->fp_trace = NULL;
                    errno = xxx;
                }
            }
        }
        if (carn->fp_trace == NULL) {
            perror(carn->tracefile);
            return;
        }
    }

    /* Zapisanie ramki do pliku */
    writeint(hdr, 0, ((int)timestamp));
    writeint(hdr, 4, usecs);        /*mikrosekondy*/
    writeint(hdr, 8, snap_len);     /*odczytany rozmiar ramki*/
    writeint(hdr, 12, orig_len);    /*oryginalny rozmiar ramki*/

    fwrite(hdr, 1, 16, carn->fp_trace);
    fwrite(buf, 1, snap_len, carn->fp_trace);
}


/**
 * Zapisuje pewien tekst do pliku dziennika (logfile).
 */
void 
logprint(Carnivore *carn, const char fmt[], ...)
{
    FILE *fp;
    struct stat s = {0};
    va_list marker;

    if (carn->logfile == NULL)
        return;
    if (stat(carn->logfile,&s) == 0)
        fp = fopen(carn->logfile, "a");
    else
        fp = fopen(carn->logfile, "w");
    if (fp == NULL) {
        perror(carn->logfile);
        return;
    }

    va_start(marker, fmt);
    vfprintf(fp, fmt, marker);
    va_end(marker);

    fclose(fp);
}

/**
 * For logging purposes, we frequently need to grab the current
 * time. This function formats the current GMT time in ISO
 * format. BUG: the time should really be retrieved from the 
 * packet, not the system time (in case we read from tracefiles
 * rather the live network).
 */
void 
formatNow(char tbuf[], int sizeof_tbuf)
{
    time_t now = time(0);
    struct tm *tmptr = gmtime(&now); /*must be GMT*/
    if (tmptr == NULL)
        strcpy(tbuf, "err");
    else
        strftime(tbuf, sizeof_tbuf, "%Y-%m-%d %H:%M:%S", tmptr);
}


/**
 * Funkcja przechwytujaca jedynie adresy email.
 */
void 
carnPenEmail(Carnivore *carn, const char sender[],
            const unsigned char rcpt[], int offset, int length)
{
    char tbuf[64];

    if (!MODE(carn, mode_email_addresses))
        return; /*not recording email addresses*/

    if (carn->logfile == NULL)
        return; /*no logfile specified by user*/

    if (sender == NULL)
        sender = "(nul)";
    
    /*format czasu: np. 2000-08-24 08:23:59*/
    formatNow(tbuf, sizeof(tbuf));
    logprint(carn, "%s, %s, %.*s\n", tbuf, sender,
                                        length, rcpt+offset);
    printf("%s, %s, %.*s\n", tbuf, sender,
                                        length, rcpt+offset);
}




enum {
    parsing_envelope, parsing_message
};


/**
 * Sprawdzenie, czy dane pakietu TCP rozpoczynaja sie od okreslonego
 * polecenia.
 */
int 
SMTP_COMMAND(const unsigned char buf[], int offset, 
                                int max_offset, const char cmd[])
{
    int cmd_length = strlen(cmd);
    int line_length = max_offset-offset;

    if (line_length < cmd_length)
        return FALSE;
    if (memcasecmp(buf+offset, cmd, cmd_length) != 0)
        return FALSE;
    offset += cmd_length;

    /*DO ZROBIENIA: sprawdzenie pewnych warunkow brzegowych*/
    return TRUE;
}

/**
 * Sprawdzenie, czy tresc wiadomosci zawiera znak kropki '.' jako jedyny
 * w pustej poza tym linii.
 */
int 
SMTP_IS_DOT(const unsigned char buf[], int offset, int max_offset)
{
    int i;
    char last_char = '\0';

    for (i=offset; i<max_offset; i++) {
        char c = buf[i];
        if (c == '.') {
            if (i+1 < max_offset 
                && (buf[i+1] == '\n' || buf[i+1] == '\r')
                && (last_char == '\n' || last_char == '\r')) {
                return TRUE;
            }
        }
        last_char = c;
    }
    return FALSE;
}


static const char *MAIL_FROM = "MAIL FROM:";
static const char *RCPT_TO = "RCPT TO:";


/**
 * Przetwarza adres email w polu RCPT TO: albo MAIL FROM:
 */
void 
match_email(Carnivore *carn, const char *cmd,
        const unsigned char buf[], int offset, int max_offset, 
                                                TcpCnxn *cx)
{
    int length = -1;
    int address_matched = FALSE;


    /* See if this starts with RCPT TO: or MAIL FROM:, and then
     * skip beyond it. */
    if (!SMTP_COMMAND(buf, offset, max_offset, cmd))
        return;
    offset += strlen(cmd);

    /* Skip beyond leading whitespace and the initial '<' character
     * (if they exist). */
    while (offset < max_offset 
        && (isspace(buf[offset]) || buf[offset] == '<'))
        offset++;

    /* Figure out how long the email address is */
    for (length=0; offset+length<max_offset; length++) {
        char c = (char)buf[offset+length];
        if (c == '\r' || c == '\n' || c == '>')
            break;
    }
    if (length < 0)
        return;

    if (MODE(carn, mode_email_addresses) && cmd == MAIL_FROM ) {
        /* If we are doing a pen-register style capturing of email
         * addresses, then save off the SOURCE email address. */
        if (cx->sender)
            free(cx->sender);
        cx->sender = (char*)malloc(length+1);
        memcpy(cx->sender, buf+offset, length);
        cx->sender[length] = '\0';
    }

    /* See if the email addresses match */
    if (matchName((char*)buf+offset, length, 
                                        &carn->email_addresses)) {
        cx->do_filter = TRUE;
        address_matched = TRUE;
    }

    if (cmd == MAIL_FROM) {
        if (address_matched)
            cx->sender_matches = TRUE;
    } else if (cmd == RCPT_TO) {
        if (address_matched || cx->sender_matches)
            carnPenEmail(carn, cx->sender, buf, offset, length);
    }
}


/**
 * Odczyt liczby pozostalych znakow w wierszu.
 */
int 
readLine(const unsigned char buf[], int offset, int max_offset)
{
    int length = 0;
    while (offset + length < max_offset) {
        char c = buf[offset+length];
        length++;
        if (c == '\n')
            break;
    }
    return length;
}

/**
 * Examine the line from the packet in order to determine whether
 * it constitutes a legal RFC822 email header. We stop processing
 * data at the end of the headers.
 */
int 
isEmailHeader(const unsigned char buf[], int offset, int max_offset)
{
    int leading_space = 0;
    int saw_colon = 0;

    while (offset < max_offset && isspace(buf[offset])) {
        offset++; /*strip leading whitespace*/
        leading_space++;
    }

    if (offset >= max_offset)
        return FALSE; /*wiersze puste nie stanowia naglowka*/

    if (buf[offset] == '>')
        return FALSE;

    while (offset < max_offset) {
        if (buf[offset] == ':')
            saw_colon = TRUE;
        offset++;
    }

    if (saw_colon)
        return TRUE;
    if (leading_space)
        return TRUE;
    return FALSE;
}


/**
 * Funkcja przetwarzajaca pojedynczy segment TCP przeslany przez klienta
 * do serwera SMTP.
 */
int 
sniffSmtp(Carnivore *carn, TcpCnxn *rhs, int tcp_flags, 
              const unsigned char buf[], int offset, int max_offset)
{
    TcpCnxn *cx;
    int length;

    /* Lookup the TCP connection record to see if we are saving 
     * packets on the indicated TCP connection. */
    cx = cxLookup(carn, rhs, IGNORE_IF_NOT_FOUND);

    /* Przetworzenie danych z danego segemntu TCP */
    length = max_offset - offset;
    if (length > 0) {

        if (cx == NULL) {
            /* Add a record for this connection whenever we see a
             * an address in an envelope. */
            if (SMTP_COMMAND(buf, offset, max_offset, "RCPT TO:"))
                cx = cxLookup(carn, rhs, ADD_IF_NOT_FOUND);
            if (SMTP_COMMAND(buf, offset, max_offset, "MAIL FROM:"))
                cx = cxLookup(carn, rhs, ADD_IF_NOT_FOUND);
        } 
        
        if (cx != NULL) {
            switch (cx->state) {
            case parsing_envelope:
                match_email(carn, MAIL_FROM,
                                    buf, offset, max_offset, cx);

                match_email(carn, RCPT_TO,
                                    buf, offset, max_offset, cx);

                if (SMTP_COMMAND(buf, offset, max_offset, "DATA")) {
                    if (cx->do_filter)
                        cx->state = parsing_message;
                    else
                        cx->do_remove = TRUE;
                }
                if (SMTP_COMMAND(buf, offset, max_offset, "QUIT"))
                    cx->do_remove = TRUE;
                if (SMTP_COMMAND(buf, offset, max_offset, "RSET"))
                    cx->do_remove = TRUE;
                if (SMTP_COMMAND(buf, offset, max_offset, "ERST"))
                    cx->do_remove = TRUE;
                break;
            case parsing_message:
                if (MODE(carn, mode_email_headers)) {
                    int i;
                    char tbuf[64];
                    formatNow(tbuf, sizeof(tbuf));
                    logprint(carn, "--- %08X->%08X %s ---\n", 
                        cx->client_ip, cx->server_ip, tbuf);

                    /*Parse just the headers from the first packet*/
                    for (i=offset; i<max_offset; i++) {
                        int len;
                        len = readLine(buf, offset, max_offset);
                        if (!isEmailHeader(buf, offset, offset+len))
                            break;
                        if (len > 8 && 
                          startsWith((char*)buf+offset, "Subject:"))
                            logprint(carn, "Subject: <removed>\n");
                        else {
                            /*Write line to log file*/
                            logprint(carn, "%.*s", len, buf+offset);
                        }
                        offset += len;
                    }
                    logprint(carn,"---EOM---\n");
                    cx->do_remove = TRUE;
                    cx->do_filter = FALSE;
                    carn->do_filter = FALSE;
                }
                if (SMTP_IS_DOT(buf, offset, max_offset))
                    cx->do_remove = TRUE;
                break;
            }
        }
    }

    if (cx) {
    
        if (cx->do_filter)
            carn->do_filter = TRUE;

        if (cx->filter_one_frame) {
            carn->do_filter = TRUE;
            cx->filter_one_frame = FALSE;
        }

        if (cx->do_remove 
            || (tcp_flags & TCP_RST) || (tcp_flags & TCP_FIN))
            cxRemove(carn, rhs);
    }

    return 0;
}


/**
 * RADIUS protocol information we parse out of a packet. In the
 * future versions of this software, we are going to need to
 * store these records over time; for now, we just parse the
 * protocol into this normalized structure.
 */
struct RadiusRecord
{
    int radius_client;
    int radius_server;
    int nas_ip;
    int nas_port;
    int direction;
    int code;
    int xid;
    int status;
    char *user_name;
    char *caller_id;
    char *called_phone;
    char *session_id;
    int ip_address;
    int session_duration;
};
typedef struct RadiusRecord RadiusRecord;

/** Zwolnienie zalokowanej dla danych pamieci */
void 
radFree(RadiusRecord *rad)
{
    FREE(rad->user_name);
    FREE(rad->caller_id);
    FREE(rad->called_phone);
    FREE(rad->session_id);
}


/**
 * Process a single RADIUS command that we saw on the network.
 * For right now, we are primarily going to process radius
 * accounting packets, as these are the ones most likely to give
 * us solid information.
 */
void 
radProcess(Carnivore *carn, RadiusRecord *rad)
{
    enum {account_start=1, account_stop=2};
    if (rad->code == 4 || rad->code == 5) {
        /* ACCOUNTING packet: This packet contains an accounting
         * record. Accounting records will often contains IP address
         * assignments that normal authentication packets won't.*/
        if (rad->user_name && matchName(rad->user_name, 
                strlen(rad->user_name), &carn->radius_accounts)) {
            /* Found Alice! Therefore, we going add add Alice's
             * IP address to the list of IP addresses currently
             * being filtered. Conversely, if this is a stop
             * packet, then we will delete the IP address from
             * our list. */
            if (rad->status == account_start)
                add_integer(&carn->ip, rad->ip_address);
            else {
                /* Default: any unknown accounting message should
                 * trigger us to stop capturing data. If we make a
                 * mistake, we should err on the side of not
                 * collecting data. */
                del_integer(&carn->ip, rad->ip_address);
            }
            carn->do_filter = TRUE; /*capture this packet*/
        }

        /* Double-check: Look to see if the IP address belongs to
         * another person.*/
        else if (has_integer(&carn->ip, rad->ip_address, 0)) {
            /* The names did not match, yet we have seen some sort 
             * of packet dealing with the account that we are 
             * monitoring. This is bad -- it indicates that we might
             * have dropped a packet somewhere. Therefore, we are
             * going to immediately drop this packet.*/
            del_integer(&carn->ip, rad->ip_address);
            carn->do_filter = TRUE; /*capture this packet*/
        }
    }

}


/**
 * This function sniffs RADIUS packets off the network, then passes
 * the processed RADIUS information to another function that
 * deals with the content.
 */
int 
sniffRadius(Carnivore *carn, int ip_src, int ip_dst,
              const unsigned char buf[], int offset, int max_offset)
{
    RadiusRecord recx = {0};
    RadiusRecord *rad = &recx;
    const static int minimum_length = 20;
    int code;
    int xid;
    int radius_length;
    int i;

    if (carn->radius_accounts.length == 0)
        return 0; /*not scanning radius*/

    if (max_offset - offset <= minimum_length)
        return 0; /*corrupt*/

    /* Parse the RADIUS header info and verify */
    code = ex8(buf, offset+0);
    if (code < 1 || code > 5)
        return 0; /*unknown command/operationg*/
    xid = ex8(buf, offset+1);
    radius_length = ex16(buf, offset+2);
    if (offset + radius_length > max_offset)
        return 0; /*packet corrupt*/
    else if (offset + radius_length < minimum_length)
        return 0; /*packet corrupt*/
    else if (max_offset > offset + radius_length)
        max_offset = offset + radius_length; /*ignore padding*/

    /* Verify the attributes field */
    for (i=offset+minimum_length; i<max_offset-2; /*nul*/) {
        /*int type = buf[i];*/
        int len = buf[i+1];
        if (i+len > max_offset)
            return 0;
        i += len;
    }


    /* Grab the IP addresses of the client (the Network Access
     * Server like Livingston) and the RADIUS authentication
     * server. */
    if (code == 1 || code == 4) {
        rad->radius_client = ip_src;
        rad->radius_server = ip_dst;
    } else {
        rad->radius_client = ip_dst;
        rad->radius_server = ip_src;
    }
    rad->code = code;
    rad->xid = xid;

    /* Parse the attributes field */
    for (i=offset+minimum_length; i<max_offset-2; ) {
        int type = buf[i];
        int len = buf[i+1];
        int data_offset = i+2;
        if (i+len > max_offset)
            break;
        i += len;
        len -= 2;

        switch (type) {
        case 1: /*User-Name*/
            /*Lots of names appear to have a trailing nul that we
             *should strip from the end of the name.*/
            if (len > 1 && buf[data_offset+len-1] == '\0')
                len--;
            setString(&rad->user_name, buf, data_offset, len);
            break;
        case 2: /*User-Password*/
            break;
        case 4: /*NAS-IP-Address*/
            rad->nas_ip = ex32(buf,data_offset);
            break;
        case 5: /*NAS-Port*/
            rad->nas_port = ex32(buf,data_offset);
            break;
        case 8: /*Framed-IP-Address*/
            rad->ip_address = ex32(buf,data_offset);
            break;
        case 19: /*Callback-Number*/
        case 20: /*Callback-Id*/
            /*TODO: sounds like something we might want to record*/
            break;
        case 30: /*Called-Station-Id*/
            /*Find out the phone number of the NAS. This could be
             *important in cases where the evidence will later be
             *correlated with phone records.*/
            setString(&rad->called_phone, buf, data_offset, len);
            break;
        case 31: /*Calling-Station-Id*/
            /*True "trap-and-trace"! Assuming that caller-id is 
             *enabled, this will reveal the phone number of the 
             *person dialing in.*/
            setString(&rad->caller_id, buf, data_offset, len);
            break;
        case 40: /*Acct-Status-Type*/
            /*When scanning accounting packets, this is critical in
             *order to be able to detect when the service starts and
             *stops.*/
            rad->status = ex32(buf,data_offset);
            if (rad->status < 1 || 8 < rad->status)
                rad->status = 2; /*STOP if unknown*/
            break;
        case 44: /*Acct-Session-Id*/
            setString(&rad->session_id, buf, data_offset, len);
            break;
        case 46: /*Acct-Session-Time*/
            /*Could be interesting information to collect*/
            if (len == 4)
                rad->session_duration = ex32(buf,data_offset);
            break;
        }
    }

    /* The data was parsed from the RADIUS packet, now process that
     * data in order to trigger on the suspect.*/
    radProcess(carn, rad);
    radFree(rad);
    return 0;
}

struct iphdr {
    int offset;
    int proto;
    int src;
    int dst;
    int data_offset;
    int max_offset;
};
struct tcphdr {
    int offset;
    int src;
    int dst;
    int seqno;
    int ackno;
    int flags;
    int data_offset;
};


/**
 * This packet is called for each packet received from the wire
 * (or from test input). This function will parse the packet into
 * the consituent IP and TCP headers, then find which stream the
 * packet belongs to, then parse the remaining data according
 * to that stream.
 */
int 
sniffPacket(Carnivore *carn, const unsigned char buf[], 
                int max_offset, time_t timestamp, int usecs)
{
    struct iphdr ip;
    struct tcphdr tcp;
    TcpCnxn cn;
    
    /* Make sure that we have a frame long enough to hold the
     * Ethernet(14), IP(20), and UDP(8) or TCP(20) headers */
    if (max_offset < 14 + 20 + 20)
        return 1; /* packet fragment too small */

    if (ex16(buf,12) != 0x0800)
        return 1; /*not IP ethertype */

    /*IP*/
    ip.offset = 14; /*sizeof ethernet_header*/
    if (IP_VERSION(buf,ip.offset) != 4)
        return 1;
    ip.proto = IP_PROTOCOL(buf,ip.offset);
    ip.src = IP_SRC(buf,ip.offset);
    ip.dst = IP_DST(buf,ip.offset);
    ip.data_offset = ip.offset + IP_SIZEOF_HDR(buf,ip.offset);
    if (max_offset > IP_TOTALLENGTH(buf,ip.offset) + ip.offset)
        ip.max_offset = IP_TOTALLENGTH(buf,ip.offset) + ip.offset;
    else
        ip.max_offset = max_offset;

    /* If sniffing somebody's IP address, then sift for it */
    if (MODE(carn, mode_ip_content)
                        && has_integer(&carn->ip, ip.src, ip.dst))
        carn->do_filter = TRUE;

    if (ip.proto == 6) {
        /*TCP*/
        tcp.offset = ip.data_offset;
        tcp.dst = TCP_DST(buf,tcp.offset);
        tcp.src = TCP_SRC(buf,tcp.offset);
        tcp.flags = TCP_FLAGS(buf,tcp.offset);
        tcp.seqno = TCP_SEQNO(buf,tcp.offset);
        tcp.ackno = TCP_ACKNO(buf,tcp.offset);
        tcp.data_offset = tcp.offset 
                                + TCP_SIZEOF_HDR(buf,tcp.offset);

        if (MODE(carn, mode_server_access)) {
            /* We are watching for when the user attempts to access
             * servers of a specific type (HTTP, FTP, etc.). This
             * only tracks SYNs; though we could change the code
             * to track all packets. */
            if ((tcp.flags & TCP_SYN)
                && has_integer(&carn->ip, ip.src, ip.src)
                && has_integer(&carn->ip, tcp.dst, tcp.dst)) {
                char tbuf[64];
                formatNow(tbuf, sizeof(tbuf));
                logprint(carn, "%s, %d.%d.%d.%d, %d.%d.%d.%d, %d\n",
                    tbuf, P_IP_ADDR(ip.src), P_IP_ADDR(ip.dst),
                    tcp.dst);
            }
        }
        else
        switch (tcp.dst) {
        case 25:
            cn.server_ip = ip.dst;
            cn.client_ip = ip.src;
            cn.server_port = tcp.dst;
            cn.client_port = tcp.src;
            cn.server_seqno = tcp.ackno;
            cn.client_seqno = tcp.seqno;
            sniffSmtp(carn, &cn, tcp.flags | TCP_TO_SERVER, 
                            buf, tcp.data_offset, ip.max_offset);
            break;
        }
    } else if (ip.proto == 17) {
        /*UDP*/
        tcp.offset = ip.data_offset;
        tcp.dst = TCP_DST(buf,tcp.offset);
        tcp.src = TCP_SRC(buf,tcp.offset);
        tcp.data_offset = tcp.offset + 8;
        if (tcp.dst == 1812 || tcp.dst == 1813 
            || tcp.dst == 1645 || tcp.dst == 1646
            || tcp.src == 1812 || tcp.src == 1813 
            || tcp.src == 1645 || tcp.src == 1646) {
            /* This looks like a RADIUS packet, either using the
             * old port number or the new one. We are going to 
             * track both RADIUS authentication packets as well
             * as accounting packets (depending upon whwere we
             * are tapped into the network, we might see one,
             * the other, or both).*/
            sniffRadius(carn, ip.src, ip.dst, 
                            buf, tcp.data_offset, ip.max_offset);
        }

    }

    /* If one of the filters was successful, then save this packet
     * to the tracefile. This is only done*/
    if (carn->do_filter)
        carnSavePacket(carn, buf, max_offset, timestamp, usecs);

    return 0;
}


/**
 * Funkcja zwrotna, ktora obsluguje kazdy pakiet, ktory pobiera z sieci
 * podsystem 'libpcap'.
 */
void pcapHandlePacket(unsigned char *carn, 
    const struct pcap_pkthdr *framehdr, const unsigned char *buf)
{
    int max_offset = framehdr->caplen;
    sniffPacket((Carnivore*)carn, buf, max_offset, 
        framehdr->ts.tv_sec, framehdr->ts.tv_usec);
}





/**
 * Ustala tryb dzialania zgodnie z parametrem wywolania.
 */
void 
carnSetMode(Carnivore *carn, const char *value)
{
    if (startsWith(value, "email-head"))
        carn->mode = mode_email_headers;
    else if (startsWith(value, "email-addr"))
        carn->mode = mode_email_headers;
    else if (startsWith(value, "server-access"))
        carn->mode = mode_server_access;
    else if (startsWith(value, "email-content"))
        carn->mode = mode_email_content;
    else if (startsWith(value, "ip-content"))
        carn->mode = mode_ip_content;
    else
        carn->mode = -1;
}

/**
 * Analiza adresu IP. Uzywana zamiast rozwiazania opartego na 
 * gniazdach inet_addr() ze wzgledy na wieksza przenosnosc. 
 */
int 
my_inet_addr(const char addr[])
{
    int num = 0;
    int offset=0;

    while (addr[offset] && !isalnum(addr[offset]))
        offset++;

    for (; addr[offset]; offset++) {
        char c = addr[offset];
        if (isdigit(c))
            num = (num&0xFFFFFF00) | (((num&0xFF)*10) + (c - '0'));
        else if (c == '.')
            num <<= 8;
        else
            break;
    }

    return num;
}


/**
 * Odczytuje ustawienia konfiguracyjne z pliku, na przyklad "altivore.ini".
 */
void 
carnReadConfiguration(Carnivore *carn, const char filename[])
{
    FILE *fp;
    
    fp = fopen(filename, "r");
    if (fp == NULL)
        perror(filename);
    else {
        char line[1024];

        /* Dla wszystkich wierszy w pliku */
        while (fgets(line, sizeof(line), fp)) {
            char *name = line;
            char *value;
            while (*name && isspace(*name))
                name++; /*strip leading whitespace*/
            if (*name == '\0' || ispunct(*name))
                continue;/*ignore blank lines and comments*/
            value = strchr(name, '=');
            if (value == NULL)
                continue; /*ignore when no equals sign*/
            else 
                value++; /*skip the equals itself*/
            while (*value && isspace(*value))
                value++; /*strip leading whitespace*/
            while (*value && isspace(value[strlen(value)-1]))
                value[strlen(value)-1] = '\0'; /*strip trailing WS*/

            if (startsWith(name, "mode"))
                carn->mode = parseMode(value);
            else if (startsWith(name, "email.address"))
                straAddElement(&carn->email_addresses, value);
            else if (startsWith(name, "radius.account"))
                straAddElement(&carn->radius_accounts, value);
            else if (startsWith(name, "ip.address"))
                add_integer(&carn->ip, my_inet_addr(value));
            else if (startsWith(name, "tracefile")) 
                setString(&carn->tracefile, value, 0, -1);
            else if (startsWith(name, "logfile"))
                setString(&carn->logfile, value, 0, -1);
            else if (startsWith(name, "testinput"))
                straAddElement(&carn->testinput, value);
            else if (startsWith(name, "interface"))
                straAddElement(&carn->interfaces, value);
            else if (startsWith(name, "server.port"))
                add_integer(&carn->ip, strtol(value,0,0));
            else
                fprintf(stderr, "bad param: %s\n", line);
        }
        fclose(fp);
    }
}


/**
 * Przetworzenie testowego pliku wejsciowego.
 */
void 
processFile(Carnivore *carn, const char filename[])
{
    char errbuf[1024]; /*DO ZROBIENIA: jaka powinna byc dlugosc?*/
    pcap_t *hPcap;
    
    /* Open the file */
    hPcap = pcap_open_offline(filename, errbuf);
    if (hPcap == NULL) {
        fprintf(stderr, "%s: %s\n", filename, errbuf);
        return; /*ignore this file and go onto next*/
    }

    /* Pump packets through it */
    for (;;) {
        int packets_read = pcap_dispatch(
                                hPcap,               /*handle to PCAP*/
                                10,                  /*next 10 packets*/
                                pcapHandlePacket,    /*callback*/
                                (unsigned char*)carn /*canivore*/
                                );
        if (packets_read == 0)
            break;
    }

    /* Zamkniecie pliku i przejscie do nastepnego */
    pcap_close(hPcap);
}


/**
 * Sniff the wire for packets and process them using the libpcap
 * interface
 */
void 
processPackets(Carnivore *carn, const char devicename[])
{
    int traffic_seen = FALSE;
    int total_packets_processed = 0;
    pcap_t *hPcap;
    char errbuf[1024];

    hPcap = pcap_open_live(    (char*)devicename,
                            2000,      /*snap len*/
                            1,         /*promiscuous*/
                            10,        /*10-ms read timeout*/
                            errbuf
                            );
    if (hPcap == NULL) {
        fprintf(stderr, "%s: %s\n", devicename, errbuf);
        return;
    }

    /* Pump packets through it */
    for (;;) {
        int packets_read;
        
        packets_read = pcap_dispatch(
                                hPcap, /*handle to PCAP*/
                                10,        /*next 10 packets*/
                                pcapHandlePacket, /*callback*/
                                (unsigned char*)carn /*canivore*/
                                );
        total_packets_processed += packets_read;
        if (!traffic_seen && total_packets_processed > 0) {
            fprintf(stderr, "Traffic seen\n");
            traffic_seen = TRUE;
        }
    }

    /* Zamkniecie pliku i przejscie do nastepnego */
    pcap_close(hPcap);
}

/*----------------------------------------------------------------*/
int 
main(int argc, char *argv[])
{
    int i;
    Carnivore *carn;

    printf("--- ALTIVORE ---\n");
    printf("Copyright (c) 2000 by Network ICE Corporation\n");
    printf("Public disclosure of the source code does not\n");
    printf("constitute a license to use this software.\n");
    printf("Pomoc: \"altivore -?\".\n");

    /* Utworzenie podsystemu carnivore */
    carn = (Carnivore*)malloc(sizeof(Carnivore));
    memset(carn, 0, sizeof(*carn));


    /* Odczyt danych konfiguracyjnych z pliku "altivore.ini". */
    carnReadConfiguration(carn, "altivore.ini");

    /* Parse all the options from the command-line. Normally,
     * you wouldn't have any command-line options, you would
     * simply use the configuration file above. */
    for (i=1; i<argc; i++) {
        if (argv[i][0] != '-')
            straAddElement(&carn->email_addresses, argv[i]);
        else switch (argv[i][1]) {
            case 'h':
                add_integer(&carn->ip, my_inet_addr(argv[i]+2));
                break;
            case 'i':
                straAddElement(&carn->interfaces, argv[i]+2);
                break;
            case 'l':
                setString(&carn->logfile, argv[i]+2, 0, -1);
                break;
            case 'm':
                carn->mode = parseMode(argv[i]+2);
                break;
            case 'p':
                add_integer(&carn->port, strtol(argv[i]+2,0,0));
                break;
            case 'r':
                straAddElement(&carn->testinput, argv[i]+2);
                break;
            case 'w':
                setString(&carn->tracefile, argv[i]+2, 0, -1);
                break;
            case '?':
                printf("Options:\n"
                    "<email-address> adres podlegajacy filtrowaniu, np.:\n"
                    "  rob@altivore.com (dokladne dopasowanie)\n"
                    "  *@altivore.com (czesciowe dopasowanie)\n"
                    "  * (dopasowanie wszelkich adresow)\n"
                    );
                printf("-h<ip-address>\n"
                    "\tIP of host to sniff\n");
                printf("-i<devicename>\n"
                    "\tNetwork interface to sniff on\n");
                printf("-l<logfile>\n"
                    "\tText-output logging\n");
                printf("-m<mode>\n"
                    "\tMode to run in, see docs\n");
                printf("-p<port>\n"
                    "\tServer port to filter on\n");
                printf("-r<tracefile>\n"
                    "\tTest input\n");
                printf("-w<tracefile>\n"
                    "\tEvidence tracefile to write packets to\n");
                return 1;
            default:
                fprintf(stderr, "Nieznany parametr: %s\n", argv[i]);
                break;
        }
    }

    /* Wyswietlenie konfiguracji w celach debugowania */
    printf("\tmode = %s\n", modeNames[carn->mode]);
    if (carn->tracefile)
        printf("\ttracefile = %s\n", carn->tracefile);
    if (carn->logfile)
        printf("\tlogfile = %s\n", carn->logfile);
    for (i=0; i<carn->ip.count; i++)
        printf("\tip = %d.%d.%d.%d\n", P_IP_ADDR(carn->ip.list[i]));
    for (i=0; i<carn->port.count; i++)
        printf("\tport = %d\n", carn->port.list);
    for (i=0; i<carn->email_addresses.length; i++)    
        printf("\temail.address = %s\n", carn->email_addresses.str[i]);
    for (i=0; i<carn->radius_accounts.length; i++)    
        printf("\tradius.accounts = %s\n", carn->radius_accounts.str[i]);
    for (i=0; i<carn->testinput.length; i++)    
        printf("\ttestinput = %s\n", carn->testinput.str[i]);
    for (i=0; i<carn->interfaces.length; i++)    
        printf("\tinterface = %s\n", carn->interfaces.str[i]);

    /* Tylko testowanie: uzytkownik moze okreslic pliki sledzenia zawierajace
     * dane o ruchu sieciowym dla celow testowania. */
    if (carn->testinput.length > 0) {
        int i;
        for (i=0; i<carn->testinput.length; i++)
            processFile(carn, carn->testinput.str[i]);
        return 0;
    }

    /* Otworzenie karty oraz rea*/
    if (carn->interfaces.length > 0) {
        /*DO ZROBIENIA: zezwolenie na otwieranie wielu kart*/
        char *devicename = carn->interfaces.str[0];
        processPackets(carn, devicename);
    } else {
        char *devicename;
        char errbuf[1024];
        devicename = pcap_lookupdev(errbuf);
        if (devicename == NULL)
            fprintf(stderr, "%s\n", errbuf);
        else
            processPackets(carn, devicename);
    }

    return 0;
}

