from airflow.contrib.operators.bigquery_operator import BigQueryOperator
from airflow.contrib.operators.bigquery_check_operator import BigQueryCheckOperator
from airflow.operators.dummy_operator import DummyOperator
from airflow import DAG
from airflow.utils.dates import days_ago
import datetime

VERSION = '0.1.7' # Inkrementowane z każdą kolejną wersją DAG-u

DAG_NAME = 'ga4-transformation-' + VERSION

default_args = {
    'start_date': days_ago(1),  # Zmień to na ustaloną datę dla zapełniania wstecznego
    'email_on_failure': True,
    'email': 'mark@example.com',
    'email_on_retry': False,
    'depends_on_past': False,
    'retries': 3,
    'retry_delay': datetime.timedelta(minutes=10),
    'project_id': 'learning-ga4',
    'execution_timeout': datetime.timedelta(minutes=60)
}

schedule_interval = '2 4 * * *'  # Minuta, godzina, dzień miesiąca, miesiąc, dzień tygodnia

dag = DAG(DAG_NAME, default_args=default_args, schedule_interval=schedule_interval)


start = DummyOperator(
    task_id='start',
    dag=dag
)

# Wykorzystuje makro Airflowa {{ ds_nodash }} do wstawiania bieżącej daty w formacie RRRRMMDD
check_table = BigQueryCheckOperator(
    task_id='check_table',
    dag=dag,
    sql='''
    SELECT count(1) > 5000 
    FROM `learning-ga4.analytics_250021309.events_{{ ds_nodash }}`"
    '''
)

checked = DummyOperator(
    task_id='checked',
    dag=dag
)

# Funkcja do wykonywania pętli przez wiele tablic i plików SQL
def make_bq(table_id):

    task = BigQueryOperator(
        task_id='make_bq_'+table_id,
        write_disposition='WRITE_TRUNCATE',
        create_disposition='CREATE_IF_NEEDED',
        destination_dataset_table='learning_ga4.ga4_aggregations.{}${{ ds_nodash}}'.format(table_id),
        sql='./ga4_sql/{}.sql'.format(table_id),
        use_legacy_sql=False,
        dag=dag
    )

    return task

ga_tables = [
  'pageview-aggs',
  'ga4-join-crm',
  'ecom-fields'
]

ga_aggregations = [] # Pomocne, jeśli wykonujesz inne downstreamowe transformacje
for table in ga_tables:
  task = make_bq(table)
  checked >> task
  ga_aggregations.append(task)


# Tworzenie DAG-u 
start >> check_table >> checked