open System.Net
open System.Net.Sockets
open System.IO
open System.Text.RegularExpressions
open System.Text

/// Tabela typów MIME.
let mimeTypes =
    dict [".html", "text/html";
          ".htm", "text/html";
          ".txt", "text/plain";
          ".gif", "image/gif";
          ".jpg", "image/jpeg";
          ".png", "image/png"]

/// Wyznaczanie typu MIME na podstawie rozszerzenia pliku.
let getMimeType(ext) =
    if mimeTypes.ContainsKey(ext) then mimeTypes.[ext]
    else "binary/octet"

/// We wzorcu Regex1 wykorzystano wyrażenie regularne do dopasowania jednego elementu.
let (|Regex1|_|) (patt : string) (inp : string) =
    try Some(Regex.Match(inp, patt).Groups.Item(1).Captures.Item(0).Value)
    with _ -> None

/// Katalog główny z udostępnianymi danymi
let root = @"c:\inetpub\wwwroot"

/// Obsługa połączenia TCP używanego na potrzeby żądań HTTP GET. Używane jest zadanie asynchroniczne
/// na wypadek, gdyby późniejsze operacje związane z obsługą żądania miały być asynchroniczne.
let handleRequest(client : TcpClient) = async {
    use stream = client.GetStream()
    let out = new StreamWriter(stream)
    let headers (lines : seq<string>) =
        let printLine s = s |> fprintf out "%s\r\n"
        lines |> Seq.iter printLine
        // Przed danymi (jeśli istnieją) potrzebny jest pusty wiersz
        printLine ""
        out.Flush()
    let notFound () = headers ["HTTP/1.0 404 Not Found"]
    let inp = new StreamReader(stream)
    let request = inp.ReadLine()
    match request with
    | "GET / HTTP/1.0" | "GET / HTTP/1.1" ->
        // Żądania do katalogu głównego są przekierowywane do strony głównej
        headers ["HTTP/1.0 302 Found"; "Location: http://localhost:8090/iisstart.htm"]
    | Regex1 "GET /(.*?) HTTP/1\\.[01]$" fileName ->
        let fname = Path.Combine(root, fileName)
        let mimeType = getMimeType(Path.GetExtension(fname))
        if not(File.Exists(fname)) then notFound()
        else
            let content = File.ReadAllBytes fname
            ["HTTP/1.0 200 OK";
            sprintf "Content-Length: %d" content.Length;
            sprintf "Content-Type: %s" mimeType]
            |> headers
            stream.Write(content, 0, content.Length)
    | _ -> notFound()}

/// Serwer działa asynchronicznie; żądania są obsługiwane sekwencyjnie
let server = async { 
    let socket = new TcpListener(IPAddress.Parse("127.0.0.1"), 8090)
    socket.Start()
    while true do
        use client = socket.AcceptTcpClient()
        do! handleRequest client}

//Async.Start server;;
//val it : unit = ()
