package przyklady;

import java.sql.*;
import javax.naming.*;
import javax.ejb.*;
import java.util.*;

/**
 * Demonstracja komponentu encyjnego bezposrednio zarzadzajacego trwaloscia.
 * Ten komponent encyjny reprezentuje konto bankowe. 
 */
public class KontoKomponent implements EntityBean {
  protected EntityContext kontekst;

  //
  // Pola reprezentujace stan komponentu
  //

  private String identyfikatorKonta;     // KG 
  private String nazwiskoWlasciciela; 
  private double saldo;

  public KontoKomponent() {
    System.out.println("Nowy obiekt Javy (komponent encyjny reprezentujacy konto bankowe) zostal stworzony przez kontener EJB."); 
  }

  //
  // Metody biznesowe
  //
 
 /**
  * Wplaca okreslona kwote na konto.
  */
  public void wplac(double kwota) throws KontoException {
    System.out.println("Wywolana metoda wplac(" + kwota + ").");
    saldo += kwota; 
  }

  /**
   * Wyplaca okreslona kwote z konta.
   * @throw KontoException, jesli kwota jest wieksza niz saldo
   */
  public void wyplac(double kwota) throws KontoException {
    System.out.println("Wywolana metoda wyplac(" + kwota + ").");

    if (kwota > saldo) {
      throw new KontoException("Saldo Twojego konta wynosi " + saldo + "! Nie mozesz wyplacic" + kwota + "!");
    }

    saldo -= kwota;
  }

  // Metody zwracajace/ustawiajace pola komponentu encyjnego
  public double getSaldo() {
    System.out.println("Wywolana metoda getSaldo().");
    return saldo;
  }

  public void setNazwiskoWlasciciela(String nazwisko) {
    System.out.println("Wywolana metoda setNazwiskoWlasciciela().");
    nazwiskoWlasciciela = nazwisko;
  }

  public String getNazwiskoWlasciciela() {
    System.out.println("Wywolana metoda getNazwiskoWlasciciela().");
    return nazwiskoWlasciciela; 
  }

  public String getKontoID() {
    System.out.println("Wywolana metoda getKontoID().");
    return identyfikatorKonta;
  }

  public void setKontoID(String identyfikator) {
    System.out.println("Wywolana metoda setKontoID().");
    this.identyfikatorKonta = identyfikator;
  }

  /**
   * To jest domowa metoda biznesowa niezalezna od danych konkretnego konta
   * Zwraca laczna ilosc srodkow na wszyskich kontach w naszym banku.
   */
  public double ejbHomeGetLacznaWartoscKont() throws KontoException {
    PreparedStatement pstmt = null;
    Connection conn = null;

    try {

      System.out.println("Wywolana metoda ejbHomeGetLacznaWartoscKont().");

      /*
       * Pozyskuje polaczenie z baza danych
       */
      conn = getConnection();

      /*
       * Pobiera laczna wartosc wszystkich kont
       */
      pstmt = conn.prepareStatement("select sum(saldo) as lacznie from konta"); 
      ResultSet rs = pstmt.executeQuery();

      /*
       * Zwraca sume
       */
      if (rs.next()) {
        return rs.getDouble("lacznie"); 
      } 
    }
    catch (Exception e) {
      e.printStackTrace();
      throw new KontoException(e);
    }
    finally {
      /*
       * Zwalnia polaczenie z baza danych dla innych komponentow
       */
      try { if (pstmt != null) pstmt.close(); }
      catch (Exception e) {}
      try { if (conn != null) conn.close(); }
      catch (Exception e) {} 
    }

    throw new KontoException("Blad!");
  }

  /**
   * Pobiera polaczenie JDBC z puli polaczen.
   *
   * @return Polaczenie JDBC
   */
  public Connection getConnection() throws Exception {
    try {
      Context kontekst = new InitialContext();
      javax.sql.DataSource ds = (javax.sql.DataSource)kontekst.lookup("java:comp/env/jdbc/ejbPool");
      return ds.getConnection();
    } 
    catch (Exception e) {
      System.err.println("Nie mozna odnalezc zrodla danych!");
      e.printStackTrace();
      throw e;
    }
  }

  //
  // Metody wymagane przez EJB
  //

  /**
   * Metoda wywolywana przez kontener. Sluzy do pozyskiwania
   * potrzebnych zasobow.
   */
  public void ejbActivate() {
    System.out.println("Wywolana metoda ejbActivate().");
  }

  /**
   * Usuwa dane komponentu encyjnego z bazy danych.
   * Odpowiada wywolaniu przez klienta metody domowej remove().
   */
  public void ejbRemove() throws RemoveException {
    System.out.println("Wywolana metoda ejbRemove().");

    /*
     * Pamietaj, ze klasa komponentu encyjnego moze byc wykorzystana
     * do reprezentowania roznych egzemplarzy danych. Skad wiec metoda
     * wie, ktory egzemplarz usunac z bazy danych?
     *
     * Rozwiazaniem jest odpowiednie zapytanie do kontenera
     * realizowane za pomoca obiektu kontekstu encji. Za pomoca
     * uzyskanego w ten sposob klucza glownego, okreslamy,
     * ktory egzemplarz danych, wskazywany przez KG, powinnismy
     * usunac z bazy danych.
     */
    KontoKG kg = (KontoKG) kontekst.getPrimaryKey();
    String identyfikator = kg.identyfikatorKonta;

    PreparedStatement pstmt = null;
    Connection conn = null;
    try {
      /*
       * 1) Pozyskanie polaczenia JDBC
       */
      conn = getConnection();

      /*
       * 2) Usuniecie konta z bazy danych
       */
      pstmt = conn.prepareStatement("delete from konta where identyfikator = ?");
      pstmt.setString(1, identyfikator);

      /*
       * 3) wygenerowanie wyjatku na poziomie systemu, jesli wystapil blad
       */
      if (pstmt.executeUpdate() ==0) {
        throw new RemoveException("Konto " + kg +" nie moglo zostac usuniete z bazy danych");
      }
    }
    catch (Exception ex) {
      throw new EJBException(ex.toString());
    }
    finally {
      /*
       * 4) Zwolnienie polaczenia z baza danych
       */
      try { if (pstmt != null) pstmt.close(); }
      catch (Exception e) {}
      try { if (conn != null) conn.close(); }
      catch (Exception e) {}
    }
  }

  /**
   * Metoda wywolywana przez kontener. Zwalnia przetrzymywane zasoby
   * przed pasywacja.
   */
  public void ejbPassivate() {
    System.out.println("Wywolana metoda ejbPassivate().");
  }

  /**
   * Metoda wywolywana przez kontener. Aktualizuje komponent encyjny
   * w pamieci operacyjnej tak, by reprezentowal aktualne wartosci
   * przechowywane w bazie danych.
   */
  public void ejbLoad() {
    System.out.println("Wywolana metoda ejbLoad().");

    /*
     * Ponownie wykorzystujemy kontekst encji do pozyskania
     * aktualnego klucza glownego po to, bysmy mogli wczytac
     * wlasciwe dane.
     */
    KontoKG kg = (KontoKG) kontekst.getPrimaryKey();
    String identyfikator = kg.identyfikatorKonta;

    PreparedStatement pstmt = null;
    Connection conn = null;
    try {
      /*
       * 1) Pozyskuje nowe polaczenie z baza danych
       */
      conn = getConnection();

      /*
       * 2) Pobiera dane konta z bazy danych na podstawie jego identyfikatora
       */
      pstmt = conn.prepareStatement("select nazwisko, saldo from konta where identyfikator = ?");
      pstmt.setString(1, identyfikator);
      ResultSet rs = pstmt.executeQuery();
      rs.next();
      nazwiskoWlasciciela = rs.getString("nazwisko");
      saldo = rs.getDouble("saldo");
    }
    catch (Exception ex) {
      throw new EJBException("Konto " + kg + " nie moglo zostac wczytane z bazy danych", ex);
    } 
    finally {
      /*
       * 3) Zwalnia polaczenie z baza danych
       */
      try { if (pstmt != null) pstmt.close(); }
      catch (Exception e) {}
      try { if (conn != null) conn.close(); }
      catch (Exception e) {} 
    }
  }

  /**
   * Metoda wywolywana przez kontener. Aktualizuje baze danych po to,
   * by przechowywane w niej wartosci odpowiadaly altualnym wartosciom
   * przetrzymywanym przez egzemplarz komponentu encyjnego w pamieci operacyjnej.
   */
  public void ejbStore() {
    System.out.println("Wywolana metoda ejbStore().");
    PreparedStatement pstmt = null;
    Connection conn = null;
    try {
      /*
       * 1) Pozyskanie nowego polaczenia z baza danych
       */
      conn = getConnection();

      /*
       * 2) Zapisanie informacji o koncie w bazie danych
       */
      pstmt = conn.prepareStatement("update konta set nazwisko = ?, saldo = ? where identyfikator = ?");
      pstmt.setString(1, nazwiskoWlasciciela);
      pstmt.setDouble(2, saldo);
      pstmt.setString(3, identyfikatorKonta);
      pstmt.executeUpdate();
    }
    catch (Exception ex) {
      throw new EJBException("Konto " + identyfikatorKonta + " nie moglo zostac zapisane w bazie danych", ex);
    }
    finally {
      /*
       * 3) Zwalnia polaczenie z baza danych
       */
      try { if (pstmt != null) pstmt.close(); }
      catch (Exception e) {}
      try { if (conn != null) conn.close(); }
      catch   (Exception e) {}
    }
  }

  /**
   * Metoda wywolywana przez kontener. Laczy egzemplarz komponentu
   * z konkretnym kontekstem. Za posrednictwem tego kontekstu uzyskujemy
   * dostep do charakterystycznych cech komponentu.
   */
  public void setEntityContext(EntityContext kontekst) {
    System.out.println("Wywolana metoda setEntityContext().");
    this.kontekst = kontekst;
  }

  /**
   * Metoda wywolywana przez kontener. Odlacza egzemplarz komponentu
   * od srodowiska konkretnego kontekstu.
   */
  public void unsetEntityContext() {
    System.out.println("Wywolana metoda unsetEntityContext().");
    this.kontekst = null;
  }

  /**
   * Metoda wywolywana po ejbCreate(). Komponent uzyskuje z jej pomoca
   * swoj obiekt EJB z kontekstu i przekazuje go jako argument.
   */
  public void ejbPostCreate(String identyfikatorKonta, String nazwiskoWlasciciela) {
  }

  /**
   * To jest metoda inicjalizujaca odpowiadajaca metodzie
   * create() z interfejsu domowego. W momencie, gdy klient
   * wywoluje metode create() obiektu domowego, obiekt
   * domowy wywoluje metode ejbCreate().
   *
   * @return Klucz glowny dla tego konta bankowego
   */
  public KontoKG ejbCreate(String identyfikatorKonta, String nazwiskoWlasciciela) throws CreateException {
    PreparedStatement pstmt = null;
    Connection conn = null;
    try {
      System.out.println("Wywolana metoda ejbCreate().");
      this.identyfikatorKonta = identyfikatorKonta;
      this.nazwiskoWlasciciela = nazwiskoWlasciciela;
      this.saldo = 0;

      /*
       * Pozyskuje polaczenie z baza danych
       */
      conn = getConnection();

      /*
       * Dodaje nowe konto do bazy danych
       */
      pstmt = conn.prepareStatement("insert into konta (identyfikator, nazwisko, saldo) values (?, ?, ?)");
      pstmt.setString(1, identyfikatorKonta);
      pstmt.setString(2, nazwiskoWlasciciela);
      pstmt.setDouble(3, saldo);
      pstmt.executeUpdate();

      /*
       * Generuje i zwraca klucz glowny konta
       */
      return new KontoKG(identyfikatorKonta);
    }
    catch (Exception e) {
      throw new CreateException(e.toString());
    }
    finally {
      /*
       * Zwalnia polaczenie z baza danych dla innych komponentow
       */
      try { if (pstmt != null) pstmt.close(); }
      catch (Exception e) {}
      try { if (conn != null) conn.close(); }
      catch (Exception e) {}
    }
  }

  /**
   * Odnajduje konto na podstawie jego klucza glownego
   */
  public KontoKG ejbFindByPrimaryKey(KontoKG klucz) throws FinderException {
    PreparedStatement pstmt = null;
    Connection conn = null;
    try {
      System.out.println("Wywolana metoda ejbFindByPrimaryKey(" + klucz + ").");

      /*
       * Pozyskuje polaczenie z baza danych
       */
      conn = getConnection();

      /*
       * Znajduje odpowiednia encje w bazie danych
       */
      pstmt = conn.prepareStatement("select identyfikator from konta where identyfikator = ?");
      pstmt.setString(1, klucz.toString());
      ResultSet rs = pstmt.executeQuery();
      rs.next();

      /*
       * Nie wystapily zadne bledy, mozna wiec zwrocic klucz glowny
       */
      return klucz;
    }
    catch (Exception e) {
      throw new FinderException(e.toString());
    }
    finally {
      /*
       * Zwolnij polaczenie z baza danych dla innych komponentow
       */
      try { if (pstmt != null) pstmt.close(); }
      catch (Exception e) {}
      try { if (conn != null) conn.close(); }
      catch (Exception e) {}
    }
  }

  /**
   * Wyszukuje konto na podstawie nazwiska wlasciciela
   */
  public Collection ejbFindDlaNazwiskaWlasciciela(String nazwisko) throws FinderException {
    PreparedStatement pstmt = null;
    Connection conn = null;
    Vector v = new Vector();

    try {
      System.out.println("Wywolana metoda ejbFindDlaNazwiskaWlasciciela(" + nazwisko + ").");

      /*
       * Pozyskuje polaczenie z baza danych
       */
      conn = getConnection();

      /*
       * Znajduje klucze glowne w bazie danych
       */
      pstmt = conn.prepareStatement("select identyfikator from konta where nazwisko = ?");
      pstmt.setString(1, nazwisko);
      ResultSet rs = pstmt.executeQuery();

      /*
       * Dodaje wszystkie znalezione klucze glowne do wektora
       */
      while (rs.next()) {
        String identyfikator = rs.getString("identyfikator");
        v.addElement(new KontoKG(identyfikator));
      }

      /*
       * Zwraca wektor zawierajacy klucze glowne
       */
      return v;
    }
    catch (Exception e) {
      throw new FinderException(e.toString());
    }
    finally {
      /*
       * Zwalnia polaczenie z baza danych dla innych komponentow
       */
      try { if (pstmt != null) pstmt.close(); }
      catch (Exception e) {}
      try { if (conn != null) conn.close(); }
      catch (Exception e) {}
    }
  }
}

