#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
def try_finally_example(filename):
    print("* Otwarcie pliku")
    handle = open(filename, encoding="utf-8")  # Może spowodować zgłoszenie wyjątku OSError
    try:
        print("* Odczytanie danych")
        return handle.read()                   # Może spowodować zgłoszenie wyjątku UnicodeDecodeError
    finally:
        print("* Wywołanie funkcji close()")
        handle.close()                         # Zawsze wykonywane po bloku try


print("Przykład 2")
try:
    filename = "random_data.txt"
    
    with open(filename, "wb") as f:
        f.write(b"\xf1\xf2\xf3\xf4\xf5")  # Nieprawidłowa wartość utf-8
    
    data = try_finally_example(filename)
    # Ten kod nie powinien być wykonany
    import sys
    
    sys.exit(1)
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 3")
try:
    try_finally_example("nieistniejący_klucz.txt")
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 4")
import json

def load_json_key(data, key):
    try:
        print("* Wczytywanie danych JSON")
        result_dict = json.loads(data)  # Może spowodować zgłoszenie wyjątku ValueError
    except ValueError:
        print("* Obsługa wyjątku ValueError")
        raise KeyError(key)
    else:
        print("* Wyszukiwanie klucza")
        return result_dict[key]         # Może spowodować zgłoszenie wyjątku KeyError


print("Przykład 5")
assert load_json_key('{"foo": "bar"}', "foo") == "bar"


print("Przykład 6")
try:
    load_json_key('{"foo": nieprawidłowe dane', "foo")
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 7")
try:
    load_json_key('{"foo": "bar"}', "nie istnieje")
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 8")
UNDEFINED = object()
DIE_IN_ELSE_BLOCK = False

def divide_json(path):
    print("* Otwarcie pliku")
    handle = open(path, "r+")                        # Może spowodować zgłoszenie wyjątku OSError
    try:
        print("* Odczytanie danych")
        data = handle.read()                         # Może spowodować zgłoszenie wyjątku UnicodeDecodeError
        print("* Wczytywanie danych JSON")
        op = json.loads(data)                        # Może spowodować zgłoszenie wyjątku ValueError
        print("* Przetwarzanie danych")
        value = op["numerator"] / op["denominator"]  # Może spowodować zgłoszenie wyjątku ZeroDivisionError
    except ZeroDivisionError:
        print("* Obsługa wyjątku ZeroDivisionError")
        return UNDEFINED
    else:
        print("* Zapisywanie danych")
        op["result"] = value
        result = json.dumps(op)
        handle.seek(0)                               # Może spowodować zgłoszenie wyjątku OSError
        if DIE_IN_ELSE_BLOCK:
            import errno
            import os

            raise OSError(errno.ENOSPC, os.strerror(errno.ENOSPC))
        handle.write(result)                         # Może spowodować zgłoszenie wyjątku OSError
        return value
    finally:
        print("* Wywołanie funkcji close()")
        handle.close()                               # Zawsze wykonywane


print("Przykład 9")
temp_path = "random_data.json"

with open(temp_path, "w") as f:
    f.write('{"numerator": 1, "denominator": 10}')

assert divide_json(temp_path) == 0.1


print("Przykład 10")
with open(temp_path, "w") as f:
    f.write('{"numerator": 1, "denominator": 0}')

assert divide_json(temp_path) is UNDEFINED


print("Przykład 11")
try:
    with open(temp_path, "w") as f:
        f.write('{"numerator": 1 nieprawidłowe dane')
    
    divide_json(temp_path)
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 12")
try:
    with open(temp_path, "w") as f:
        f.write('{"numerator": 1, "denominator": 10}')
    DIE_IN_ELSE_BLOCK = True
    
    divide_json(temp_path)
except:
    logging.exception('Oczekiwane')
else:
    assert False
