#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
import subprocess

# Dzięki tym poleceniom kod będzie działał także w Windows
# import os
# os.environ['COMSPEC'] = 'powershell'
result = subprocess.run(
    ["echo", "Witaj z procesu potomnego!"],
    capture_output=True,
    # Dzięki temu poleceniu kod będzie działał także w Windows
    # shell=True,
    encoding="utf-8",
)

result.check_returncode()  # Brak wyjątku oznacza poprawne zakończenie zadania
print(result.stdout)


print("Przykład 2")
# Dzięki temu poleceniu kod będzie działał także w Windows
# proc = subprocess.Popen(['sleep', '1'], shell=True)
proc = subprocess.Popen(["sleep", "1"])
while proc.poll() is None:
    print("Pracuję...")
    # Miejsce na zadania, których wykonanie wymaga dużo czasu
    import time

    time.sleep(0.3)

print("Kod wyjścia", proc.poll())


print("Przykład 3")
import time

start = time.perf_counter()
sleep_procs = []
for _ in range(10):
    # Dzięki temu poleceniu kod będzie działał także w Windows
    # proc = subprocess.Popen(['sleep', '1'], shell=True)
    proc = subprocess.Popen(["sleep", "1"])
    sleep_procs.append(proc)


print("Przykład 4")
for proc in sleep_procs:
    proc.communicate()

end = time.perf_counter()
delta = end - start
print(f"Zakończono w ciągu {delta:.3f} sekund")


print("Przykład 5")
import os

# W systemie Windows po zainstalowaniu OpenSSL może wystąpić
# konieczność utworzenia aliasu w ścieżce PowerShell:
# $env:path = $env:path + ";C:\Program Files\OpenSSL-Win64\bin"

def run_encrypt(data):
    env = os.environ.copy()
    env["password"] = "zf7ShyBhZOraQDdE/FiZpm/m/8f9X+M1"
    proc = subprocess.Popen(
        ["openssl", "enc", "-des3", "-pbkdf2", "-pass", "env:password"],
        env=env,
        stdin=subprocess.PIPE,
        stdout=subprocess.PIPE,
    )
    proc.stdin.write(data)
    proc.stdin.flush()  # Gwarantujemy, że proces potomny otrzyma dane wejściowe
    return proc


print("Przykład 6")
procs = []
for _ in range(3):
    data = os.urandom(10)
    proc = run_encrypt(data)
    procs.append(proc)


print("Przykład 7")
for proc in procs:
    out, _ = proc.communicate()
    print(out[-10:])


print("Przykład 8")
def run_hash(input_stdin):
    return subprocess.Popen(
        ["openssl", "dgst", "-sha256", "-binary"],
        stdin=input_stdin,
        stdout=subprocess.PIPE,
    )


print("Przykład 9")
encrypt_procs = []
hash_procs = []
for _ in range(3):
    data = os.urandom(100)

    encrypt_proc = run_encrypt(data)
    encrypt_procs.append(encrypt_proc)

    hash_proc = run_hash(encrypt_proc.stdout)
    hash_procs.append(hash_proc)

    # Trzeba zagwarantować, że proces potomny używa danych wejściowych,
    # a metoda communicate() przypadkowo nie zabierze tych danych wejściowych 
    # procesowi potomnemu, ponadto trzeba pozwolić SIGPIPE na propagowanie 
    # procesu upstream, jeśli proces downstream zostanie zakończony.
    encrypt_proc.stdout.close()
    encrypt_proc.stdout = None


print("Przykład 10")
for proc in encrypt_procs:
    proc.communicate()
    assert proc.returncode == 0

for proc in hash_procs:
    out, _ = proc.communicate()
    print(out[-10:])
    assert proc.returncode == 0


print("Przykład 11")
# Dzięki temu poleceniu kod będzie działał także w Windows
# proc = subprocess.Popen(['sleep', '10'], shell=True)
proc = subprocess.Popen(["sleep", "10"])
try:
    proc.communicate(timeout=0.1)
except subprocess.TimeoutExpired:
    proc.terminate()
    proc.wait()

print("Kod wyjścia", proc.poll())
