#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
names = ['Sokrates', 'Archimedes', 'Platon', 'Arystoteles']
names.sort(key=len)
print(names)


# Przykład 2.
def log_missing():
    print('Klucz został dodany.')
    return 0


# Przykład 3.
from collections import defaultdict

current = {'zielony': 12, 'niebieski': 3}
increments = [
    ('czerwony', 5),
    ('niebieski', 17),
    ('pomarańczowy', 9),
]
result = defaultdict(log_missing, current)
print('Przed:', dict(result))
for key, amount in increments:
    result[key] += amount
print('Po: ', dict(result))


# Przykład 4.
def increment_with_report(current, increments):
    added_count = 0

    def missing():
        nonlocal added_count  # Domknięcie wraz z informacjami o stanie.
        added_count += 1
        return 0

    result = defaultdict(missing, current)
    for key, amount in increments:
        result[key] += amount

    return result, added_count


# Przykład 5.
result, count = increment_with_report(current, increments)
assert count == 2
print(result)


# Przykład 6.
class CountMissing:
    def __init__(self):
        self.added = 0

    def missing(self):
        self.added += 1
        return 0


# Przykład 7.
counter = CountMissing()
result = defaultdict(counter.missing, current)  # Odwołanie do metody.
for key, amount in increments:
    result[key] += amount
assert counter.added == 2
print(result)


# Przykład 8.
class BetterCountMissing:
    def __init__(self):
        self.added = 0

    def __call__(self):
        self.added += 1
        return 0

counter = BetterCountMissing()
assert counter() == 0
assert callable(counter)


# Przykład 9.
counter = BetterCountMissing()
result = defaultdict(counter, current)  # Oparcie na metodzie __call__().
for key, amount in increments:
    result[key] += amount
assert counter.added == 2
print(result)
