#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
# my_module.py
def determine_weight(volume, density):
    if density <= 0:
        raise ValueError('Gęstość musi być wartością dodatnią')

try:
    determine_weight(1, 0)
except ValueError:
    pass
else:
    assert False


# Przykład 2.
# my_module.py
class Error(Exception):
    """Klasa bazowa dla wszystkich wyjątków zgłaszanych przez ten moduł."""

class InvalidDensityError(Error):
    """Wystąpił problem z podaną wartością gęstości."""

class InvalidVolumeError(Error):
    """Wystąpił problem z podaną wartością wagi."""

def determine_weight(volume, density):
    if density < 0:
        raise InvalidDensityError('Gęstość musi być wartością dodatnią.')
    if volume < 0:
        raise InvalidVolumeError('Objętość musi być dodatnia.')
    if volume == 0:
        density / volume


# Przykład 3.
class my_module:
    Error = Error
    InvalidDensityError = InvalidDensityError

    @staticmethod
    def determine_weight(volume, density):
        if density < 0:
            raise InvalidDensityError('Gęstość musi być wartością dodatnią.')
        if volume < 0:
            raise InvalidVolumeError('Objętość musi być dodatnia.')
        if volume == 0:
            density / volume

try:
    weight = my_module.determine_weight(1, -1)
except my_module.Error:
    logging.exception('Nieoczekiwany błąd')
else:
    assert False


# Przykład 4.
SENTINEL = object()
weight = SENTINEL
try:
    weight = my_module.determine_weight(-1, 1)
except my_module.InvalidDensityError:
    weight = 0
except my_module.Error:
    logging.exception('Wystąpił błąd w kodzie wywołującym')
else:
    assert False

assert weight is SENTINEL


# Przykład 5.
try:
    weight = SENTINEL
    try:
        weight = my_module.determine_weight(0, 1)
    except my_module.InvalidDensityError:
        weight = 0
    except my_module.Error:
        logging.exception('Wystąpił błąd w kodzie wywołującym')
    except Exception:
        logging.exception('Wystąpił błąd w kodzie API!')
        raise  # Ponowne zgłoszenie wyjątku komponentowi wywołującemu ten kod.
    else:
        assert False
    
    assert weight == 0
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 6.
# my_module.py

class NegativeDensityError(InvalidDensityError):
    """Podano ujemną wartość dla gęstości."""


def determine_weight(volume, density):
    if density < 0:
        raise NegativeDensityError('Gęstość musi być wartością dodatnią')


# Przykład 7.
try:
    my_module.NegativeDensityError = NegativeDensityError
    my_module.determine_weight = determine_weight
    try:
        weight = my_module.determine_weight(1, -1)
    except my_module.NegativeDensityError:
        raise ValueError('Wartość dla gęstości musi być dodatnia')
    except my_module.InvalidDensityError:
        weight = 0
    except my_module.Error:
        logging.exception('Wystąpił błąd w kodzie wywołującym')
    except Exception:
        logging.exception('Wystąpił błąd w kodzie API!')
        raise
    else:
        assert False
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 8.
# my_module.py
class Error(Exception):
    """Klasa bazowa dla wszystkich wyjątków zgłaszanych przez ten moduł."""

class WeightError(Error):
    """Klasa bazowa dla błędów związanych z obliczaniem wagi."""

class VolumeError(Error):
    """Klasa bazowa dla błędów związanych z obliczaniem pojemności."""

class DensityError(Error):
    """Klasa bazowa dla błędów związanych z obliczaniem gęstości."""
