#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
class OldResistor:
    def __init__(self, ohms):
        self._ohms = ohms

    def get_ohms(self):
        return self._ohms

    def set_ohms(self, ohms):
        self._ohms = ohms


# Przykład 2.
r0 = OldResistor(50e3)
print('Przed:', r0.get_ohms())
r0.set_ohms(10e3)
print('Po: ', r0.get_ohms())


# Przykład 3.
r0.set_ohms(r0.get_ohms() - 4e3)
assert r0.get_ohms() == 6e3


# Przykład 4.
class Resistor:
    def __init__(self, ohms):
        self.ohms = ohms
        self.voltage = 0
        self.current = 0

r1 = Resistor(50e3)
r1.ohms = 10e3
print(f'{r1.ohms} om, '
      f'{r1.voltage} wolt, '
      f'{r1.current} amperów')


# Przykład 5.
r1.ohms += 5e3


# Przykład 6.
class VoltageResistance(Resistor):
    def __init__(self, ohms):
        super().__init__(ohms)
        self._voltage = 0

    @property
    def voltage(self):
        return self._voltage

    @voltage.setter
    def voltage(self, voltage):
        self._voltage = voltage
        self.current = self._voltage / self.ohms


# Przykład 7.
r2 = VoltageResistance(1e3)
print(f'Przed: {r2.current:.2f} amperów')
r2.voltage = 10
print(f'Po:  {r2.current:.2f} amperów')


# Przykład 8.
class BoundedResistance(Resistor):
    def __init__(self, ohms):
        super().__init__(ohms)

    @property
    def ohms(self):
        return self._ohms

    @ohms.setter
    def ohms(self, ohms):
        if ohms <= 0:
            raise ValueError(f'Wartość {ohms} musi być większa niż 0')
        self._ohms = ohms


# Przykład 9.
try:
    r3 = BoundedResistance(1e3)
    r3.ohms = 0
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 10.
try:
    BoundedResistance(-5)
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 11.
class FixedResistance(Resistor):
    def __init__(self, ohms):
        super().__init__(ohms)

    @property
    def ohms(self):
        return self._ohms

    @ohms.setter
    def ohms(self, ohms):
        if hasattr(self, '_ohms'):
            raise AttributeError("Nie można ustawić atrybutu.")
        self._ohms = ohms


# Przykład 12.
try:
    r4 = FixedResistance(1e3)
    r4.ohms = 2e3
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 13.
class MysteriousResistor(Resistor):
    @property
    def ohms(self):
        self.voltage = self._ohms * self.current
        return self._ohms

    @ohms.setter
    def ohms(self, ohms):
        self._ohms = ohms


# Przykład 14.
r7 = MysteriousResistor(10)
r7.current = 0.01
print(f'Przed: {r7.voltage:.2f}')
r7.ohms
print(f'Po:  {r7.voltage:.2f}')
