#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
from time import sleep
from datetime import datetime

def log(message, when=datetime.now()):
    print(f'{when}: {message}')

log('Witajcie!')
sleep(0.1)
log('Witajcie ponownie!')


# Przykład 2.
def log(message, when=None):
    """Wyświetlenie komunikatu debugowania wraz ze znacznikiem czasu.

    Argumenty:
        message: Komunikat przeznaczony do wyświetlenia.
        when: Data i godzina wyświetlenia tego komunikatu.
            Wartości domyślne to aktualna data i godzina.
    """
    if when is None:
        when = datetime.now()
    print(f'{when}: {message}')


# Przykład 3.
log('Witaj!')
sleep(0.1)
log('Witaj ponownie!')


# Przykład 4.
import json

def decode(data, default={}):
    try:
        return json.loads(data)
    except ValueError:
        return default


# Przykład 5.
foo = decode('nieprawidłowe dane')
foo['stuff'] = 5
bar = decode('także nieprawidłowe dane')
bar['meep'] = 1
print('Foo:', foo)
print('Bar:', bar)


# Przykład 6.
assert foo is bar


# Przykład 7.
def decode(data, default=None):
    """Wczytanie danych JSON z ciągu tekstowego

    Argumenty:
        data: Dane JSON przeznaczone do dekodowania.
        default: Wartość zwrotna, jeśli dekodowanie zakończy się niepowodzeniem.
            Domyślnie będzie to pusty słownik.
    """
    try:
        return json.loads(data)
    except ValueError:
        if default is None:
            default = {}
        return default


# Przykład 8.
foo = decode('nieprawidłowe dane')
foo['stuff'] = 5
bar = decode('także nieprawidłowe dane')
bar['meep'] = 1
print('Foo:', foo)
print('Bar:', bar)
assert foo is not bar
