#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
a = 0b10111011
b = 0xc5f
print('Wartość binarna wynosi %d, wartość szesnastkowa wynosi %d' % (a, b))


# Przykład 2.
key = 'my_var'
value = 1.234
formatted = '%-10s = %.2f' % (key, value)
print(formatted)


# Przykład 3.
try:
    reordered_tuple = '%-10s = %.2f' % (value, key)
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 4.
try:
    reordered_string = '%.2f = %-10s' % (key, value)
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 5.
pantry = [
    ('awokado', 1.25),
    ('banany', 2.5),
    ('wiśnie', 15),
]
for i, (item, count) in enumerate(pantry):
    print('#%d: %-10s = %.2f' % (i, item, count))


# Przykład 6.
for i, (item, count) in enumerate(pantry):
    print('#%d: %-10s = %d' % (
        i + 1,
        item.title(),
        round(count)))


# Przykład 7.
template = '%s uwielbia jeść. Zobacz, jak %s gotuje.'
name = 'Maciej'
formatted = template % (name, name)
print(formatted)


# Przykład 8.
name = 'bartek'
formatted = template % (name.title(), name.title())
print(formatted)


# Przykład 9.
key = 'my_var'
value = 1.234

old_way = '%-10s = %.2f' % (key, value)

new_way = '%(key)-10s = %(value).2f' % {
    'key': key, 'value': value}  # Kolejność pierwotna.

reordered = '%(key)-10s = %(value).2f' % {
    'value': value, 'key': key}  # Kolejność zmieniona.

assert old_way == new_way == reordered


# Przykład 10.
name = 'Maciej'

template = '%s uwielbia jeść. Zobacz, jak %s gotuje.'
before = template % (name, name)   # Krotka.

template = '%(name)s uwielbia jeść. Zobacz, jak %(name)s gotuje.'
after = template % {'name': name}  # Słownik.

assert before == after


# Przykład 11.
for i, (item, count) in enumerate(pantry):
    before = '#%d: %-10s = %d' % (
        i + 1,
        item.title(),
        round(count))

    after = '#%(loop)d: %(item)-10s = %(count)d' % {
        'loop': i + 1,
        'item': item.title(),
        'count': round(count),
    }

    assert before == after


# Przykład 12.
soup = 'pomidorowa'
formatted = 'Dzisiejsza zupa to %(soup)s.' % {'soup': soup}
print(formatted)


# Przykład 13.
menu = {
    'soup': 'pomidorowa',
    'oyster': 'kumamoto',
    'special': 'sznycel',
}
template = ('Dzisiejsza zupa to %(soup)s, '
            'kup jedną i otrzymaj dwie ostrygi %(oyster)s, '
            'a także naszą przystawkę specjalną %(special)s.')
formatted = template % menu
print(formatted)


# Przykład 14.
a = 1234.5678
formatted = format(a, ',.2f')
print(formatted)

b = 'mój ciąg tekstowy'
formatted = format(b, '^30s')
print('*', formatted, '*')


# Przykład 15.
key = 'my_var'
value = 1.234

formatted = '{} = {}'.format(key, value)
print(formatted)


# Przykład 16.
formatted = '{:<10} = {:.2f}'.format(key, value)
print(formatted)


# Przykład 17.
print('%.2f%%' % 12.5)
print('{} zastępuje {{}}'.format(1.23))


# Przykład 18.
formatted = '{1} = {0}'.format(key, value)
print(formatted)


# Przykład 19.
formatted = '{0} uwielbia jeść. Zobacz, jak {0} gotuje.'.format(name)
print(formatted)


# Przykład 20.
for i, (item, count) in enumerate(pantry):
    old_style = '#%d: %-10s = %d' % (
        i + 1,
        item.title(),
        round(count))

    new_style = '#{}: {:<10s} = {}'.format(
        i + 1,
        item.title(),
        round(count))

    assert old_style == new_style


# Przykład 21.
formatted = 'Pierwsza litera to {menu[oyster][0]!r}'.format(
    menu=menu)
print(formatted)


# Przykład 22.
old_template = (
    'Dzisiejsza zupa to %(soup)s, '
    'kup jedną i otrzymaj dwie ostrygi %(oyster)s, '
    'a także naszą przystawkę specjalną %(special)s.')
old_formatted = template % {
    'soup': 'pomidorowa',
    'oyster': 'kumamoto',
    'special': 'sznycel',
}

new_template = (
    'Dzisiejsza zupa to {soup}, '
    'kup jedną i otrzymaj dwie ostrygi {oyster}, '
    'a także naszą przystawkę specjalną {special}.')
new_formatted = new_template.format(
    soup='pomidorowa',
    oyster='kumamoto',
    special='sznycel',
)

assert old_formatted == new_formatted


# Przykład 23.
key = 'my_var'
value = 1.234

formatted = f'{key} = {value}'
print(formatted)


# Przykład 24.
formatted = f'{key!r:<10} = {value:.2f}'
print(formatted)


# Przykład 25.
f_string = f'{key:<10} = {value:.2f}'

c_tuple  = '%-10s = %.2f' % (key, value)

str_args = '{:<10} = {:.2f}'.format(key, value)

str_kw   = '{key:<10} = {value:.2f}'.format(key=key, value=value)

c_dict   = '%(key)-10s = %(value).2f' % {'key': key, 'value': value}

assert c_tuple == c_dict == f_string
assert str_args == str_kw == f_string


# Przykład 26.
for i, (item, count) in enumerate(pantry):
    old_style = '#%d: %-10s = %d' % (
        i + 1,
        item.title(),
        round(count))

    new_style = '#{}: {:<10s} = {}'.format(
        i + 1,
        item.title(),
        round(count))

    f_string = f'#{i+1}: {item.title():<10s} = {round(count)}'

    assert old_style == new_style == f_string


# Przykład 27.
for i, (item, count) in enumerate(pantry):
    print(f'#{i+1}: '
          f'{item.title():<10s} = '
          f'{round(count)}')


# Przykład 28.
places = 3
number = 1.23456
print(f'Moja liczba to {number:.{places}f}')
