/******************************************************************************* 
 * Copyright (c) 2005, 2007 Naci Dai, Lawrence Mandel, and Arthur Ryman. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * This sample developed for the book 
 *     Eclipse Web Tools Platform: Developing Java Web Applications
 * See http://eclipsewtp.org 
 *******************************************************************************/ 
/**
 * Query.java
 *
 * This class implements the query Web service.
 * 
 * @author Arthur Ryman
 */

package com.leagueplanet;

import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Set;
import java.util.Vector;

import com.leagueplanet.message.query.GetScheduleRequest;
import com.leagueplanet.model.Game;
import com.leagueplanet.model.League;
import com.leagueplanet.model.Location;
import com.leagueplanet.model.Schedule;
import com.leagueplanet.model.Score;
import com.leagueplanet.model.Team;
import com.leagueplanet.resource.schedule.GameType;
import com.leagueplanet.resource.schedule.LeagueResourceType;
import com.leagueplanet.resource.schedule.LocationResourceType;
import com.leagueplanet.resource.schedule.ScheduleContent;
import com.leagueplanet.resource.schedule.ScheduleType;
import com.leagueplanet.resource.schedule.ScoreType;
import com.leagueplanet.resource.schedule.TeamResourceType;
import com.leagueplanet.services.IceHockeyFacade;
import com.leagueplanet.services.LeagueFacade;

public class Query {

	// dla tej usugi stosujemy implementacj dla ligi hokejowej
	private LeagueFacade facade = IceHockeyFacade.getLeagueFacade();

	public ScheduleContent getSchedule(GetScheduleRequest request) {

		long scheduleId = request.getScheduleId();
		Schedule schedule = facade.findSchedule(scheduleId);

		ScheduleType scheduleType;
		if (schedule == null) {
			scheduleType = makeUnknownScheduleType(scheduleId);
		} else {
			scheduleType = makeScheduleType(schedule);
		}

		return new ScheduleContent(scheduleType);
	}

	private ScheduleType makeUnknownScheduleType(long scheduleId) {
		ScheduleType scheduleType = new ScheduleType();
		scheduleType.setScheduleId(scheduleId);
		scheduleType.setName("nieznany terminarz");
		scheduleType.setLeague(new LeagueResourceType("nieznana liga", 0));
		scheduleType.setGames(new GameType[0]);

		return scheduleType;
	}

	private ScheduleType makeScheduleType(Schedule schedule) {
		ScheduleType scheduleType = new ScheduleType();
		scheduleType.setScheduleId(schedule.getId());
		scheduleType.setName(schedule.getName());
		scheduleType.setLeague(makeResourceType(schedule.getLeague()));
		scheduleType.setGames(makeGamesType(schedule.getEvents()));

		return scheduleType;
	}

	private GameType[] makeGamesType(Set events) {
		Iterator eventIterator = events.iterator();
		Vector games = new Vector();
		while (eventIterator.hasNext()) {
			Object event = eventIterator.next();
			if (event instanceof Game) {
				Game game = (Game) event;
				GameType gameType = makeGameType(game);
				games.add(gameType);
			}
		}
		GameType[] gamesType = new GameType[games.size()];
		games.copyInto(gamesType);

		// sort the games by date
		Arrays.sort(gamesType, new Comparator() {
			public int compare(Object o1, Object o2) {
				long t1 = ((GameType) o1).getDateTime().getTimeInMillis();
				long t2 = ((GameType) o2).getDateTime().getTimeInMillis();
				if (t1 < t2)
					return -1;
				if (t1 > t2)
					return 1;
				return 0;
			}
		});

		return gamesType;
	}

	private GameType makeGameType(Game game) {
		GameType gameType = new GameType();
		gameType.setGameId(game.getId());
		gameType.setDateTime(game.toCalendar());
		gameType.setArena(makeResourceType(game.getLocation()));
		gameType.setVisitor(makeResourceType(game.getVisitor()));
		gameType.setHome(makeResourceType(game.getHome()));
		gameType.setScore(makeScoreType(game.getScore()));

		return gameType;
	}

	private ScoreType makeScoreType(Score score) {
		return new ScoreType(score.getVisitor(), score.getHome());
	}

	private TeamResourceType makeResourceType(Team team) {
		return new TeamResourceType(team.getName(), team.getId());
	}

	private LeagueResourceType makeResourceType(League league) {
		return new LeagueResourceType(league.getName(), league.getId());
	}

	private LocationResourceType makeResourceType(Location location) {
		return new LocationResourceType(location.getName(),
				//location.getTimeZoneId(),
				location.getId());
	}

}
