# ggplot2 w praktyce: tworzenie mapy eksmisji w San Francisco

# Wczytywanie pakietw do przeksztacania i wizualizowania danych
library("dplyr")
library("tidyr")

# W tym kodzie uywana jest okrelona ga z wersj rozwojow pakietu ggmap
library(devtools) # Na potrzeby instalowania pakietw z GitHuba.
devtools::install_github("dkahle/ggmap", ref = "tidyup")
library(ggmap)

# Wczytywanie pliku .csv z nakazami eksmisji.
# Dane pochodz ze strony https://catalog.data.gov/dataset/eviction-notices.
eviction_notices <- read.csv("data/Eviction_Notices.csv", stringsAsFactors = F)

# Przeksztacanie danych: formatowanie dat, filtrowanie nakazw (pozostawianie tylko nakazw z 2017 r.), 
# pobieranie danych o dugoci i szerokoci geograficznej.
notices <- eviction_notices %>%
  mutate(date = as.Date(File.Date, format="%m/%d/%y")) %>%
  filter(format(date, "%Y") == "2017") %>%
  separate(Location, c("lat", "long"), ", ") %>% # Podzia kolumny w miejscu wystpowania przecinkw.
  mutate(
    lat = as.numeric(gsub("\\(", "", lat)), # Usuwanie pocztkowego nawiasu.
    long = as.numeric(gsub("\\)", "", long)) # Usuwanie kocowego nawiasu.
  ) 

# Tworzenie mapy San Francisco z punktami okrelajcymi kady adres nakazu eksmisji.

# Rejestrowanie klucza API Map Google
# Zobacz: https://developers.google.com/maps/documentation/geocoding/get-api-key.
register_google(key="TWJ_KLUCZ_GOOGLE")

# Tworzenie widocznej w tle mapy z kafelek.
base_plot <- qmplot(
  data = notices,               # Nazwa ramki danych.
  x = long,                     # Atrybut reprezentujcy dugo geograficzn.
  y = lat,                      # Atrybut reprezentujcy szeroko geograficzn.
  geom = "blank",               # Nie naley (na razie) wywietla punktw danych.
  maptype = "toner-background", # Pobierane kafelki mapy.
  darken = .7,                  # Przyciemnianie kafelkw mapy.
  legend = "topleft"            # Lokalizacja legendy na stronie.
)

# Dodawanie lokalizacji eksmisji na mapie.
base_plot +
  geom_point(mapping = aes(x = long, y = lat), color = "red", alpha = .3) +
  labs(title = "Eksmisje w San Francisco, 2017 r.") +
  theme(plot.margin = margin(.3, 0, 0, 0, "cm")) # Dostosowanie przestrzeni wok mapy.

# Rysowanie mapy cieplnej wskanika eksmisji z wyznaczaniem konturw z uyciem pakietu ggplot2.
base_plot +
  geom_polygon(
    stat = "density2d", # Obliczanie gstoci punktw (konturw) na dwch wymiarach.
    mapping = aes(fill = stat(level)), # Uywanie obliczonej gstoci do ustawienia argumentu fill.
    alpha = .3 # Ustawianie kanau alpha (przezroczystoci).
  ) +
  scale_fill_gradient2("Liczba eksmisji", low = "white", mid = "yellow", high = "red") +
  labs(title="Liczba eksmisji w San Francisco, 2017 r.") +
  theme(plot.margin = margin(.3, 0, 0, 0, "cm"))
