/* ========================================================================
* JCommon : a free general purpose class library for the Java(tm) platform
* ========================================================================
*
* (C) Copyright 2000-2005, by Object Refinery Limited and Contributors.
*
...
*
*/

package org.jfree.date;

import static org.jfree.date.Month.FEBRUARY;

import java.util.*;

/**
 * Reprezentuje daty przy uyciu liczb integer, podobnie jak w
 * programie Microsoft Excel. Obsugowany zakres dat to od
 * 1 stycznia 1900 do 31 grudnia 9999.
 * <p/>
 * Trzeba zwrci uwag. e w Excelu wystpuje bd, powodujcy uznanie roku
 * 1900 za rok przestpny, cho nim nie jest. Wicej infromacji na ten temat
 * mona znale na witrynie Microsoft w artykule Q181370:
 * <p/>
 * http://support.microsoft.com/support/kb/articles/Q181/3/70.asp
 * <p/>
 * Excel korzysta z konwencji,w  ktrej 1 stycznia 1900 = 1. Ta klasa korzysta
 * z konwencji w ktrej 1 stycznia 1900 = 2.
 * W wyniku tego numer dnia w tej klasie jest inny ni w
 * Excelu dla stycznia i lutego 1900... ale wtedy Excel wprowadza dodatkowy dzie
 * (29 lutego 1900 ktry faktycznie nie istnia!) i od tego momentu
 * numery dni zgadzaj si.
 *
 * @author David Gilbert
*/
public class SpreadsheetDate extends DayDate {
  public static final int EARLIEST_DATE_ORDINAL = 2; // 1/1/1900
  public static final int LATEST_DATE_ORDINAL = 2958465; // 12/31/9999
  public static final int MINIMUM_YEAR_SUPPORTED = 1900;
  public static final int MAXIMUM_YEAR_SUPPORTED = 9999;
  static final int[] AGGREGATE_DAYS_TO_END_OF_PRECEDING_MONTH =
    {0, 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365};
  static final int[] LEAP_YEAR_AGGREGATE_DAYS_TO_END_OF_PRECEDING_MONTH =
    {0, 0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366};

  private int ordinalDay;
  private int day;
  private Month month;
  private int year;

  public SpreadsheetDate(int day, Month month, int year) {
    if (year < MINIMUM_YEAR_SUPPORTED || year > MAXIMUM_YEAR_SUPPORTED)
      throw new IllegalArgumentException(
        "Argument 'year' musi zawiera si w zakresie " +
         MINIMUM_YEAR_SUPPORTED + " do " + MAXIMUM_YEAR_SUPPORTED + ".");
     if (day < 1 || day > DateUtil.lastDayOfMonth(month, year))
       throw new IllegalArgumentException("Nieprawidowy argument 'day'.");

     this.year = year;
     this.month = month;
     this.day = day;
     ordinalDay = calcOrdinal(day, month, year);
     }

    public SpreadsheetDate(int day, int month, int year) {
      this(day, Month.fromInt(month), year);
    }

    public SpreadsheetDate(int serial) {
      if (serial < EARLIEST_DATE_ORDINAL || serial > LATEST_DATE_ORDINAL)
        throw new IllegalArgumentException(
          "SpreadsheetDate: warto 'serial' musi by z zakresu od 2 do 2958465.");

      ordinalDay = serial;
      calcDayMonthYear();
   }

    public int getOrdinalDay() {
      return ordinalDay;
    }

    public int getYear() {
      return year;
    }

    public Month getMonth() {
      return month;
    }

    public int getDayOfMonth() {
      return day;
    }

    protected Day getDayOfWeekForOrdinalZero() {return Day.SATURDAY;}

    public boolean equals(Object object) {
      if (!(object instanceof DayDate))
        return false;

      DayDate date = (DayDate) object;
      return date.getOrdinalDay() == getOrdinalDay();
    }

    public int hashCode() {
      return getOrdinalDay();
    }

    public int compareTo(Object other) {
      return daysSince((DayDate) other);
    }

    private int calcOrdinal(int day, Month month, int year) {
      int leapDaysForYear = DateUtil.leapYearCount(year - 1);
      int daysUpToYear = (year - MINIMUM_YEAR_SUPPORTED) * 365 + leapDaysForYear;
      int daysUpToMonth = AGGREGATE_DAYS_TO_END_OF_PRECEDING_MONTH[month.toInt()];
      if (DateUtil.isLeapYear(year) && month.toInt() > FEBRUARY.toInt())
        daysUpToMonth++;
      int daysInMonth = day - 1;
      return daysUpToYear + daysUpToMonth + daysInMonth + EARLIEST_DATE_ORDINAL;
    }

    private void calcDayMonthYear() {
      int days = ordinalDay - EARLIEST_DATE_ORDINAL;
      int overestimatedYear = MINIMUM_YEAR_SUPPORTED + days / 365;
      int nonleapdays = days - DateUtil.leapYearCount(overestimatedYear);
      int underestimatedYear = MINIMUM_YEAR_SUPPORTED + nonleapdays / 365;
  
      year = huntForYearContaining(ordinalDay, underestimatedYear);
      int firstOrdinalOfYear = firstOrdinalOfYear(year);
      month = huntForMonthContaining(ordinalDay, firstOrdinalOfYear);
      day = ordinalDay - firstOrdinalOfYear - daysBeforeThisMonth(month.toInt());
    }

    private Month huntForMonthContaining(int anOrdinal, int firstOrdinalOfYear) {
      int daysIntoThisYear = anOrdinal - firstOrdinalOfYear;
      int aMonth = 1;
      while (daysBeforeThisMonth(aMonth) < daysIntoThisYear)
        aMonth++;

      return Month.fromInt(aMonth - 1);
    }

    private int daysBeforeThisMonth(int aMonth) {
      if (DateUtil.isLeapYear(year))
        return LEAP_YEAR_AGGREGATE_DAYS_TO_END_OF_PRECEDING_MONTH[aMonth] - 1;
      else
        return AGGREGATE_DAYS_TO_END_OF_PRECEDING_MONTH[aMonth] - 1;
    }

    private int huntForYearContaining(int anOrdinalDay, int startingYear) {
      int aYear = startingYear;
      while (firstOrdinalOfYear(aYear) <= anOrdinalDay)
        aYear++;

      return aYear - 1;
    }

    private int firstOrdinalOfYear(int year) {
      return calcOrdinal(1, Month.JANUARY, year);
    }

    public static DayDate createInstance(Date date) {
      GregorianCalendar calendar = new GregorianCalendar();
      calendar.setTime(date);
      return new SpreadsheetDate(calendar.get(Calendar.DATE),
                                 Month.fromInt(calendar.get(Calendar.MONTH) + 1),
                                 calendar.get(Calendar.YEAR));

    }
}