"""Czysty kod w Pythonie - Rozdział 6: Deskryptory

> Jak Python używa wewnętrznie deskryptorów: @clasmethod

"""
from types import MethodType


class ClassMethod:
    def __init__(self, method):
        self.method = method

    def __call__(self, *args, **kwargs):
        return self.method(*args, **kwargs)

    def __get__(self, instance, owner):
        return MethodType(self.method, owner)


class MyClass:
    """
    >>> MyClass().class_method("pierwszy", "drugi")
    'MyClass wywołana z argumentami: pierwszy i drugi'

    >>> MyClass.class_method("jeden", "dwa")
    'MyClass wywołana z argumentami: jeden i dwa'

    >>> MyClass().method()  # doctest: +ELLIPSIS
    'MyClass wywołana z argumentami: self oraz z metody'
    """

    @ClassMethod
    def class_method(cls, arg1, arg2) -> str:
        return f"{cls.__name__} wywołana z argumentami: {arg1} i {arg2}"  # type: ignore

    def method(self):
        return self.class_method("self", "z metody")


class classproperty:
    def __init__(self, fget):
        self.fget = fget

    def __get__(self, instance, owner):
        return self.fget(owner)


def read_prefix_from_config():
    return ""


class TableEvent:
    """
    >>> TableEvent.topic
    'public.user'

    >>> TableEvent().topic
    'public.user'
    """

    schema = "public"
    table = "user"

    @classproperty
    def topic(cls):
        prefix = read_prefix_from_config()
        return f"{prefix}{cls.schema}.{cls.table}"
