"""Czysty kod w Pythonie - Rozdział 4: Zasady SOLID

Zasada podstawiania Liskov (LSP)

Wykrywanie naruszeń zasady LSP w metodach naruszających zdefiniowany kontrakt.

* Zdarzenia są zdefiniowane wg kontraktut (DbC, Design by Contract).
* Warunki kontraktu są egzekwowane tylko raz. Następnie
  ``SystemMonitor`` powinien mieć możliwość działania z dowolnym zdarzeniem wymiennie.

"""
from collections.abc import Mapping


class Event:
    def __init__(self, raw_data):
        self.raw_data = raw_data

    @staticmethod
    def meets_condition(event_data: dict) -> bool:
        return False

    @staticmethod
    def validate_precondition(event_data: dict):
        """Warunek wstępny kontraktu tego interfejsu.

        Weryfikuje, czy parametr ``event_data`` ma prawidłowy format.
        """
        if not isinstance(event_data, Mapping):
            raise ValueError(f"{event_data!r} nie jest słownikiem")
        for moment in ("before", "after"):
            if moment not in event_data:
                raise ValueError(f"słownik {event_data} nie zawiera klucza {moment}")
            if not isinstance(event_data[moment], Mapping):
                raise ValueError(f"wartość event_data[{moment!r}] nie jest słownikiem")


class UnknownEvent(Event):
    """Typ zdarzenia, którego nie można zidentyfikować na podstawie jego danych."""


class LoginEvent(Event):
    @staticmethod
    def meets_condition(event_data: dict) -> bool:
        return (
            event_data["before"].get("session") == 0
            and event_data["after"].get("session") == 1
        )


class LogoutEvent(Event):
    @staticmethod
    def meets_condition(event_data: dict) -> bool:
        return (
            event_data["before"].get("session") == 1
            and event_data["after"].get("session") == 0
        )


class TransactionEvent(Event):
    """Reprezentuje transakcję, która właśnie odbyła się w systemie."""

    @staticmethod
    def meets_condition(event_data: dict) -> bool:
        return event_data["after"].get("transaction") is not None


class SystemMonitor:
    """Identyfikuje zdarzenia, które wystąpiły w systemie.

    >>> l1 = SystemMonitor({"before": {"session": 0}, "after": {"session": 1}})
    >>> l1.identify_event().__class__.__name__
    'LoginEvent'

    >>> l2 = SystemMonitor({"before": {"session": 1}, "after": {"session": 0}})
    >>> l2.identify_event().__class__.__name__
    'LogoutEvent'

    >>> l3 = SystemMonitor({"before": {"session": 1}, "after": {"session": 1}})
    >>> l3.identify_event().__class__.__name__
    'UnknownEvent'

    >>> l4 = SystemMonitor({"before": {}, "after": {"transaction": "Tx001"}})
    >>> l4.identify_event().__class__.__name__
    'TransactionEvent'

    """

    def __init__(self, event_data):
        self.event_data = event_data

    def identify_event(self):
        Event.validate_precondition(self.event_data)
        event_cls = next(
            (
                event_cls
                for event_cls in Event.__subclasses__()
                if event_cls.meets_condition(self.event_data)
            ),
            UnknownEvent,
        )
        return event_cls(self.event_data)
