using System;
using System.Windows.Forms;

/// <summary>
/// Wersja klasy TpeakFunction ledzca zmiany pozycji suwaka.
/// </summary>
public abstract class TpeakFunction
{
    private double _tPeak;
    private TrackBar _slider;
    public event EventHandler Change;

    /// <summary>
    /// Tworzy funkcj aktualizujc swj stan i powiadamiajc
    /// zainteresowane obiekty o zmianach pozycji suwaka.
    /// </summary>
    /// <param name="tPeak">pocztkowa warto tPeak</param>
    /// <param name="slider">obserwowany suwak</param>
    public TpeakFunction(double tPeak, TrackBar slider)
    {
        Tpeak = tPeak;
        _slider = slider;
        slider.Scroll += new EventHandler(SliderScroll);
    }
    public double Tpeak 
    {
        get 
        {
            return _tPeak;
        }
        set
        {
            _tPeak = value;
            if (Change != null) 
            {
                Change(this, EventArgs.Empty);
            }
        }
    }
    private void SliderScroll(object sender, EventArgs e)
    {
        double val = _slider.Value;
        Tpeak = (val - _slider.Minimum) / (_slider.Maximum - _slider.Minimum);
    }
    public abstract double F(double t);
}
    /// <summary>
    /// Tempo spalania paliwa rakiety jest wyraone rwnaniem podanym
    /// w rozdziale Observer. Klasa stanowi rwnie przykad dostarczenia
    /// standardowej funkcji samego tylko czasu, cho pierwotne rwnanie
    /// zaley te od drugiego parametru tPeak. 
    /// Chcemy skorzysta z delegacji o nazwie Function, oczekujcej funkcji
    /// o sygnaturze z jednym argumentem typu double. Jednak funkcja tempa
    /// spalania jest zalena od dwch argumentw: czasu i wartoci tPeak.
    /// Dlatego te skadujemy tPeak jako zmienn instancji, do ktrej
    /// odwouje si funkcja tempa spalania. Aplikacja "ShowBallistics"
    /// ilustruje sposb tworzenia instancji delegacji za pomoc odwoania
    /// do obiektu.
    /// </summary>
public class BurnRate : TpeakFunction
{
    public BurnRate(double tPeak, TrackBar slider) : base(tPeak, slider)
    {
    }
    /// <summary>
    /// Tempo spalania jako funkcja czasu.
    /// </summary>
    /// <param name="t">Czas (przebiega od 0 do 1 w miar spalania paliwa)</param>
    /// <returns>Tempo spalania</returns>
    public override double F(double t) 
    {
        return F(t, Tpeak);
    }
    /// <summary>
    /// Tempo spalania jako funkcja czasu i tPeak.
    /// </summary>
    /// <param name="t">Czas</param>
    /// <param name="tPeak">tPeak (moment maksymalnej powierzchni spalania)</param>
    /// <returns>Tempo spalania</returns>
    public static double F(double t, double tPeak)
    {
        return .5 * Math.Pow(25, -Math.Pow((t - tPeak), 2));
    }
}
    /// <summary>
    /// Cig rakiety jest wyraony rwnaniem podanym w rozdziale Observer.
    /// </summary>
public class Thrust : TpeakFunction
{
    public Thrust(double tPeak, TrackBar slider) : base(tPeak, slider)
    {
    }
    /// <summary>
    /// Cig jako funkcja czasu.
    /// </summary>
    /// <param name="t">Czas (przebiega od 0 do 1 w miar spalania paliwa)</param>
    /// <returns>Cig</returns>
    public override double F(double t) 
    {
        return F(t, Tpeak);
    }
    /// <summary>
    /// Cig jako funkcja czasu i tPeak.
    /// </summary>
    /// <param name="t">Czas</param>
    /// <param name="tPeak">tPeak (moment maksymalnej powierzchni spalania)</param>
    /// <returns>Thrust</returns>
    public static double F(double t, double tPeak) 
    {
        return 1.7 * Math.Pow((BurnRate.F(t, tPeak) / .6), (1 / .3));
    }
}