using System;
using System.Drawing;
using System.Collections;

namespace Visualizations
{
    /// <summary>
    /// Model fabryki. W tej chwili zawiera on jedynie lokalizacje maszyn,
    /// ale obsuguje operacj cofnicia poprzez odtwarzanie zapamitanego
    /// przeszego stanu przy kadej zmianie konfiguracji fabryki.
    /// </summary>
    public class FactoryModel
    {
        private Stack _mementos;   
        public event AddHandler AddEvent; 
        public event DragHandler DragEvent; 
        public event RebuildHandler RebuildEvent; 
        public static readonly Point DEFAULT_LOCATION = new Point(10, 10);

        /// <summary>
        /// Tworzy nowy model fabryki.
        /// </summary>
        public FactoryModel()
        {
            // na pocztku lista lokalizacji maszyn jest pusta
            _mementos = new Stack();
            _mementos.Push(new ArrayList());
        }
        /// <summary>
        /// Dodaje maszyn do modelu. Maszyna jest ustawiana w lokalizacji
        /// domylnej.
        /// </summary>
        public void AddMachine()
        {
            // umieszcza now maszyn przed pozostaymi (najpycej)
            IList newLocs = new ArrayList();
            Point newP = DEFAULT_LOCATION;
            newLocs.Add(newP);
            foreach (Point p in (IList)_mementos.Peek())
            {
                newLocs.Add(new Point(p.X, p.Y));
            }
            _mementos.Push(newLocs);
            if (AddEvent != null) AddEvent(newP);
        }

        /// <summary>
        /// Przesuwa maszyn z jednego miejsca na drugie. Skutkiem ubocznym
        /// tej operacji jest to, e nowa lokalizacja zostanie umieszczona
        /// na pierwszym miejscu listy lokalizacji biecego modelu, co pomaga
        /// aplikacji ustala wzajemn gboko ikon. Rwnie kliknicie maszyny
        /// spowoduje umieszczenie jej na pocztku listy, a tym samym na przedzie
        /// panelu symulacji.
        /// </summary>
        /// <param name="oldP">stara lokalizacja maszyny</param>
        /// <param name="newP">nowa lokalizacja maszyny</param>
        public void Drag(Point oldP, Point newP) 
        {
            // umieszcza now lokalizacj na pocztku listy
            IList newLocs = new ArrayList(); 
            newLocs.Add(newP);
            // tworzy now list, kopiujc wszystkie maszyny poza przecigan
            bool foundDragee = false;
            foreach (Point p in (IList)_mementos.Peek())
            {
                if ( !foundDragee && p.Equals(oldP)) 
                {
                    foundDragee = true;
                }
                else
                {
                    newLocs.Add(new Point(p.X, p.Y)); 
                }
            }
            _mementos.Push(newLocs);
            if (DragEvent != null) DragEvent(oldP, newP);
        }
 
        /// <summary>
        /// Zmienia model fabryki na poprzedni wersj.
        /// </summary>
        public void Pop()
        {
            if (_mementos.Count > 1) 
            {
                _mementos.Pop(); // zdjcie biecego stanu ze stosu
                if (RebuildEvent != null) RebuildEvent();
            }
        }
        /// <summary>
        /// Dodaje now konfiguracj.
        /// </summary>
        /// <param name="list">lista punktw (obiektw Point) odpowiadajcych lokalizacjom maszyn</param>
        public void Push(IList list)
        {
            _mementos.Push(list);
            if (RebuildEvent != null) RebuildEvent();
        }
        /// <summary>
        /// Zwraca biecy zbir lokalizacji maszyn.
        /// </summary>
        public IList Locations
        {
            get
            {
                return (IList)_mementos.Peek();
            }
        }

        /// <summary>
        /// Zwraca liczb zapisanych konfiguracji. Pomaga to aplikacji
        /// ustali, czy ma oferowa funkcj cofnicia.
        /// </summary>
        public int MementoCount
        {
            get 
            {
                return _mementos.Count;
            }
        }

    }
}