using System;
using System.Text;
namespace Filters
{
    /// <summary>
    /// Obiekt WrapFilter usuwa nadmiarowe biae znaki i przenosi tekst
    /// we wskazanym miejscu, z dodatkow moliwoci wyrodkowania.
    /// </summary>
    public class WrapFilter : OozinozFilter 
    {
        protected int _width;
        protected StringBuilder lineBuf = new StringBuilder();
        protected StringBuilder wordBuf = new StringBuilder();
        protected bool _center = false;
        protected bool _inWhite = false;
        protected bool _needBlank = false;

        /// <summary>
        /// Konstrukcja filtra przenoszcego pisany tekst we wskazanym miejscu.
        /// </summary>
        /// <param name="writer">filtr docelowy</param>
        /// <param name="width">szeroko przeniesienia</param>
        public WrapFilter(ISimpleWriter writer, int width) : base (writer)
        {
            this._width = width;
        }
        /// <summary>
        /// Znacznik okrelajcy, czy tekst ma by wyrodkowany.
        /// </summary>
        public bool Center
        {
            get 
            {
                return _center;
            }
            set
            {
                _center = value;
            }
        }
        /// <summary>
        /// Oprnienie i zamknicie strumienia.
        /// </summary>
        public override void Close() 
        {
            Flush();
            base.Close();
        }
        /// <summary>
        /// Wypisuje wszelkie znaki przetrzymywane w buforze w oczekiwaniu 
        /// na peen wiersz.
        /// </summary>
        public void Flush()
        {
            if (wordBuf.Length > 0)
            {
                PostWord();
            }
            if (lineBuf.Length > 0)
            {
                PostLine();
            }
        }
        /// <summary>
        /// Wypisuje wszelkie znaki w buforze wiersza, w razie potrzeby
        /// wyrodkowujc kady wiersz.
        /// </summary>
        protected void PostLine()  
        {
            if (Center)
            {
                for (int i = 0; i < (_width - lineBuf.Length) / 2; i++)
                {
                    _writer.Write(' ');
                }
            }
            _writer.Write(lineBuf.ToString());
            _writer.WriteLine();
        }
        /// <summary>
        /// Dopisuje bufor sowa do buforu wiersza, chyba e przez to wiersz
        /// staby si zbyt dugi. W takim przypadku wypisuje bufor wiersza
        /// i zastpuje jego zawarto treci bufora sowa.
        /// </summary>
        protected void PostWord()  
        {
            if (lineBuf.Length + 1 + wordBuf.Length > _width && (lineBuf.Length > 0))
            {
                PostLine();
                lineBuf = wordBuf;
                wordBuf = new StringBuilder();
            }
            else
            {
                if (_needBlank)
                {
                    lineBuf.Append(" ");
                }
                lineBuf.Append(wordBuf);
                _needBlank = true;
                wordBuf = new StringBuilder();
            }
        }
        /// <summary>
        /// Dopisuje przekazany znak do biecego bufora sowa, chyba e jest
        /// to znak biay, oznaczajcy koniec sowa. Napotkanie znaku biaego
        /// powoduje dopisanie sowa do bufora wiersza.
        /// </summary>
        /// <param name="c">zapisywany znak</param>
        public override void Write(char c)  
        {
            if (Char.IsWhiteSpace(c))
            {
                if (!_inWhite)
                {
                    PostWord();
                }
                _inWhite = true;
            }
            else
            {
                wordBuf.Append(c);
                _inWhite = false;
            }
        }
    }
}
