using System;

   /// <summary>
    /// Klasa abstrakcyjna definiujca funkcj zalen nie tylko od czasu,
    /// ale rwnie wartoci tPeak, czyli momentu, w ktrym wystpuje
    /// szczytowa powierzchnia spalania i tempo spalania paliwa staego 
    /// dla rakiety.
    /// </summary>
public abstract class TpeakFunction
{
    private double _tPeak;

    /// <summary>
    /// Tworzy funkcj zalen od wartoci tPeak.
    /// </summary>
    /// <param name="tPeak">warto pocztkowa</param>
    public TpeakFunction(double tPeak)
    {
        Tpeak = tPeak;
    }
    /// <summary>
    /// Warto tPeak, czyli moment wystpowania maksymalnej powierzchni
    /// i tempa spalania.
    /// </summary>
    public double Tpeak 
    {
        get 
        {
            return _tPeak;
        }
        set
        {
            _tPeak = value;
        }
    }
    /// <summary>
    /// Funkcja implementowana przez podklasy konkretne.
    /// </summary>
    /// <param name="t">parametr czasu</param>
    /// <returns>warto funkcji</returns>
    public abstract double F(double t);
}
    /// <summary>
    /// Tempo spalania paliwa rakiety jest wyraone rwnaniem chemicznym
    /// (z rozdziau Observer). Klasa stanowi rwnie przykad dostarczenia
    /// standardowej funkcji samego tylko czasu, cho pierwotne rwnanie
    /// zaley te od drugiego parametru tPeak. 
    /// Chcemy skorzysta z delegacji o nazwie Function, oczekujcej funkcji
    /// o sygnaturze z jednym argumentem typu double. Jednak funkcja tempa
    /// spalania jest zalena od dwch argumentw: czasu i wartoci tPeak.
    /// Dlatego te skadujemy tPeak jako zmienn instancji, do ktrej
    /// odwouje si funkcja tempa spalania. Aplikacja "ShowBallistics"
    /// ilustruje sposb tworzenia instancji delegacji za pomoc odwoania
    /// do obiektu.
    /// </summary>
public class BurnRate : TpeakFunction
{
    /// <summary>
    /// Tworzy obiekt tempa spalania.
    /// </summary>
    /// <param name="tPeak">warto pocztkowa</param>
    public BurnRate(double tPeak) : base(tPeak)
    {
    }
    /// <summary>
    /// Tempo spalania jako funkcja czasu.
    /// </summary>
    /// <param name="t">Czas (przebiega od 0 do 1 w miar spalania paliwa)</param>
    /// <returns>Tempo spalania</returns>
    public override double F(double t) 
    {
        return F(t, Tpeak);
    }
    /// <summary>
    /// Tempo spalania jako funkcja czasu i tPeak.
    /// </summary>
    /// <param name="t">Czas</param>
    /// <param name="tPeak">tPeak (moment maksymalnej powierzchni spalania)</param>
    /// <returns>Tempo spalania</returns>
    public static double F(double t, double tPeak)
    {
        return .5 * Math.Pow(25, -Math.Pow((t - tPeak), 2));
    }
}
    /// <summary>
    /// Cig rakiety jest wyraony rwnaniem chemicznym (z rozdziau
    /// Observer).
    /// </summary>
public class Thrust : TpeakFunction
{
    /// <summary>
    /// Tworzy obiekt cigu.
    /// </summary>
    /// <param name="tPeak">warto pocztkowa</param>
    public Thrust(double tPeak) : base(tPeak)
    {
    }
    /// <summary>
    /// Cig jako funkcja czasu.
    /// </summary>
    /// <param name="t">Czas (przebiega od 0 do 1 w miar spalania paliwa)</param>
    /// <returns>Cig</returns>
    public override double F(double t) 
    {
        return F(t, Tpeak);
    }
    /// <summary>
    /// Cig jako funkcja czasu i tPeak.
    /// </summary>
    /// <param name="t">Czas</param>
    /// <param name="tPeak">tPeak (moment maksymalnej powierzchni spalania)</param>
    /// <returns>Thrust</returns>
    public static double F(double t, double tPeak) 
    {
        return 1.7 * Math.Pow((BurnRate.F(t, tPeak) / .6), (1 / .3));
    }
}