using System;
using System.Text.RegularExpressions;

namespace Reservations
{
    /// <summary>
    /// Klasa przetwarzajca tekst zgoszenia pokazu fajerwerkw. Zgoszenie
    /// musi by rozdzielan przecinkami list nazw i wartoci parametrw.
    /// Oczekiwane parametry dla pokazu fajerwerkw to: data, ilo widzw,
    /// miasto, cena od osoby i konkretna lokalizacja. Oto przykad poprawnego
    /// zgoszenia:
    /// 
    /// <blockquote><pre>
    ///   Date, November 5, Headcount, 250, City, Springfield,
    ///   DollarsPerHead, 9.95, HasSite, No      	   
    ///</pre></blockquote>
    ///
    /// Format daty to nazwa miesica i dzie. Parser przyjmuje, e rok 
    /// jest rokiem najbliszego wystpienia podanej daty.
    /// </summary>
    public class ReservationParser 
    {
        private ReservationBuilder _builder;
        /// <summary>
        /// Parsuje zgoszenie rezerwacji i przekazuje pobrane dane do obiektu
        /// budujcego rezerwacj.
        /// </summary>
        /// <param name="s">tekst zgoszenia</param>
        public void Parse(String s)
        {         
            string[] tokens = new Regex(@",\s*").Split(s);
            for (int i = 0; i < tokens.Length; i += 2 ) 
            {
                String type = tokens[i];
                String val = tokens[i + 1];

                if (String.Compare("date", type, true) == 0)
                { 
                    DateTime d = DateTime.Parse(val);
                    _builder.Date = ReservationBuilder.Futurize(d);
                }
                else if (String.Compare("headcount", type, true) == 0)
                {
                    _builder.Headcount = Int32.Parse(val);
                }
                else if (String.Compare("City", type, true) == 0)
                {
                    _builder.City = val.Trim();
                }
                else if (String.Compare("DollarsPerHead", type, true) == 0)
                {
                    _builder.DollarsPerHead = (decimal)Double.Parse(val);
                }
                else if (String.Compare("HasSite", type, true) == 0)
                {
                    _builder.HasSite = Boolean.Parse(val);
                     
                } 
            }
        }
        /// <summary>
        /// Tworzy parser przekazujcy pobrane dane do wskazanego obiektu
        /// budujcego rezerwacj.
        /// </summary>
        /// <param name="builder">docelowy obiekt budujcy</param>
        public ReservationParser(ReservationBuilder builder) 
        {
            _builder = builder;
        }
    }
}
