using System;
using System.Data;
using System.Data.OleDb;
using System.Collections;
using System.Reflection;
using Utilities;

namespace DataLayer
{
    /// <summary>
    /// Delegacja definiujca interfejs dla metod otrzymujcych (poyczajcych)
    /// obiekt czytajcy z bazy danych. Obiekt wypoyczajcy moe wywoywa
    /// delegacj, tworzc obiekt czytajcy przed wywoaniem i zwalniajc zasoby
    /// tego obiektu po zakoczeniu wywoania.
    /// </summary>
    public delegate object BorrowReader(IDataReader reader);
    /// <summary>
    /// Dostarcza podstawowe usugi dostpu do bazy danych Oozinoz.
    /// </summary>
    public class DataServices
    {
        /// <summary>
        /// Utworzenie obiektu czytajcego na podstawie podanego zapytania SQL,
        /// a nastpnie wykonanie dostarczonej delegacji (metody poyczajcej
        /// obiekt czytajcy) i zamknicie obiektu czytajcego.
        /// </summary>
        /// <param name="sql">Wykonywanie zapytanie SQL.</param>
        /// <param name="method">Wywoywana metoda korzystajca z obiektu czytajcego.</param>
        /// <returns>Warto zwracana przez dostarczon metod.</returns>
        public static object LendReader(string sql, BorrowReader borrower) 
        {
            using (OleDbConnection conn = CreateConnection())
            {
                conn.Open(); 
                OleDbCommand c = new OleDbCommand(sql, conn);  
                OleDbDataReader r = c.ExecuteReader();
                return borrower(r);
            }     
        }
        /// <summary>
        /// Utworzenie i zwrcenie poczenia z baz danych Access Oozinoz.
        /// </summary>
        /// <returns>poczenie</returns>
        public static OleDbConnection CreateConnection()
        { 
            String dbName = FileFinder.GetFileName("db", "oozinoz.mdb");
            OleDbConnection c = new OleDbConnection();
            c.ConnectionString = 
                "Provider=Microsoft.Jet.OLEDB.4.0;" +
                "Data Source=" + dbName;
            return c;
        }
        /// <summary>
        /// Utworzenie i zwrcenie adaptera bazy danych dla dostarczonego
        /// zapytania SQL.
        /// </summary>
        /// <param name="select">Wykonywane zapytanie SQL</param>
        /// <returns>Adapter</returns>
        public static OleDbDataAdapter CreateAdapter(string select) 
        {   
            return new OleDbDataAdapter(select, CreateConnection());  
        }        
        /// <summary>
        /// Utworzenie i zwrcenie obiektu DataTable, skadujcego wyniki
        /// zwrcone przez podane zapytanie SQL.
        /// </summary>
        /// <param name="select">Zapytanie SQL</param>
        /// <returns>Obiekt DataTable</returns>
        public static DataTable CreateTable(string select) 
        {   
            return (DataTable) LendReader(select, new BorrowReader(CreateTable));
        } 
        // Tworzy obiekt DataTable z zadanego obieku czytajcego
        internal static object CreateTable(IDataReader reader) 
        {
            DataTable table = new DataTable(); 
            for (int i = 0; i < reader.FieldCount; i++) 
            { 
                table.Columns.Add(reader.GetName(i), reader.GetFieldType(i));
            }   
            while (reader.Read()) 
            {
                DataRow dr = table.NewRow();
                for (int i = 0; i < reader.FieldCount; i++) 
                {
                    dr[i] = reader.GetValue(i);
                }     
                table.Rows.Add(dr);
            }
            return table;
        }
        /// <summary>
        /// Zwraca instancj podanego typu, wypenion danymi z bazy Oozinoz.
        /// </summary>
        /// <param name="t">Typ tworzonego obiektu</param>
        /// <param name="name">Nazwa obiektu szukanego w bazie</param>
        /// <returns>Gotowy obiekt</returns>
        public static Object Find(Type t, string name)         
        {
            string sel = "SELECT * FROM " + t.Name + " WHERE NAME = '" + name + "'";
            return LendReader(sel, new BorrowReader(new ObjectLoader(t).LoadObject));
        } 
        /// <summary>
        /// Wyszukuje w bazie wszystkie obiekty danego typu.
        /// </summary>
        /// <param name="t">Typ tworzonego obiektu</param>
        /// <returns>List obiektw odpowiadajcych rekordom w tabeli w bazie,
        /// dla ktrych nazwa odpowiada podanemu typowi.
        /// </returns>
        public static IList FindAll(Type t)         
        {
            string sel = "SELECT * FROM " + t.Name;
            return (IList) LendReader(sel, new BorrowReader(new ObjectLoader(t).LoadAll));
        } 
        //
        // Instancje tej klasy skaduj typ obiektu do utworzenia 
        // i zapenienia danymi z obiektu czytajcego z bazy
        //
        internal class ObjectLoader
        {
            private Type _type;
            //
            // Tworzy obiekt ObjectLoader dla danego typu.
            //
            public ObjectLoader(Type t)
            {
                this._type = t;
            }
            //
            // Odczytanie nastpnego rekordu z obiektu czytajcego i wczytanie
            // danych do pojedynczego obiektu.
            //
            internal Object LoadObject(IDataReader reader)
            {
                if (reader.Read()) 
                { 
                    return LoadFromCurrent(reader);
                }
                return null;
            }
            //
            // Tworzy list obiektw odpowiadajcych kolejnym rekordom
            // w bazie danych.
            //
            internal Object LoadAll(IDataReader reader)
            {
                ArrayList list = new ArrayList();
                while (reader.Read()) 
                {
                    list.Add(LoadFromCurrent(reader));
                }
                return list;
            }
            //
            // Tworzy i aduje pojedynczy obiekt z biecego rekordu 
            // w dostarczonym obiekcie czytajcym.
            //
            internal Object LoadFromCurrent(IDataReader reader) 
            {
                ConstructorInfo c = _type.GetConstructor(new Type[]{});
                Object o = c.Invoke(new Object[]{});
                foreach (PropertyInfo p in _type.GetProperties()) 
                {
                    MethodInfo m = p.GetSetMethod();
                    try 
                    {
                        m.Invoke(o, new Object[]{reader[p.Name]});
                    }
                    catch (System.IndexOutOfRangeException) {}
                }
                return o;
            }
        }
    }
}
