// Kontener implementujcy  
// tablic dynamiczn o zmiennych indeksach 
// 
// Ten plik nazwij ra.h 
// 
#include <iostream> 
#include <iterator> 
#include <algorithm> 
#include <cstdlib> 
#include <stdexcept> 
 
using namespace std; 
 
// Klasa obsugujca wyjtki dla kontenera RangeArray. 
class RAExc { 
  string err; 
public: 
 
  RAExc(string e) { 
    err = e; 
  } 
 
  string geterr() { return err; } 
}; 
 
// Kontener tablicy dynamicznej o zmiennych indeksach. 
template<class T, class Allocator = allocator<T> > 
class RangeArray { 
  T *arrayptr; // wskanik tablicy bdcej podstaw kontenera 
 
  unsigned len;   // przechowuje dugo kontenera 
  int upperbound; // dolny kraniec
  int lowerbound; // grny kraniec 
 
  Allocator a; // alokator 
public: 
  // Typy wymagane dla kontenera. 
  typedef T value_type; 
  typedef Allocator allocator_type; 
  typedef typename Allocator::reference reference; 
  typedef typename Allocator::const_reference const_reference; 
  typedef typename Allocator::size_type size_type; 
  typedef typename Allocator::difference_type difference_type; 
  typedef typename Allocator::pointer pointer; 
  typedef typename Allocator::const_pointer const_pointer; 
 
  // Iteratory postpowe.
  typedef T * iterator; 
  typedef const T * const_iterator; 
 
  // Uwaga: Ten kontener nie obsuguje iteratorw wstecznych, 
  // lecz moesz doda ich implementacj samodzielnie .  
 
 
  // *****  Konstruktory i destruktory ***** 
 
  // Konstruktor domylny. 
  RangeArray() 
  { 
    upperbound = lowerbound = 0; 
    len = 0; 
    arrayptr = a.allocate(0); 
  } 
 
  // Konstruktor tablicy o okrelonym zakresie indeksw
  // inicjalizujcy kady element okrelon wartoci pocztkow.
  RangeArray(int low, int high, const T &t); 
 
  // Konstruktor tablicy indeksowanej od 0
  // posiadajcej num elementw o wartoci t.
  // Konstruktor ten wymagany jest dla zgodnoci z bibliotek STL. 
  RangeArray(int num, const T &t=T()); 
 
  // Konstruktor tworzcy tablic indeksowan od zera
  // na podstawie zakresu iteratorw.
  // Konstruktor ten wymagany jest dla zgodnoci z bibliotek STL. 
  RangeArray(iterator start, iterator stop); 
 
  // Konstruktor kopiujcy. 
  RangeArray(const RangeArray &o); 
   
  // Destruktor. 
  ~RangeArray(); 
 
 
  // *****  Funkcje operatorowe ***** 
 
  // Zwr referencj okrelonego elementu. 
  T &operator[](int i) 
  { 
    return arrayptr[i - lowerbound]; 
  } 
 
  // Zwr referencj const do okrelonego elementu. 
  const T &operator[](int i) const 
  { 
    return arrayptr[i - lowerbound]; 
  } 
   
  // Przypisz kontener innemu kontenerowi. 
  RangeArray &operator=(const RangeArray &o); 
 
 
  // *****  Funkcje wstawiania elementw ***** 
 
  // Wstaw warto val w miejscu okrelonym przez p. 
  iterator insert(iterator p, const T &val); 
 
  // Wstaw num kopii wartoci val w miejscu okrelonym przez p. 
  void insert(iterator p, int num, const T &val) 
  { 
    for(; num>0; num--) p = insert(p, val) + 1; 
  } 
 
  // Wstaw zakres elementw okrelonych przez parametry start i stop
  // w miejscu okrelonym przez p.
  void insert(iterator p, iterator start, iterator stop) 
  { 
    while(start != stop) { 
      p = insert(p, *start) + 1; 
      start++; 
    } 
  } 
 
 
  // *****  Funkcje usuwajce ***** 
 
  // Usu element w miejscu okrelonym przez p.
  iterator erase(iterator p); 
 
  // Usu elementy z okrelonego zakresu. 
  iterator erase(iterator start, iterator stop) 
  { 
    iterator p = end(); 
 
    for(int i=stop-start; i > 0; i--) 
      p = erase(start); 
 
    return p; 
  } 
 
 
  // *****  Funkcje do doczania i pobierania elementw *****  
 
  // Dodaj element na kocu. 
  void push_back(const T &val) 
  { 
    insert(end(), val); 
  } 
 
  // Usu element z koca. 
  void pop_back() 
  { 
    erase(end()-1); 
  } 
 
  // Dodaj element na pocztku. 
  void push_front(const T &val) 
  { 
    insert(begin(), val); 
  } 
 
  // Usu element z pocztku. 
  void pop_front() 
  { 
    erase(begin()); 
  } 
 
 
  // *****  Funkcje front() i back() ***** 
 
  // Zwr referencj do pierwszego elementu. 
  T &front() 
  { 
    return arrayptr[0]; 
  } 
 
  // Zwr referencj const do pierwszego elementu. 
  const T &front() const 
  { 
    return arrayptr[0]; 
  } 
 
  // Zwr referencj do ostatniego elementu. 
  T &back() 
  { 
    return arrayptr[len-1]; 
  } 
 
  // Zwr referencj const do ostatniego elementu.
  const T &back() const 
  { 
    return arrayptr[len-1]; 
  } 
   
 
  // *****  Funkcje iterujce ***** 
 
  // Zwr iterator pierwszego elementu. 
  iterator begin() 
  { 
    return &arrayptr[0]; 
  } 
  
  // Zwr iterator ostatniego elementu.
  iterator end() 
  { 
    return &arrayptr[upperbound - lowerbound]; 
  } 
 
  // Zwr iterator const pierwszego elementu. 
  const_iterator begin() const 
  { 
    return &arrayptr[0]; 
  } 
  
  // Zwr iterator const ostatniego elementu. 
  const_iterator end() const 
  { 
    return &arrayptr[upperbound - lowerbound]; 
  } 
 
 
  // *****  Funkcje specjalne ***** 
 
  // Metoda at() sprawdza zakres indeksw
  // i zwraca referencj okrelonego elementu.
  T &at(int i) 
  { 
    if(i < lowerbound || i >= upperbound) 
      throw out_of_range("Indeks poza dozwolonym zakresem"); 
 
    return arrayptr[i - lowerbound]; 
  } 
 
  // Zwr referencj const okrelonego elementu.
  const T &at(int i) const 
  { 
    if(i < lowerbound || i >= upperbound) 
      throw out_of_range("Indeks poza dozwolonym zakresem"); 
 
    return arrayptr[i - lowerbound]; 
  }     
 
  // Zwr rozmiar kontenera.
  size_type size() const 
  {  
    return end() - begin(); 
  } 
 
  // Zwr maksymalny rozmiar tablicy RangeArray. 
  size_type max_size() 
  { 
    return a.max_size(); 
  } 
 
  // Zwr warto true, jeeli kontener jest pusty.
  bool empty() 
  {  
    return size() == 0; 
  } 
 
  // Zamie wartoci dwch kontenerw.
  void swap(RangeArray &b) 
  { 
    RangeArray<T> tmp; 
 
    tmp = *this; 
    *this = b; 
    b = tmp; 
  } 
 
  // Usu wszystkie elementy i oczy pami. 
  void clear() 
  { 
    erase(begin(), end()); 
  } 
 
  // ***** Funkcje nie bdce funkcjami STL ***** 
 
  // Zwr krace zakresu indeksw. 
  int getlowerbound() 
  { 
    return lowerbound; 
  } 
 
  int getupperbound() 
  { 
    return upperbound; 
  } 
 
}; 
 
 
// ***** Implementacja funkcji nie bdcych funkcjami inline ***** 
 
// Skonstruuj tablic o okrelonym zakresie indeksw
// i nadaj kademu elementowi okrelon warto pocztkow. 
template <class T, class A>  
RangeArray<T, A>::RangeArray(int low, int high, 
                             const T &t) 
{ 
  if(high <= low) throw RAExc("Nieprawidowy zakres"); 
 
  high++;  
 
  // Zapisz krace. 
  upperbound = high; 
  lowerbound = low; 
 
  // Alokuj pami na kontener. 
  arrayptr = a.allocate(high - low); 
     
  // Zapisz dugo kontenera. 
  len = high - low; 
 
  // Twrz elementy. 
  for(size_type i=0; i < size(); i++) 
    a.construct(&arrayptr[i], t); 
} 
 
// Twrz tablic indeksowan od zera zawierajc num elementw o wartoci t. 
// Konstruktor ten wymagany jest dla zgodnoci z bibliotek STL.
template <class T, class A>  
RangeArray<T, A>::RangeArray(int num, const T &t) { 
 
  // Zapisz krace. 
  upperbound = num; 
  lowerbound = 0; 
 
  // Alokuj pami na kontener. 
  arrayptr = a.allocate(num); 
     
  // Zapisz dugo kontenera.
  len = num; 
 
  // Twrz elementy. 
  for(size_type i=0; i < size(); i++) 
    a.construct(&arrayptr[i], t); 
 
} 
 
  // Konstruktor tworzcy tablic indeksowan od zera
  // na podstawie zakresu iteratorw.
  // Konstruktor ten wymagany jest dla zgodnoci z bibliotek STL. 
template <class T, class A>  
RangeArray<T, A>::RangeArray(iterator start, 
                             iterator stop) 
{ 
  // Alokuj pami potrzebn na elementy. 
  arrayptr = a.allocate(stop - start); 
 
  upperbound = stop - start; 
  lowerbound = 0; 
 
  len = stop - start; 
 
  // Twrz elementy w oparciu o zakres
  // okrelony iteratorami. 
    for(size_type i=0; i < size(); i++) 
    a.construct(&arrayptr[i], *start++); 
} 
 
// Konstruktor kopiujcy. 
template <class T, class A>  
RangeArray<T, A>::RangeArray(const RangeArray<T, A> &o) 
{ 
  // Alokuj pami na kopi. 
  arrayptr = a.allocate(o.size()); 
 
  upperbound = o.upperbound; 
  lowerbound = o.lowerbound; 
  len = o.len; 
 
  // Twrz kopi. 
  for(size_type i=0; i < size(); i++) 
    a.construct(&arrayptr[i], o.arrayptr[i]); 
} 
 
// Destruktor. 
template <class T, class A> 
RangeArray<T, A>::~RangeArray() 
{ 
  // Wywoaj destruktory elementw kontenera.
  for(size_type i=0; i < size(); i++) 
    a.destroy(&arrayptr[i]); 
 
  // Zwolnij pami. 
  a.deallocate(arrayptr, size()); 
} 
 
  // Przypisz kontener innemu kontenerowi. 
template <class T, class A> RangeArray<T, A> &  
RangeArray<T, A>::operator=(const RangeArray<T, A> &o) 
{ 
  // Wywoaj destruktory elementw w docelowym kontenerze. 
  for(size_type i=0; i < size(); i++) 
    a.destroy(&arrayptr[i]); 
 
  // Zwolnij pierwotny obszar pamici. 
  a.deallocate(arrayptr, size());  
 
  // Alokuj pami dla nowego rozmiaru. 
  arrayptr = a.allocate(o.size()); 
 
  upperbound = o.upperbound; 
  lowerbound = o.lowerbound; 
  len = o.len; 
 
  // Utwrz kopi. 
  for(size_type i=0; i < size(); i++) 
    arrayptr[i] = o.arrayptr[i]; 
 
  return *this; 
} 
 

 // Wstaw warto val w miejscu okrelonym przez p. 
template <class T, class A> 
typename RangeArray<T, A>::iterator  
RangeArray<T, A>::insert(iterator p, const T &val) 
{ 
  iterator q; 
  size_type i, j; 
 
  // Alokuj odpowiedni ilo pamici. 
  T *tmp = a.allocate(size() + 1); 
 
  // Kopiuj istniejce elementy do nowej tablicy, 
  // wstawiajc nowy elementy, jeli to moliwe. 
  for(i=j=0; i < size(); i++, j++) { 
    if(&arrayptr[i] == p) {  
      tmp[j] = val; 
      q = &tmp[j]; 
      j++; 
    } 
    tmp[j] = arrayptr[i]; 
  } 
 
  // W przeciwnym przypadku nowy element zostanie wstawiony na kocu. 
  if(p == end()) { 
    tmp[j] = val; 
    q = &tmp[j]; 
  } 
 
  // Modyfikuj dugo i krace. 
  len++; 
  if(p < &arrayptr[abs(lowerbound)]) 
    lowerbound--; 
  else  
    upperbound++; 
 
  // Wywoaj destruktory elementw w starym kontenerze.
  for(size_type i=0; i < size()-1; i++) 
    a.destroy(&arrayptr[i]); 
 
  // Zwolnij pami zajmowan przez stary kontener. 
  a.deallocate(arrayptr, size()-1); 
 
  arrayptr = tmp; 
 
  return q; 
} 
 
// Usu element w miejscu okrelonym przez p.
template <class T, class A> 
typename RangeArray<T, A>::iterator  
RangeArray<T, A>::erase(iterator p) 
{     
  iterator q = p; 
 
  // Zniszcz usuwany element.  
  if(p != end()) a.destroy(p); 
 
  // Modyfikuj dugo i krace. 
  len--; 
  if(p < &arrayptr[abs(lowerbound)])  
    lowerbound++; 
  else  
    upperbound--; 
 
  // Przesu pozostae elementy. 
  for( ; p < end(); p++) 
    *p = *(p+1); 
 
  return q; 
} 
 
 
// ********  Operatory relacyjne ************** 
 
template<class T, class Allocator> 
  bool operator==(const RangeArray<T, Allocator> &a, 
                  const RangeArray<T, Allocator> &b) 
{ 
  if(a.size() != b.size()) return false; 
 
  return equal(a.begin(), a.end(), b.begin()); 
} 
 
template<class T, class Allocator> 
  bool operator!=(const RangeArray<T, Allocator> &a, 
                  const RangeArray<T, Allocator> &b) 
{ 
  if(a.size() != b.size()) return true; 
 
  return !equal(a.begin(), a.end(), b.begin()); 
} 
 
template<class T, class Allocator> 
  bool operator<(const RangeArray<T, Allocator> &a, 
                 const RangeArray<T, Allocator> &b) 
{ 
  return lexicographical_compare(a.begin(), a.end(), 
                                 b.begin(), b.end()); 
} 
 
template<class T, class Allocator> 
  bool operator>(const RangeArray<T, Allocator> &a, 
                 const RangeArray<T, Allocator> &b) 
{ 
  return b < a; 
} 
 
template<class T, class Allocator> 
  bool operator<=(const RangeArray<T, Allocator> &a, 
                 const RangeArray<T, Allocator> &b) 
{ 
  return !(a > b); 
} 
 
template<class T, class Allocator> 
  bool operator>=(const RangeArray<T, Allocator> &a, 
                 const RangeArray<T, Allocator> &b) 
{ 
  return !(a < b); 
}
