/* Poniszy kod przykadowy zosta zaczerpnity z ksiki
 * "C++ Templates - The Complete Guide" autorstwa
 * Davida Vandevoorde'a i Nicolai'a M. Josuttisa, wydanej
 * w Polsce przez wydawnictwo HELION S.A. w roku 2003
 * (wydanie oryginalne: Addison-Wesley, 2002)
 *
 * (C) Copyright David Vandevoorde i Nicolai M. Josuttis 2002.
 * Kopiowanie, wykorzystanie, modyfikacja, sprzeda i
 * rozpowszechnianie tego oprogramowania jest dozwolone pod
 * warunkiem zachowania niniejszej noty o prawach autorskich
 * w kadej z wykonanych kopii.
 * Oprogramowanie to zostao udostpnione bez adnych jawnych
 * i niejawnych gwarancji; Autorzy nie gwarantuj te poprawnoci
 * dziaania niezalenie od zastosowania.
 */
template<typename T,
         typename CounterPolicy = SimpleReferenceCount,
         typename ObjectPolicy = StandardObjectPolicy>
class CountingPtr : private CounterPolicy, private ObjectPolicy {
  private:
    // skrty:
    typedef CounterPolicy CP;
    typedef ObjectPolicy  OP;

    T* object_pointed_to;      // obiekt wskazywany (bd NULL jeeli takowego nie ma)

  public:
    // konstruktor domylny (bez ograniczenia inicjalizacji jawnej):
    CountingPtr() {
        this->object_pointed_to = NULL;
    }

    // konstruktor konwertujcy (ze wskanika wbudowanego):
    explicit CountingPtr (T* p) {
        this->init(p);         // inicjalizacja wskanikiem zwykym
    }

    // konstruktor kopiujcy:
    CountingPtr (CountingPtr<T,CP,OP> const& cp)
     : CP((CP const&)cp),      // wytyczne kopiowania
       OP((OP const&)cp) {
        this->attach(cp);      // kopiowanie wskanika ze zwikszeniem licznika
    }

    // destruktor:
    ~CountingPtr() {
        this->detach();        // zmniejszenie licznika
                               //  (i jego ewentualne zniszczenie)
    }

    // przypisanie wskanika wbudowanego
    CountingPtr<T,CP,OP>& operator= (T* p) {
        // aden wskanik ze zliczaniem odwoa nie powinien jeszcze wskazywa do *p:
        assert(p != this->object_pointed_to);
        this->detach();        // zmniejszenie licznika
                               //  (i jego ewentualne zniszczenie)
        this->init(p);         // inicjalizacja wskanikiem zwykym
        return *this;
    }

    // przypisanie z kopiowaniem (uwaga na przypisania zwrotne):
    CountingPtr<T,CP,OP>&
    operator= (CountingPtr<T,CP,OP> const& cp) {
        if (this->object_pointed_to != cp.object_pointed_to) {
            this->detach();    // zmniejszenie licznika
                               //  (i jego ewentualne zniszczenie)
            CP::operator=((CP const&)cp);  // wytyczne przypisania
            OP::operator=((OP const&)cp);
            this->attach(cp);  // kopiowanie wskanika i zwikszenie licznika
        }
        return *this;
    }

    // operatory dla "inteligentnego" wskanika:
    T* operator-> () const {
        return this->object_pointed_to;
    }

    T& operator* () const {
        return *this->object_pointed_to;
    }

    // dodatkowe wywoania interfejsu, do przyszych potrzeb
    //...

  private:
    // metody pomocnicze:
    // - inicjalizacja wskanikiem zwykym
    void init (T* p) {
        if (p != NULL) {
            CounterPolicy::init(p);
        }
        this->object_pointed_to = p;
    }

    // - kopiowanie wskanika i zwikszenie licznika
    void attach (CountingPtr<T,CP,OP> const& cp) {
        this->object_pointed_to = cp.object_pointed_to;
        if (cp.object_pointed_to != NULL) {
            CounterPolicy::increment(cp.object_pointed_to);
        }
    }

    // - zmniejszenie licznika (i jego ewentualne zniszczenie)
    void detach() {
        if (this->object_pointed_to != NULL) {
            CounterPolicy::decrement(this->object_pointed_to);
            if (CounterPolicy::is_zero(this->object_pointed_to)) {
                // zniszczenie licznika obiektu wskazywanego:
                CounterPolicy::dispose(this->object_pointed_to);
                // wykorzystanie wytycznych obiektu do zniszczenia obiektu wskazywanego:
                ObjectPolicy::dispose(this->object_pointed_to);
            }
        }
    }
};
