/* Copyright (c) 1992 AT&T Bell Laboratories. */
/* C++. Styl i technika zaawansowanego programowania */
/* James O. Coplien */
/* Wszystkie prawa zastrzeone. */

#include <math.h>
// #include <X11/X.h>

typedef long Display, Window, GContext;

/*
 * Plik Shapes.h
 * Autor:  J. O. Coplien 
 *
 * Plik nagwkowy zawierajcy deklaracje klas uywanych
 * przez bibliotek figur.  Wiele funkcji skadowych
 * zostao w nim zdefiniowanych jako funkcje rozwijane
 * w miejscu wywoania.
 */

/*-----------------------------------------------------------*/
/* Klasa Angle reprezentuje kt w stopniach lub radianach.   */
/* Moe by atwo przeniesiona do systemw uywajcych       */
/* stopni lub radianw, a precyzja arytmetyczna              */
/* (float, double, or int) moe zosta zmodyfikowana.        */
/*-----------------------------------------------------------*/

class Angle {
friend double cos(Angle);
friend double sin(Angle);
public:
    Angle(double d)       { radians = d; }
    Angle ()              { radians = 0; }
    Angle(const Angle& a) { radians = a.radians; }
    inline Angle &operator=(const Angle& a) {
        radians = a.radians;
        return *this;
    }
    Angle operator+(const Angle&) const;
private:
    double    radians;
};

/*-----------------------------------------------------------   */
/* Klasa Coordinate reprezentuje punkt w ukadzie kartezjaskim.*/
/* Wsprzdne mog by wyraone w mikrometrach, pikselach,     */
/* calach lub dowolnych innych jednostkach.                     */
/*-----------------------------------------------------------   */

class Coordinate {
public:
    inline long x() const       { return x_rep; }
    inline long y() const       { return y_rep; }
    Coordinate    &operator=(const Coordinate &c) {
        x_rep = c.x_rep;
        y_rep = c.y_rep;
        return *this;
    }
    Coordinate(long x, long y) {
        x_rep = x; y_rep = y;
    }
    Coordinate(const Coordinate &c) {
        x_rep = c.x_rep;
        y_rep = c.y_rep;
    }
    Coordinate()        { x_rep = y_rep = 0;}
    ~Coordinate()       { }

    void rotateAbout(Coordinate,Angle);
private:
    long x_rep, y_rep;
};


/*-----------------------------------------------------------*/
/* Ponisze funkcje przeciaj identyfikator standardowych, */
/* funkcji trygonometrycznych tak, by mogy by one          */
/* wywoywane dla parametrw typu Angle.                     */
/* Poniewa przecianie moe dotyczy jedynie typu          */
/* parametrw, a nie wartoci zwracanej przez funkcj,       */
/* to funkcja atan, ktra powinna zwraca obiekt klasy       */
/* Angle nie moe by przeciona i wobec tego otrzymuje     */
/* zupenie nowu identyfikator Angleatan.                    */
/*-----------------------------------------------------------*/

inline double cos(Angle a)      { return ::cos(a.radians); }
inline double sin(Angle a)      { return ::sin(a.radians); }
inline Angle Angleatan(double t) {
    Angle a(::atan(t)); return a;
}

/*-----------------------------------------------------------*/
/* Klasa Color hermetyzuje sposb kodowania kolorw.         */
/* Jej wewntrzne dane dostepne s jedynie dla procedur      */
/* graficznych niskiego poziomu, a funkcje wysokiego         */
/* poziomu dziaaj na bardziej oglnej reprezentacji        */
/* kolorw. Zapewnia to moliwo przenoszenia pakietu       */
/* Na rne platformy. Miganie jest dodatkowym atrybutem     */
/* koloru. Reprezentacja czsto uywanych kolorw (biel,     */
/* czer) zostaa zadeklarowana w postaci staych, globalnych*/
/* obiektw White i Black.                                   */
/*-----------------------------------------------------------*/

enum Flash { FlashBlack, FlashWhite, FlashOff };

class Color {
public:
    inline    double    red()         { return red_rep;  }
    inline    double    green()       { return green_rep;  }
    inline    double    blue()        { return blue_rep;  }
    	    Color(double r=0.0, double g=0.0,
    	          double b=0.0, Flash f=FlashOff) {
        red_rep = r;
        green_rep = g;
	  blue_rep = b;
        flashing = f;
    }
    inline    Color&    operator=(const Color& c) {
        red_rep = c.red_rep;
        green_rep=c.green_rep;
        blue_rep = c.blue_rep;
    	return *this;
    }
    	    ~Color()    	{ }
private:
    double    red_rep, green_rep, blue_rep;
    Flash    flashing;
};

const Color Black, White = Color(1.0, 1.0, 1.0);

/*-----------------------------------------------------------*/
/* Klasa Shape jest klas bazow dla klas reprezentujcych.  */
/* figury geometryczne. Definiuje ona sygnatur (operacje)   */
/* wykonywane na figurach oraz dostarcza rzeczywistej        */
/* implementacji operacji wsplnych dla wszystkich           */
/* rodzajw figur (na przykad move).                        */
/*-----------------------------------------------------------*/

class Shape { 
public:
    // obrt dookoa rodka
    virtual    void    rotate(Angle a)    	{ } 
    virtual    void    move(Coordinate xy) {
        erase(); center_val=xy; draw();
    }
    virtual    void    draw()  = 0;     // odrysowanie
    virtual    void    erase() {
        color = Black; draw();
    }

    virtual    void    redraw() {
        erase(); draw();
    }
    Shape();
    virtual   ~Shape();
    virtual   Coordinate origin() const  { return center_val; }
    virtual   Coordinate center() const  { return center_val; }
    inline    Shape   *next() const   { return next_pointer; }
    inline    Shape   *prev() const   { return prev_pointer; }
    inline    void    insert(Shape* i) {
        i->next_pointer= this;
        i->prev_pointer=prev_pointer;
        prev_pointer->next_pointer=i;
        prev_pointer = i;
    }
    inline    void    append(Shape* i)    {
                          i->next_pointer=next();
    	    	    	  i->prev_pointer = this;
    	    	    	  next_pointer->prev_pointer=i;
    	    	    	  next_pointer = i;
    	    	    	}
protected:

    /*
     * Ponisze funkcje nie s prywatne, poniewa
     * musz by dostpne w klasach pochodnych
     */

    Coordinate    center_val;    // rodek figury
    Color    	color;    	// kolor figury

    /*
     * Ponisze zmienne statyczne wykorzystywane s przez
     * pakiet okienkowy w celu oglnej konfiguracji.
    */

    static    	Display    	display;
    static    	Window    	window;
    static    	GContext    	gc;

private:
    Shape    	*next_pointer;   // wskanik nastpnego elementu listy
    Shape    	*prev_pointer;   // wskanik poprzedniego elementu listy
};

/*-----------------------------------------------------------*/
/* Klasa Line stanowi specjalny rodzaj figury, poniewa      */
/* wykorzystywana jest jako komponent podczas rysowania      */
/* pozostaych figur. Tylko ta klasa posiada funkcj skadow*/
/* rotateAbout wykorzystywan do obrotw innych figur.       */
/* Operacja rotate stanowi zdegenerowan wersj operacji .   */
/* rodek odcinka klasy Line zdefiniowany jest jako rednia  */
/* jego kocw Pocztkiem jest pierwszy z podanych punktw.  */
/*-----------------------------------------------------------*/

class Line : public Shape {
public:
    void    rotate(Angle a)    { rotateAbout(center(),a); }
    void    rotateAbout(Coordinate,Angle);
    void    draw();
    inline Line &operator=(const Line &l) {
	 (void)this->Shape::operator=(*this);
         p=Coordinate(l.p); q=Coordinate(l.q); return *this;
    }
    Line(Coordinate,Coordinate);
    Line(Line& l) {
        p = l.p; q = l.q;
        center_val = l.center_val;
    }
    Line()    	    { p = Coordinate(0,0);
                      q = Coordinate(0,0); }
   ~Line()    	    { erase(); }
    inline  Coordinate      e() const   { return q; }
    inline  Coordinate origin() const   { return p; }
private:
    Coordinate    p, q;    	    // koce odcinka
};

/*-----------------------------------------------------------*/
/* Obiekt klasy Rect skada si z czterech odcinkw.         */
/* Za punkt pocztkowy przyjmowany jest lewy-grny           */
/* wierzchoek, ktry jest pierwszym parametrem konstruktora.*/
/* rodek figury jest, zgodnie z oczekiwaniami, jej rodkiem */
/* geometrycznym. Operacje erase dziedziczona jest po klasie */
/* Shape. Odcinki reprezentujce boki prostokta nie s      */
/* skadowymi prywatnymi, poniewa musz by dostepne w      */
/* klasie pochodnej Square                                   */
/*-----------------------------------------------------------*/

class Rect : public Shape {
public:
    void    	rotate(Angle);
    void    	draw();
    void    	move(Coordinate);
    	    //lewy-grny wierzchoek, rozmiar na osi x, rozmiar na osi y
    	    Rect(Coordinate,long,long);
    	    ~Rect()    	    { erase(); }
    inline    Coordinate    origin() const { return l1.origin(); }
protected:
    Line    	l1, l2, l3, l4;
};

/*-----------------------------------------------------------*/
/* rodek i pocztek elipsy stanowi jej rodek geometryczny. */
/* Klasa Ellipse nie zostaa zaimplementowana.               */
/*-----------------------------------------------------------*/

class Ellipse : public Shape {
public:
    void    	rotate(Angle);
    void    	draw();
    	    // rodek, wikszy promie, mniejszy promie
    	    Ellipse(Coordinate,long,long);
    	    ~Ellipse()    	{ erase(); }
protected:
    long    	major, minor;
};

/*-----------------------------------------------------------   */
/* Trjkt skada si z trzech odcinkw.  Pooenie             */
/* rodka wyznaczane jest jako rednia wsprzdnych wszystkich */
/* wierzchokw.                                                */
/*-----------------------------------------------------------   */

class Triangle : public Shape {
public:
    void    	rotate(Angle);
    void    	draw();
    void    	move(Coordinate);
    	    Triangle(Coordinate,Coordinate,Coordinate);
    	    ~Triangle()    	{ erase(); }
private:
    Line    	l1, l2, l3;
};

/*-----------------------------------------------------------*/
/* Kwadrat stanowi "zdegenrowany" przypadek prostokta.      */
/* Klasa Square dziedziczy wszystkie operacje po klasie Rect.*/
/* Konstruktor klasy Square wykorzystuje konstruktor klasy   */
/*  Rect.                                                    */
/*-----------------------------------------------------------*/

class Square : public Rect {
public:
    	    Square(Coordinate ctr, long x):
    	    	Rect(ctr, x, x)         { }
};

/*-----------------------------------------------------------*/
/*                                                           */
/* Plik Shapes.c, zawiera implementacj klas                 */
/*                                                           */
/*-----------------------------------------------------------*/


extern "C" {
    extern void doXInitialization(Shape&);
    extern void
       XDrawLine(Display, Window, GContext, int, int, int, int);
}

/*
 * KLASA ANGLE
 */

Angle
Angle::operator+(const Angle& angle) const {
    Angle retval = angle;
    retval.radians += radians;
    return retval;
}

/*
 * KLASA COORDINATE
 */

void
Coordinate::rotateAbout(Coordinate pivot, Angle angle)
{
    long xdistance = pivot.x()-x(), ydistance = pivot.y()-y();
    long xdistsquared = xdistance * xdistance,
    	ydistsquared = ydistance * ydistance;
    double r = ::sqrt( xdistsquared + ydistsquared );
    Angle newangle = angle +
                Angleatan(double(ydistance)/double(xdistance));
    x_rep = pivot.x() + long(r * ::cos(newangle));
    y_rep = pivot.y() + long(r * ::sin(newangle));
}

/*
 * KLASA SHAPE
 */

/* znaczniki dla pakietu graficznego niszego poziomu */

static int X_initialized = 0;

Shape::Shape() {
    center_val = Coordinate(0,0);
    next_pointer=prev_pointer=0;
    color = White;
    if( !X_initialized) {
    	doXInitialization(*this);
    	X_initialized = 1;
    }
}

/*
 * KLASA LINE
 */

void
Line::rotateAbout(Coordinate c, Angle angle) {
    erase();
    p.rotateAbout(c, angle);
    q.rotateAbout(c, angle);
    draw();
}

void
Line::draw() {
    if (p.x()-q.x() && p.y()-q.y()) {
//    	XDrawLine(display, window, gc, p.x(),
//                                         p.y(), q.x(), q.y());
    }
}

Line::Line(Coordinate p1, Coordinate p2) {
    p = p1; q = p2;
    center_val = Coordinate((p.x()+q.x())/2, (p.y()+q.y())/2);
    color = Color(White);
    draw();
}

/*
 * KLASA RECTANGLE
 */

void
Rect::rotate(Angle angle) {
    erase();
    l1.rotateAbout(center(), angle);
    l2.rotateAbout(center(), angle);
    l3.rotateAbout(center(), angle);
    l4.rotateAbout(center(), angle);
    draw();
}

void
Rect::draw() {
    l1.draw();
    l2.draw();
    l3.draw();
    l4.draw();
}

void
Rect::move(Coordinate c) {
    /*
     * Parametr funkcji reprezentuje docelowe pooenie rodka
     * figury.
     */

    erase();
    long xmoved = c.x() - center().x();
    long ymoved = c.y() - center().y();
    center_val = c;
    l1 = Line(Coordinate(l1.origin().x()+xmoved,
                                      l1.origin().y()+ymoved),
    	    Coordinate(l1.e().x()+xmoved,l1.e().y()+ymoved) );
    l2 = Line(Coordinate(l2.origin().x()+xmoved,
                                      l2.origin().y()+ymoved),
    	    Coordinate(l2.e().x()+xmoved,l2.e().y()+ymoved) );
    l3 = Line(Coordinate(l3.origin().x()+xmoved,
                                      l3.origin().y()+ymoved),
    	    Coordinate(l3.e().x()+xmoved,l3.e().y()+ymoved) );
    l4 = Line(Coordinate(l4.origin().x()+xmoved,
                                      l4.origin().y()+ymoved),
    	    Coordinate(l4.e().x()+xmoved,l4.e().y()+ymoved) );
    draw();
}

Rect::Rect(Coordinate topLeft, long xsize, long ysize) {
    Coordinate    a(topLeft);
    Coordinate    b(a.x()+xsize, a.y());
    Coordinate    c(a.x(),a.y()+ysize);
    Coordinate    d(b.x(),c.y());
    l1 = Line(a,b);
    l2 = Line(b,c);
    l3 = Line(c,d);
    l4 = Line(d,a);
    center_val = Coordinate((l1.origin().x()+l2.e().x())/2,
    	    	    (l4.origin().y()+l4.e().y())/2);
    draw();
}

/*
 * KLASA TRIANGLE
 */

void
Triangle::rotate(Angle angle) {
    erase();
    l1.rotateAbout(center(), angle);
    l2.rotateAbout(center(), angle);
    l3.rotateAbout(center(), angle);
    draw();
}

void
Triangle::move(Coordinate c) {
    /*
     * Parametr funkcji reprezentuje docelowe pooenie rodka
     * figury.
     */

    erase();
    long xmoved = c.x() - center().x();
    long ymoved = c.y() - center().y();
    center_val = c;
    l1 = Line(Coordinate(l1.origin().x()+xmoved,
                                    l1.origin().y()+ymoved),
    	    Coordinate(l1.e().x()+xmoved,l1.e().y()+ymoved) );
    l2 = Line(Coordinate(l2.origin().x()+xmoved,
                                    l2.origin().y()+ymoved),
    	    Coordinate(l2.e().x()+xmoved,l2.e().y()+ymoved) );
    l3 = Line(Coordinate(l3.origin().x()+xmoved,
                                    l3.origin().y()+ymoved),
    	    Coordinate(l3.e().x()+xmoved,l3.e().y()+ymoved) );
    draw();
}

void
Triangle::draw() {
    l1.draw(); l2.draw(); l3.draw();
}

Triangle::Triangle(Coordinate a, Coordinate b, Coordinate c) {
    l1 = Line(a,b);  l2 = Line(b,c);  l3 = Line(c,a);
    center_val =
               Coordinate((l1.e().x()+l2.e().x()+l3.e().x())/3,
        	    	    (l1.e().y()+l2.e().y()+l3.e().y())/3);
    draw();
}

/*-----------------------------------------------------------*/
/* Plik main.c -- program wykorzystujcy klasy Shape         */
/*-----------------------------------------------------------*/

const int XSIZE = 800;
const int YSIZE = 1024;

void
rotateList(Shape *s, Angle a)
{
    for (Shape *f = s;  f;  f = f->next())
    	f->rotate(a);
}

int
main()
{
    Coordinate origin (0,0);

    /*
     * windowBorder reprezentuje nie tylko ramk na ekranie,
     * ale rwnie czoo listy figur
     */

    Rect windowBorder(origin, XSIZE, YSIZE);

    Shape * t = new Triangle(
    	Coordinate(100, 100),
    	Coordinate(200, 200),
    	Coordinate(300, 100)
    );
    windowBorder.append(t);

//    Shape * c = new Circle(Coordinate(500,652), 150);
//    t->insert(c);

    rotateList(&windowBorder, 90.0);
    return 0;
}


