// idep_nimap.c
#include "idep_nameindexmap.h"
#include "idep_namearray.h"

#include <string.h>     // strcmp()
#include <memory.h>     // memcpy() memset()
#include <iostream.h>
#include <assert.h>

                // -*-*-*- funkcje statyczne -*-*-*-

enum { DEFAULT_TABLE_SIZE = 521 };
enum { BAD_INDEX = -1 };

static unsigned hash(register const char* name) // Uwaga: metoda zwraca warto unsigned!
{
    register unsigned sum = 1000003; // 1 000 003 jest  78 498-t liczb pierwsz
    while (*name) {
        sum *= *name++; // mnoenie liczb cakowitych jest standardowym podprogramem na                         // stacjach roboczych SPARC
    }
    return sum; // unsigned zapewnia dodatni warto wykorzystywan przez operator (%).
}

                // -*-*-*- idep_NameIndexMapLink -*-*-*-

struct idep_NameIndexMapLink {
    const char *d_name_p;                       // nazwa
    int d_index;                                // indeks nazwy
    idep_NameIndexMapLink *d_next_p;            // wskanik do nastpnego cza
    idep_NameIndexMapLink(const char *name, int index,
                          idep_NameIndexMapLink *d_next_p);
};

idep_NameIndexMapLink::idep_NameIndexMapLink(const char *name, int index,
                                                idep_NameIndexMapLink *next)
: d_name_p(name)
, d_index(index)
, d_next_p(next)
{
}

static const idep_NameIndexMapLink *find(const idep_NameIndexMapLink *p,
                                                            const char *name)
{
    while (p && 0 != strcmp(p->d_name_p, name)) {
        p = p->d_next_p;
    }
    return p;
}

                // -*-*-*- idep_NameIndexMap_i -*-*-*-

struct idep_NameIndexMap_i {
    idep_NameArray d_array;                     // tablica nazw
    idep_NameIndexMapLink **d_table_p;          // tablica mieszajca nazw
    int d_tableSize;                            // rozmiar tablicy mieszajcej

    idep_NameIndexMap_i(int size);
        // utworzenie odwzorowania przyjmujc okrelony rozmiar maksymalny
    ~idep_NameIndexMap_i();

    idep_NameIndexMapLink *& findSlot(const char *name);
        // odszukanie odpowiedniego "gniazda" dla nazwy

    int insert(idep_NameIndexMapLink *& slot, const char *name);
        // wstawienie nazwy do podanego "gniazda"
};

idep_NameIndexMap_i::idep_NameIndexMap_i(int size)
: d_array(size)
, d_tableSize(size > 0 ? size : DEFAULT_TABLE_SIZE)
{ 
    d_table_p = new idep_NameIndexMapLink *[d_tableSize];
    assert(d_table_p);
    memset(d_table_p, 0, d_tableSize * sizeof *d_table_p);
}

idep_NameIndexMap_i::~idep_NameIndexMap_i()
{ 
    for (int i = 0; i < d_tableSize; ++i) {
        idep_NameIndexMapLink *p = d_table_p[i];
        while (p) {
        idep_NameIndexMapLink *q = p;
            p = p->d_next_p;
            delete q;
        }
    }
    delete [] d_table_p;
}

idep_NameIndexMapLink *& idep_NameIndexMap_i::findSlot(const char *name)
{
    int index = hash(name) % d_tableSize;
    assert(index >= 0 && index < d_tableSize);
    return d_table_p[index];
}

int idep_NameIndexMap_i::insert(idep_NameIndexMapLink *& slot, const char *nm)
{
    int index = d_array.append(nm); // indeks znajduje si wewntrz zarzdzanej tablicy 
                                    // cigw znakw
    slot = new idep_NameIndexMapLink(d_array[index], index, slot);
    return index;
}

                // -*-*-*- idep_NameIndexMap -*-*-*-

idep_NameIndexMap::idep_NameIndexMap(int maxEntriesHint)
: d_this(new idep_NameIndexMap_i(maxEntriesHint))
{
}

idep_NameIndexMap::~idep_NameIndexMap()
{
    delete d_this;
}

int idep_NameIndexMap::add(const char* name)
{
    idep_NameIndexMapLink *& slot = d_this->findSlot(name);
    return find(slot, name) ? BAD_INDEX : d_this->insert(slot, name);
}

int idep_NameIndexMap::entry(const char* name)
{
    idep_NameIndexMapLink *& slot = d_this->findSlot(name);
    const idep_NameIndexMapLink *link = find(slot, name);
    return link ? link->d_index : d_this->insert(slot, name);
}

const char *idep_NameIndexMap::operator[](int i) const
{
    return d_this->d_array[i];
}

int idep_NameIndexMap::length() const
{
    return d_this->d_array.length();
}

int idep_NameIndexMap::lookup(const char *name) const
{
    idep_NameIndexMapLink *& slot = d_this->findSlot(name);
    const idep_NameIndexMapLink *link = find(slot, name);
    return link ? link->d_index : BAD_INDEX;
}

ostream& operator<<(ostream& out, const idep_NameIndexMap& map)
{
    int fieldWidth = 10;
    int maxIndex = map.length() - 1;
    assert (sizeof (long int) >= 4);
    long int x = 1000 * 1000 * 1000;    // wymaga 4-bajtowej liczby cakowitej.
    while (fieldWidth > 1 && 0 == maxIndex / x) {
        --fieldWidth;
        x /= 10;
    }
 
    for (int i = 0; i < map.length(); ++i) {
        out.width(fieldWidth);
        out << i << ". " << map[i] << endl;
    }
 
    return out;
}

